<?php

/**
 * This is a common store component which will be used across all Module-store-controllers
 */

namespace ComponentStoreSpace\Controllers;

use App\Components\Controllers\Component;
use App\Modules\Products\Models\ProductOptions;
use App\Modules\Products\Models\ProductOptionRel;
use App\Modules\Products\Models\ProductVariants;
use App\Modules\Products\Models\ProductVariantRel;
use App\Modules\Products\Models\ProductImage;
use App\Modules\Products\Models\ProductImageSides;
use App\Modules\Products\Models\ProductSetting;
use App\Modules\Products\Models\CatalogProductRel;

class StoreComponent extends Component
{
    public $shopDomain;
    private $token;
    private $apiKey;
    private $secret;
    private $apiVersion;
    private $lastResponseHeaders = null;
    public $thumbSize = "small";
    public $fullSize = "";
    public $colorAttrName = "color";
    public $sizeAttrName = "size";

    function __construct()
    {
        if (!isset($_SESSION)) {
            session_start();
        }
        parent::__construct();
        $this->name = "ShopifyClient";
        $this->shopDomain = SHOPIFY_SHOP . '.myshopify.com';
        $this->token = APIPASS;
        $this->apiKey = APIUSER;
        $this->secret = SECRETKEY;

        $settingDIR = ASSETS_PATH_W . 'settings/stores/1/settings.json';
        $getSettingDetails = file_get_contents($settingDIR);
        $getSettingDetails = json_decode($getSettingDetails, true);

        if (!empty($getSettingDetails['store']['color'])) {
            $this->colorAttrName = strtolower($getSettingDetails['store']['color']);
        }

        if (!empty($getSettingDetails['store']['size'])) {
            $this->sizeAttrName = strtolower($getSettingDetails['store']['size']);
        }

        $versionDir = str_replace('assets/', 'shopify/data/api_version.json', ASSETS_PATH_W);
        if (!file_exists($versionDir)) {
            $this->getLatestApiVersion();
            $installedVersion = json_decode(file_get_contents($versionDir), true);
            $this->apiVersion = $installedVersion[0]['api_version'];
        } else {
            $installedVersion = json_decode(file_get_contents($versionDir), true);
        }

        $this->apiVersion = $installedVersion[0]['api_version'];
    }

    /**
     * Get: get Shopify authorization URL
     *
     * @param $scope  Shopify inut parameter
     * @param $redirectURL Shopify returned URL incase of error
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return Shopify rest autorised URL
     */
    public function getAuthorizeUrl($scope, $redirectURL = '')
    {
        $url = "http://{$this->shopDomain}/admin/oauth/authorize?client_id={$this->apiKey}&scope=" . urlencode($scope);
        if ($redirectURL != '') {
            $url .= "&redirect_uri=" . urlencode($redirectURL);
        }
        return $url;
    }

    /**
     * Get: get latest version of api and save it to the config.xml file
     *
     * @author divya@imprintnext.com
     * @date   30 January 2023
     * @return Shopify API version
     */
    public function getLatestApiVersion()
    {

        $currentMonth = date("m");
        $monthDiff = $this->getVersionDifference();
        $previousYear =  date("Y", strtotime("-1 year"));
        if ($monthDiff > 6) {
            if ($currentMonth > 01 && $currentMonth <= 04) {
                $this->apiVersion = date("Y-" . '01');
            } elseif ($currentMonth > 04 && $currentMonth <= 07) {
                $this->apiVersion = date("Y-" . '04');
            } elseif ($currentMonth > 07 && $currentMonth <= 10) {
                $this->apiVersion = date("Y-" . '07');
            } elseif (($currentMonth > 10 && $currentMonth < 12) || $currentMonth == 01) {
                $this->apiVersion = date("Y-" . '10');
            }
        }
        if ($monthDiff <= -6) {
            if ($currentMonth > 01 && $currentMonth <= 04) {
                $this->apiVersion = date("Y-" . '01');
            } elseif ($currentMonth > 04 && $currentMonth <= 07) {
                $this->apiVersion = $previousYear . '-' . '04';
            } elseif ($currentMonth > 07 && $currentMonth <= 10) {
                $this->apiVersion = $previousYear . '-' . '07';
            } elseif (($currentMonth > 10 && $currentMonth < 12) || $currentMonth == 01) {
                $this->apiVersion = $previousYear . '-' . '10';
            }
        }
        $versionJsonPath = str_replace('assets/', 'shopify/data/api_version.json', ASSETS_PATH_W);
        $installedVersion = json_decode(file_get_contents($versionJsonPath), true);
        $installedVersion[0]['api_version'] = $this->apiVersion;
        $jsonData = json_encode($installedVersion);
        file_put_contents($versionJsonPath, $jsonData);

        $baseDIR = getcwd();
        $xmlPath = $baseDIR . '/../../config.xml';
        $dom = new \DomDocument();
        $dom->load($xmlPath);
        $versionFileData = str_replace('assets/', 'shopify/data/api_version.json', ASSETS_PATH_R);
        $versionFileData = json_decode(file_get_contents($versionFileData), true);
        if (isset($dom->getElementsByTagName('apiVersion')->item(0)->nodeValue) && $dom->getElementsByTagName('apiVersion')->item(0)->nodeValue != $this->apiVersion) {
            $dom->getElementsByTagName('apiVersion')->item(0)->nodeValue = $versionFileData[0]['api_version'];
            $dom->save($xmlPath);
        } else {
            $dataElement = $dom->createElement("apiVersion", $versionFileData[0]['api_version']);
            $dom->getElementsByTagName('url_detail')->item(0)->appendChild($dataElement);
            $dom->save($xmlPath);
        }
    }

    /**
     * Get: Get diffrenece between api version month
     *
     * @author divya@imprintnext.com
     * @date   31 January 2023
     * @return year-month combination string
     */
    public function getVersionDifference()
    {
        $versionDir = str_replace('assets/', 'shopify/data', ASSETS_PATH_W);
        $versionJsonPath = str_replace('assets/', 'shopify/data/api_version.json', ASSETS_PATH_W);
        if (!is_file($versionJsonPath)) {
            $baseDIR = getcwd();
            $xmlPath = $baseDIR . '/../../config.xml';
            $dom = new \DomDocument();
            $dom->load($xmlPath);
            $versionValue = $dom->getElementsByTagName('apiVersion')->item(0)->nodeValue;
            $vesionJsonData = [
                ["api_version" => $versionValue]
            ];
            mkdir($versionDir, 0777, true);
            $targetFile = $versionDir . '/api_version.json';
            file_put_contents($targetFile, json_encode($vesionJsonData));
            $installedVersion = json_decode(file_get_contents($versionJsonPath), true);
        } else {
            $installedVersion = json_decode(file_get_contents($versionJsonPath), true);
        }

        $currentdate = date("Y-m");
        $apiVersionMonth = $installedVersion[0]['api_version'];
        $ts1 = strtotime($currentdate);
        $ts2 = strtotime($apiVersionMonth);
        $year1 = date('Y', $ts1);
        $year2 = date('Y', $ts2);
        $month1 = date('m', $ts1);
        $month2 = date('m', $ts2);
        return (($year1 - $year2) * 12) + ($month1 - $month2);
    }

    /**
     * Get: get Shopify Access token
     *
     * @param $code  Shopify Oauth token
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return Shopify rest access token
     */
    public function getAccessToken($code)
    {
        // POST to  POST https://SHOP_NAME.myshopify.com/admin/oauth/access_token
        $url = "https://{$this->shopDomain}/admin/oauth/access_token";
        $payload = "client_id={$this->apiKey}&client_secret={$this->secret}&code=$code";
        $response = $this->curlHttpApiRequest('POST', $url, '', $payload, array());
        $response = json_decode($response, true);
        if (isset($response['access_token'])) {
            return $response['access_token'];
        }
        return '';
    }

    /**
     * Internal function to get API call credit done per second
     *
     * @return string
     */
    private function callsMade()
    {
        return $this->shopApiCallLimitParam(0);
    }

    /**
     * Internal function to get API call credit limit per second
     *
     * @return string
     */
    private function callLimit()
    {
        return $this->shopApiCallLimitParam(1);
    }

    /**
     * Internal function to get API call credit left per second
     *
     * @return string
     */
    private function callsLeft()
    {
        try {
            return $this->callLimit() - $this->callsMade();
        } catch (\Exception $e) {
            return 40;
        }
    }

    /**
     * Shopify REST aPI call helper function
     *
     * @param $method  REST API call method (GET, POST, PUT, DELETE)
     * @param $path  Shopify API end point
     * @param $params  Shopify API parameters if any
     * @param $sessionKey  Session identifying key for pagination.
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return Shopify REST API call response in array
     */
    private function call($method, $path, $params = array(), $sessionKey = "", $fetchRaw = false)
    {
        $APIData = array();
        if ($method == 'GET' && (strpos($path, '/admin/products/') !== false || strpos($path, '/admin/variants/') !== false) && !$fetchRaw) {
            $APIData = $this->getAPIDataFromCache($path);
        }
        if (empty($APIData)) {
            if ($this->callsLeft() < 5) {
                sleep(8);
            }
            //checking difference between current api version and version in config.xml file
            if ($this->getVersionDifference()  > 6 || $this->getVersionDifference() <= -6) {
                $this->getLatestApiVersion();
            }
            $needle   = 'customer_saved_searches';
            $versionString = 'api/' . $this->apiVersion . '/';
            $newPath = (!empty($this->apiVersion) && strpos($path, $this->apiVersion) === false && strpos($path, $needle) === false) ? str_replace("admin/", "admin/" . $versionString, $path) : $path;
            $baseurl = "https://{$this->shopDomain}/";
            $url = $baseurl . ltrim($newPath, '/');
            $query = in_array($method, array('GET', 'DELETE')) ? $params : array();
            $payload = in_array($method, array('POST', 'PUT')) ? stripslashes(str_replace("\u", "\\\\u", json_encode($params))) : array();
            $requestHeaders = in_array($method, array('POST', 'PUT')) ? array("Content-Type: application/json; charset=utf-8", 'Expect:') : array();
            // add auth headers
            $requestHeaders[] = 'X-Shopify-Access-Token: ' . $this->token;
            $response = $this->curlHttpApiRequest($method, $url, $sessionKey, $query, $payload, $requestHeaders);
            $response = json_decode($response, true);
            if (isset($response['errors']) || ($this->lastResponseHeaders['http_status_code'] >= 400)) {
                return array("method" => $method, "path" => $newPath, "params" => $params, "header" => $this->lastResponseHeaders, "response" => $response);
            }
            $APIData = (is_array($response) && (!empty($response))) ? array_shift($response) : $response;
        }
        return $APIData;
    }

    private function getAPIDataFromCache($path)
    {
        $APIContent = array();
        // some places the variant detail is called as:: admin/products/<product_id>/variants/<variant_id>.json
        // added extra conditions for that
        if (strpos($path, '/admin/products/') !== false && strpos($path, '/variants/') === false) {
            $APIQry = substr($path, strrpos($path, '/') + 1);
            $thisProductID = substr($APIQry, 0, strpos($APIQry, '.'));
            $thisProdCacheDIR = SHOPIFY_CACHE_FOLDER . $thisProductID . ".json";
            if (!empty($thisProductID) && file_exists($thisProdCacheDIR)) {
                $cacheContent = file_get_contents($thisProdCacheDIR);
            }
            $APIContent = json_decode($cacheContent, true);
        }

        // alternative path for variant
        $vUrlPattern = "/\/admin\/products\\/[a-z0-9]+\/variants\//i";
        if (strpos($path, '/admin/variants/') !== false || preg_match($vUrlPattern, stripslashes($path))) {
            $APIQry = substr($path, strrpos($path, '/') + 1);
            $thisVariantID = substr($APIQry, 0, strpos($APIQry, '.'));
            $thisVarCacheDIR = SHOPIFY_CACHE_FOLDER . "variants/" . $thisVariantID . ".json";
            if (!empty($thisVariantID) && file_exists($thisVarCacheDIR)) {
                $cacheContent = file_get_contents($thisVarCacheDIR);
            }
            $APIContent = json_decode($cacheContent, true);
        }
        return $APIContent;
    }

    /**
     * Shopify REST aPI signature validator
     *
     * @param $query  Shopify API query params
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return Shopify REST API signature
     */
    private function validateSignature($query)
    {
        if (!is_array($query) || empty($query['signature']) || !is_string($query['signature'])) {
            return false;
        }
        foreach ($query as $k => $v) {
            if ($k == 'signature') {
                continue;
            }
            $signature[] = $k . '=' . $v;
        }
        sort($signature);
        $signature = md5($this->secret . implode('', $signature));
        return $query['signature'] == $signature;
    }

    /**
     * Shopify API curl function
     *
     * @param $method  REST API call method (GET, POST, PUT, DELETE)
     * @param $path  Shopify API end point
     * @param $params  Shopify API parameters if any
     * @param $sessionKey  Session identifying key for pagination.
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return Curlresponse
     */
    private function curlHttpApiRequest($method, $url, $sessionKey, $query = '', $payload = '', $requestHeaders = array())
    {
        $url = $this->curlAppendQuery($url, $query);
        $ch = curl_init($url);
        $this->curlSetopts($ch, $method, $payload, $requestHeaders);
        $response = curl_exec($ch);
        $errno = curl_errno($ch);
        $error = curl_error($ch);
        curl_close($ch);
        if ($errno) {
            print_r($errno . ": " . $error);
        }
        list($message_headers, $message_body) = preg_split("/\r\n\r\n|\n\n|\r\r/", $response, 2);
        $this->lastResponseHeaders = $this->curlParseHeaders($message_headers);
        if ($sessionKey != '') {
            $headerLink = $this->lastResponseHeaders['link'];
            if ($this->returnHeaderArray($headerLink) != '') {
                $_SESSION[$sessionKey] = $this->returnHeaderArray($headerLink);
            }
        }
        // var_dump($this->lastResponseHeaders);exit();
        return $message_body;
    }

    /**
     * Shopify API curl header return
     *
     * @param $path  Shopify API header response link
     *
     * @author debashisd@riaxe.com
     * @date   02 March 2020
     * @return pagination links if available
     */
    private function returnHeaderArray($linkHeader)
    {
        $cleanArray = [];
        if (strpos($linkHeader, ',') !== false) {
            //Split into two or more elements by comma
            $linkHeaderArr = explode(',', $linkHeader);
        } else {
            //Create array with one element
            $linkHeaderArr[] = $linkHeader;
        }
        foreach ($linkHeaderArr as $linkHeader) {
            $cleanArray += [
                $this->extractRel($linkHeader) => $this->extractLink($linkHeader)
            ];
        }
        return $cleanArray;
    }

    /**
     * Internal function to get pagination links
     *
     * @return string
     */
    private function extractLink($element)
    {
        if (preg_match('/<(.*?)>/', $element, $match) == 1) {
            return $match[1];
        }
    }

    /**
     * Internal function to get pagination links
     *
     * @return string
     */
    private function extractRel($element)
    {
        if (preg_match('/rel="(.*?)"/', $element, $match) == 1) {
            return $match[1];
        }
    }

    /**
     * Internal function to get pagination links
     *
     * @return string
     */
    private function curlAppendQuery($url, $query)
    {
        if (empty($query)) {
            return $url;
        }
        if (is_array($query)) {
            return "$url?" . http_build_query($query);
        } else {
            return "$url?$query";
        }
    }

    /**
     * Internal function to set curl params
     *
     * @return string
     */
    private function curlSetopts($ch, $method, $payload, $requestHeaders)
    {
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_USERAGENT, 'ohShopify-php-api-client');
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 90);
        curl_setopt($ch, CURLOPT_TIMEOUT, 500);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if (!empty($requestHeaders)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $requestHeaders);
        }
        if ($method != 'GET' && !empty($payload)) {
            if (is_array($payload)) {
                $payload = http_build_query($payload);
            }
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        }
    }

    /**
     * Internal function to get parsed CURL header return
     *
     * @return string
     */
    private function curlParseHeaders($message_headers)
    {
        $headerLines = preg_split("/\r\n|\n|\r/", $message_headers);
        $headers = array();
        list(, $headers['http_status_code'], $headers['http_status_message']) = explode(' ', trim(array_shift($headerLines)), 3);
        foreach ($headerLines as $headerLine) {
            list($name, $value) = explode(':', $headerLine, 2);
            $name = strtolower($name);
            $headers[$name] = trim($value);
        }
        return $headers;
    }

    /**
     * Internal function to get Shopify API call limits left
     *
     * @return string
     */
    private function shopApiCallLimitParam($index)
    {
        if ($this->lastResponseHeaders == null) {
            throw new \Exception('Cannot be called before an API call.');
        }
        $params = explode('/', $this->lastResponseHeaders['http_x_shopify_shop_api_call_limit']);
        return (int) $params[$index];
    }

    /**
     * Internal function to get sanitized API text contents
     *
     * @return string
     */
    public static function shopify_body($body)
    {
        return strip_tags(str_ireplace('"', "''", $body));
    }

    /**
     * Internal function to get Show in designer collection id
     *
     * @return string
     */
    private function getShowInDesignerId()
    {
        $collections = $this->call('GET', '/admin/custom_collections.json?handle=show-in-designer');
        if (empty($collections)) {
            $collections = $this->call('GET', '/admin/smart_collections.json?handle=show-in-designer');
        }
        return $collections[0]['id'];
    }

    /**
     * Getting pre-designed-product collection id
     * @param 
     * @return integer (pre designed product collection id)
     * 
     */
    public function getPreDesignedCollectionId()
    {
        $collections = $this->call('GET', '/admin/custom_collections.json?handle=pre-designed-product');
        if ($collections[0]['id'] != "") {
            return $collections[0]['id'];
        } else {
            $collectionArray = array(
                "custom_collection" => array(
                    "title" => 'Pre Designed Product',
                    "handle" => 'pre-designed-product',
                    "body_html" => "<strong>The products under this collection are designed in designer tool.</strong>",
                    "published_scope" => false,
                )
            );

            $collections = $this->call('POST', '/admin/custom_collections.json', $collectionArray);
            return $collections['id'];
        }
    }

    /**
     * Getting catalog-product collection id
     * @param 
     * @return integer (pre designed product collection id)
     * 
     */
    public function getCatalogueCollectionId()
    {
        $collections = $this->call('GET', '/admin/smart_collections.json?handle=imprint_catalogue');
        if ($collections[0]['id'] != "") {
            return $collections[0]['id'];
        } else {
            $collectionArray = array(
                "smart_collection" => array(
                    "title" => 'imprint catalogue',
                    "handle" => 'imprint_catalogue',
                    "body_html" => "<strong>The products under this collection are imported through ImprintNext catalogue.</strong>",
                    "rules" => array(array(
                        "column" => "tag",
                        "relation" => "equals",
                        "condition" => "catalogue"
                    )),
                    "published_scope" => false,
                )
            );
            $collections = $this->call('POST', '/admin/smart_collections.json', $collectionArray);
            return $collections['id'];
        }
    }

    /**
     *  to get Customized collection id
     *
     * @return integer
     */
    private function getCustomizedCollectionId()
    {
        $collection = $this->call('GET', '/admin/smart_collections.json?handle=customized');
        if (!empty($collection)) {
            return $collection[0]['id'];
        }
    }

    /** ************************* Product Module Start **************************** **/

    /**
     * GET: Fetching all products from store
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param filter parameters
     * @return Array of all product
     */
    public function getAllProducts($filters, $isCustomize = '')
    {
        $productParams['limit'] = empty($filters['limit']) ? 250 : $filters['limit'];
        $orderBy = empty($filters['orderby']) ? 'id' : 'title';
        $order = ($filters['order'] == "desc") ? SORT_DESC : SORT_ASC;
        $productParams['fields'] = 'id,title,variants,images';
        $productParams['status'] = 'active';
        $requestedPage = empty($filters['page']) ? 1 : $filters['page'];
        $categoryid = $filters['collection_id'];
        $searchstring = $filters['title'];
        if (!$categoryid && $categoryid == '') {
            $productParams['collection_id'] = $this->getShowInDesignerId();
        }
        if ($categoryid && $categoryid != '') {
            $productParams['collection_id'] = $categoryid;
        }
        if ($searchstring && $searchstring != '') {
            return $this->getSearchProduct($searchstring);
        }
        if ($isCustomize == 1) {
            $productParams['collection_id'] = $this->getPreDesignedCollectionId();
        }
        if ($filters['isCatalogue'] > 0) {
            $productParams['collection_id'] = $this->getCatalogueCollectionId();
        }
        if ($requestedPage == 1) {
            $products = $this->call('GET', '/admin/products.json', $productParams, "products");
        } elseif ($requestedPage == 0) {
            $cursorURL = substr($_SESSION['products']['previous'], strpos($_SESSION['products']['previous'], "/admin"));
            $products = $this->call('GET', $cursorURL, array(), "products");
        } else {
            $cursorURL = substr($_SESSION['products']['next'], strpos($_SESSION['products']['next'], "/admin"));
            $products = $this->call('GET', $cursorURL, array(), "products");
        }
        if (!empty($products)) {
            $keys = array_column($products, $orderBy);
            array_multisort($keys, $order, $products);
        }
        return $products;
    }

    /**
     * geting product search details from store
     * 
     * @author sonali@imprintnext.com
     * @date 25 Apr 2023
     * @param search name
     * @return product details
     */
    private function getSearchProduct($searchstring)
    {
        $payload = '{
            "query": "query { products(first: 20, query:\" NOT vendor:imprintNext title:*' . $searchstring . '*\") { edges { node { id title productType handle vendor totalInventory images (first: 3) { edges { node { id url } } }  variants(first: 1) { edges { node { id title sku price inventoryQuantity inventoryPolicy sellableOnlineQuantity } } } } } } }"
            }';
        $productsData = $this->graphQlRequest('POST', $payload);
        $productsData = $productsData['data']['products']['edges'];
        foreach ($productsData as $nodeData) {
            $imagesArr = [];
            $variants = [];
            foreach ($nodeData['node']['images'] as $prodImage) {
                foreach ($prodImage as $image) {
                    $imagesArr[] = [
                        "src" => $image['node']['url'],
                    ];
                }
            }
            $inventoryPolice = $nodeData['node']['variants']['edges'][0]['node']['inventoryPolicy'];
            $stock = $inventoryPolice == "CONTINUE" ? 10000 : $nodeData['node']['variants']['edges'][0]['node']['inventoryQuantity'];
            $variants[] = [
                "title" => $nodeData['node']['variants']['edges'][0]['node']['title'],
                "id" => substr($nodeData['node']['variants']['edges'][0]['node']['id'], 29),
                "price" => $nodeData['node']['variants']['edges'][0]['node']['price'],
                "sku" => $nodeData['node']['variants']['edges'][0]['node']['sku'],
                "inventory_quantity" => $stock
            ];
            $products[] = [
                "id" => substr($nodeData['node']['id'], 22),
                "title" => $nodeData['node']['title'],
                "vendor" => $nodeData['node']['vendor'],
                "product_type" => $nodeData['node']['productType'],
                "handle" => $nodeData['node']['handle'],
                "images" => $imagesArr,
                "variants" => $variants,
            ];
        }
        return $products;
    }

    /**
     * geting product details from store
     * 
     * @author imprintNext
     * @date 20 Dec 2018
     * @param product id
     * @return product details
     */
    public function getProductAttrOptions($filter)
    {
        $productDetails = $this->call('GET', '/admin/products/' . $filter['productID'] . '.json');
        $variantDetails = $this->call('GET', '/admin/variants/' . $filter['variantID'] . '.json');
        $thisOptionPos = $this->getAttributePostion($filter['option'], $filter['productID']);
        $thisColorPos = $this->getAttributePostion($this->colorAttrName, $filter['productID']);
        $metaData = $this->call('GET', '/admin/products/' . $filter['productID'] . '/metafields.json');
        $tierPriceData = array();
        $commonTierPrice = array();
        $variantTierPrice = array();
        $sameforAllVariants = $isTier = false;
        if (!empty($metaData)) {
            foreach ($metaData as $meta) {
                if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') {
                    $tierContent = $meta['value'];
                    $tierPriceData = json_decode($tierContent, true);
                    $isTier = true;
                    break;
                }
            }
            // little confusion with the key here but was done as sent from admin form field
            if ($tierPriceData['pricing_per_variants'] == 'true') {
                $sameforAllVariants = true;
                foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                    $commonTierPrice[] = array(
                        "upper_limit" => $discount['upper_limit'],
                        "lower_limit" => $discount['lower_limit'],
                        "discount" => $discount['discount'],
                        "discountType" => $tierPriceData['discount_type']
                    );
                }
            } else {
                foreach ($tierPriceData['price_rules'] as $variant) {
                    foreach ($variant['discounts'] as $discount) {
                        $variantTierPrice[$variant['id']][] = array(
                            "upper_limit" => $discount['upper_limit'],
                            "lower_limit" => $discount['lower_limit'],
                            "discount" => $discount['discount'],
                            "discountType" => $tierPriceData['discount_type']
                        );
                    }
                }
            }
        }
        if (!empty($thisOptionPos)) {
            $options = $checkOptions = array();
            for ($pos = 1; $pos < $thisOptionPos; $pos++) {
                $checkOptions[] = $variantDetails['option' . $pos];
            }
            foreach ($productDetails['variants'] as $variant) {
                $thisOptions = $optionDetails = array();
                for ($position = 1; $position <= $thisOptionPos; $position++) {
                    $thisOptions[] = $variant['option' . $position];
                }
                if (!empty($variant['imp_image_id'])) {
                    $ProductVar = new ProductVariants();
                    $catlogVar = $ProductVar->where('xe_id', '=', $variant['id'])->first()->toArray();
                    $variant['inventory_quantity'] = $catlogVar['inventory'];
                }
                if (count(array_intersect($checkOptions, $thisOptions)) == ($thisOptionPos - 1) && ($variantDetails['option' . $thisColorPos] == $variant['option' . $thisColorPos] || empty($variantDetails['option' . $thisColorPos]))) {
                    $collectedOptions = (!empty($options[$filter['option']])) ? array_column($options[$filter['option']], 'name') : array();
                    if (!in_array($variant['option' . $thisOptionPos], $collectedOptions)) {
                        $optionDetails = array();
                        $optionDetails['id'] = $variant['option' . $thisOptionPos];
                        $optionDetails['name'] = $variant['option' . $thisOptionPos];
                        $optionDetails['variant_id'] = $variant['id'];
                        if ($filter['is_price'] > 0) {
                            $optionDetails['inventory']['stock'] = ($variant['inventory_policy'] == "continue" || empty($variant['inventory_management'])  ? 100000 : $variant['inventory_quantity']);
                            $optionDetails['inventory']['min_quantity'] = 1;
                            $optionDetails['inventory']['max_quantity'] = ($variant['inventory_policy'] == "continue" || empty($variant['inventory_management']) ? 100000 : $variant['inventory_quantity']);
                            $optionDetails['inventory']['quantity_increments'] = 1;
                            $optionDetails['price'] = $variant['price'];
                            $optionDetails['tier_prices'] = [];
                            if ($isTier) {
                                $optionDetails['tier_prices'] = ($sameforAllVariants === true ? $this->createTierPrice($commonTierPrice, $variant['price']) : $this->createTierPrice($variantTierPrice[$variant['id']], $variant['price']));
                            }
                            $optionDetails['attributes'] = array();
                            for ($count = 1; $count <= 3; $count++) {
                                if (isset($variant['option' . $count]) && !empty($variant['option' . $count]) && $variant['option' . $count] != "Default Title") {
                                    $optCount = $count - 1;
                                    $attributeKey = strtolower($productDetails['options'][$optCount]['name']);
                                    $optionDetails['attributes'][$attributeKey] = $variant['option' . $count];
                                    $optionDetails['attributes'][$attributeKey . "_id"] = strtolower($variant['option' . $count]);
                                }
                            }
                        }
                        $options[$filter['option']][] = $optionDetails;
                    }
                }
            }
            return $options;
        } else {
            print_r("Attribute not found !!!");
            exit();
        }
    }

    private function createTierPrice($tierPriceRule, $variantPrice)
    {
        $tierPrice = array();
        foreach ($tierPriceRule as $tier) {
            $thisTier = array();
            $thisTier['quantity'] = $tier['lower_limit'];
            $thisTier['percentage'] = ($tier['discountType'] == "percentage" ? $tier['discount'] : number_format(($tier['discount'] / $variantPrice) * 100, 2));
            $thisTier['price'] = ($tier['discountType'] == "flat" ? ($variantPrice - $tier['discount']) : ($variantPrice - (($tier['discount'] / 100) * $variantPrice)));
            $thisTier['discount'] = $tier['discount'] . "_" . $tier['discountType'];
            $tierPrice[] = $thisTier;
        }

        return $tierPrice;
    }


    /**
     * geting product details from store
     * 
     * @author inkXE
     * @date 20 Dec 2018
     * @param product id
     * @return product details
     */
    public function getShopProductDetails($productId)
    {
        $product = $this->call('GET', '/admin/products/' . $productId . '.json');
        $impImagId = $product['variants'][0]['imp_image_id'];
        $variant = $product['variants'][0];
        $colorPos = $this->getAttributePostion($this->colorAttrName, $productId);
        foreach ($product['images'] as $pi) {
            $images[$pi['id']] = $pi;
        }
        // Get Product Attributes
        $attributes = [];
        foreach ($product['options'] as $attribute) {
            if (isset($attribute['name']) && strtolower($attribute['name']) != "title") {
                $attributeValues = array();
                $thisAttribute['id'] = strtolower($attribute['name']);
                $thisAttribute['name'] = $attribute['name'];
                foreach ($attribute['values'] as $storeSizeKey => $storeSize) {
                    $attributeValues[$storeSizeKey] = [
                        'id' => $storeSize,
                        'name' => $storeSize
                    ];
                }
                $thisAttribute['options'] = $attributeValues;
                $attributes[] = $thisAttribute;
            }
        }

        //Choosing variant price as product price
        $productPrice = 10000000;
        foreach ($product['variants'] as $pv) {
            if ($pv['price'] < $productPrice) {
                $productPrice = $pv['price'];
                break;
            }
        }

        $productType = (strpos(strtolower($product['variants'][0]['title']), 'default title') !== false ? "simple" : "variable");
        $products = [
            'id' => $product['id'],
            'variant_id' => $product['variants'][0]['id'],
            'name' => $product['title'],
            'description' => $product['body_html'],
            'type' => $productType,
            'sku' => (count($product['variants']) > 0) ? $product['variants'][0]['sku'] : '',
            'is_decoration_exists' => 0,
            'print_profile' => [],
            'price' => $productPrice,
        ];
        if ($colorPos > 0) {
            $color_look = $this->removeSPLChars($variant['option' . $colorPos]);
            $color_lookup = ($productType == 'simple' ? '_' : $color_look);
        } else {
            // if color is not available first option would be searched in image name
            $color_lookup = "_" . $this->removeSPLChars($variant['option1']) . "_";
        }
        $variantImageID = ($variant['image_id'] ? $variant['image_id'] : $product['image']['id']);
        // get variant images
        if (isset($images) && $variantImageID && array_key_exists($variantImageID, $images)) {
            foreach ($images as $pimg) {
                if (in_array($product['variants'][0]['id'], $pimg['variant_ids'])) {
                    $unassignedImgs['id'] = $pimg['id'];
                    $unassignedImgs['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $unassignedImgs['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                }
                $thisImage = array();
                if ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "front"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Front";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "left"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Left";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "right"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Right";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "back"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Back";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "top"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Top";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "bottom"))) {
                    $thisImage['id'] = $pimg['id'];
                    $thisImage['labels'] = "Bottom";
                    $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                    $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                } else {
                    $thisImage = [];
                }
                if (!empty($thisImage)) {
                    $products['images'][] = $thisImage;
                }
            }
        } else {
            if ($variant['request'] == 'quote') {
                $thisImage['thumbnail'] = array($product['image']['src']);
                $thisImage['src'] = array($product['image']['src']);
                $thisImage['labels'] = "Front";
                $products['images'][] = $thisImage;
            } else {
                $thisImage['thumbnail'] = array();
                $thisImage['src'] = array();
                $thisImage['labels'] = "";
                $products['images'][] = $thisImage;
            }
        }
        if (empty($unassignedImgs)) {
            $unassignedImgs['id'] = $product['images'][0]['id'];
            $unassignedImgs['src'] = self::shopify_image($product['images'][0]['src'], $this->fullSize);
            $unassignedImgs['thumbnail'] = self::shopify_image($product['images'][0]['src'], $this->thumbSize);
        }
        if (empty($products['images'])) {
            $products['images'][] = $unassignedImgs;
        }
        if (!empty($impImagId)) {
            $newSideImage = new ProductImageSides();
            $getImgSides = $newSideImage->where('product_image_id', '=', $impImagId)->get()->toArray();
            foreach ($getImgSides as $imgSide) {
                $dbImg[] = [
                    'id' => $imgSide['xe_id'],
                    'labels' => $imgSide['side_name'],
                    'src' => $imgSide['file_name'],
                    'thumbnail' => $imgSide['thumbnail']
                ];
            }
            $products['images'] = $dbImg;
        }


        $products['slug'] = $product['title'];
        $products['categories'] = $this->getProductCategories($productId);
        $products['attributes'] = $attributes;
        $products['date_created'] = date('Y-m-d', strtotime($product['created_at']));
        if (isset($products) && is_array($products) && !empty($products)) {
            $storeResponse = [
                'products' => $products
            ];
        } else {
            $storeResponse = [
                'status' => 0,
                'message' => 'No products available',
                'data' => []
            ];
        }

        return $storeResponse;
    }

    /**
     * Counting total no of products from store(Numeric value).
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param filter parameters
     * @return integer (Total no of products numeric value)
     */
    public function getTotalProductsCount($filters, $isCustomize = '')
    {
        if (!empty($isCustomize) && $isCustomize == "1") {
            $showInDesignerID = $this->getPreDesignedCollectionId();
        } else {
            $showInDesignerID = $this->getShowInDesignerId();
        }
        if (!empty($filters['collection_id'])) {
            $showInDesignerID = $filters['collection_id'];
        }
        $collections = $this->call('GET', '/admin/custom_collections/' . $showInDesignerID . '.json');
        if (empty($collections['products_count'])) {
            $collections = $this->call('GET', '/admin/smart_collections/' . $showInDesignerID . '.json');
        }
        return $collections['products_count'];
    }
    /**
     * Update smart collection rules.
     * 
     * @author sonali@imprintnext.com
     * @date 10 Feb 2023
     * @param arraynull
     * @return array of collections
     */

    public function getCollectionsData($colId)
    {
        $imprintSmartRules = [
            "column" => "vendor",
            "relation" => "not_equals",
            "condition" => "imprintNext"
        ];
        $smartCollections = $this->call('GET', '/admin/smart_collections/' . $colId . '.json');
        if ($smartCollections['method'] == "GET") {
            $collection = [
                "status" => 0,
                "message" => "it's not a smart collection"
            ];
        } else {
            foreach ($smartCollections['rules'] as $colrule) {
                if (!empty(array_diff($colrule, $imprintSmartRules))) {
                    $smartCollections['rules'][] = $imprintSmartRules;
                    $collectionArray = array(
                        "smart_collection" => array(
                            "rules" => array_unique($smartCollections['rules'], SORT_REGULAR),
                            "disjunctive" => false,
                            "published_scope" => false,
                        )
                    );
                    $this->call('PUT', '/admin/smart_collections/' . $smartCollections['id'] . '.json', $collectionArray);
                    $collection = [
                        "status" => 1,
                        "message" => "Smart collection updated sucessfully."
                    ];
                }
            }
        }
        return $collection;
    }

    /**
     * Getting Collections/attributes from store
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param arraynull
     * @return array of collections
     */
    public function getCollections()
    {
        $checkCollections = array();
        $result = array();
        $customCollections = $this->call('GET', '/admin/custom_collections.json?limit=250');
        $smartCollections = $this->call('GET', '/admin/smart_collections.json?limit=250');
        foreach ($customCollections as $collection) {
            if ($collection['handle'] != "show-in-designer" && $collection['handle'] != "pre-designed-product" && !in_array($collection['title'], $checkCollections)) {
                $checkCollections[] = $collection['title'];
                $result[] = array(
                    'id' => $collection['id'],
                    'name' => $collection['title'],
                    'type' => 'custom',
                    'parent_id' => 0
                );
            }
        }
        foreach ($smartCollections as $collection) {
            if ($collection['handle'] != "customized" && $collection['handle'] != "all" && $collection['handle'] != "imprint_catalogue"  && $collection['handle'] != "show-in-designer") {
                $checkCollections[] = $collection['title'];
                $result[] = array(
                    'id' => $collection['id'],
                    'name' => $collection['title'],
                    'type' => 'smart',
                    'parent_id' => 0
                );
            }
        }
        return $result;
    }

    /**
     * Getting smart collections from store
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param null
     * @return array of ids 
     */
    public function getSmartCollections()
    {
        $collectionIds = array();
        $smartCollections = $this->call('GET', '/admin/smart_collections.json?limit=250&fields=id');
        foreach ($smartCollections as $collection) {
            $collectionIds[] = $collection['id'];
        }
        return $collectionIds;
    }
    /**
     * Fetch all categories for a single product
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param product id
     * @return array of categories 
     */
    public function getProductCategories($productId)
    {
        $productData = json_decode($productId, true);
        $productId = !empty($productData['pid']) ? $productData['pid'] : $productId;
        $customCategories = $smtCategories = array();
        $categories = $this->call('GET', '/admin/custom_collections.json?limit=250&product_id=' . $productId);
        $smartCategories = $this->call('GET', '/admin/smart_collections.json?limit=250&product_id=' . $productId);
        foreach ($categories as $category) {
            if ($category['handle'] != 'pre-designed-product' && $category['handle'] != 'show-in-designer') {
                $customCategories[] = [
                    'id' => $category['id'],
                    'name' => $category['title'],
                    'slug' => $category['handle'],
                    'parent_id' => 0,
                ];
            }
        }
        foreach ($smartCategories as $sCategory) {
            $excludeColls = ["all", "customized", "catalog", "show-in-designer"];
            if (!in_array($sCategory['handle'], $excludeColls)) {
                $smtCategories[] = [
                    'id' => $sCategory['id'],
                    'name' => $sCategory['title'],
                    'slug' => $sCategory['handle'],
                    'parent_id' => 0,
                ];
            }
        }
        return array_merge($customCategories, $smtCategories);
    }

    /**
     * Getting only color attribute list from store
     * 
     * @author inkXE
     * @date 16 Dec 2018
     * @param null
     * @return array of color attribute value
     */
    public function getColorAttributes()
    {
        $filterParams['collection_id'] = $this->getShowInDesignerId();
        $filterParams['limit'] = 250;
        $products = $this->call('GET', '/admin/products.json', $filterParams);
        $colors = array();
        $checkColors = array();
        foreach ($products as $p) {
            $colorPos = '';
            foreach ($p['options'] as $option) {
                if (strtolower($option['name']) == $this->colorAttrName) {
                    $colorPos = $option['position'];
                }
            }
            if ($colorPos != '') {
                foreach ($p['variants'] as $pv) {
                    if ($pv['option' . $colorPos] !== null && !in_array(str_replace(' ', '_', strtolower($pv['option' . $colorPos])), $checkColors)) {
                        $colors['data'][] = array(
                            'id' => str_replace([' ','/'], '_', strtolower($pv['option' . $colorPos])),
                            'name' => strtolower($pv['option' . $colorPos]),
                        );
                        $checkColors[] = str_replace(' ', '_', strtolower($pv['option' . $colorPos]));
                    }
                }
            }
        }
        return $colors;
    }
    /**
     * Fetching all attributes from store
     *
     * @author inkXE
     * @date 18 Dec 2019
     * @param null
     * @return array - Array of all attribute list
     */
    public function getAttributes($productID = "", $fetchRaw = false)
    {
        $filterParams = [];
        $dataKey = "terms";
        if (!empty($productID)) {
            $product = $this->call('GET', '/admin/products/' . $productID . '.json', $filterParams, '', $fetchRaw);
            $products[] = $product;
            $dataKey = "options";
        } else {
            $filterParams = [
                'collection_id' => $this->getShowInDesignerId(),
                'limit' => 250
            ];
            $products = $this->call('GET', '/admin/products.json', $filterParams, '', $fetchRaw);
        }
        $options = array();
        $checkOptions = array();
        foreach ($products as $p) {
            foreach ($p['options'] as $option) {
                if (!in_array(str_replace(' ', '_', strtolower($option['name'])), $checkOptions) && strtolower($option['name']) != 'title') {
                    $checkOptions[] = strtolower($option['name']);
                    $thisOption = array(
                        'id' => str_replace(' ', '_', strtolower($option['name'])),
                        'name' => $option['name'],
                        'type' => $option['name']
                    );
                    $optionValueList = array();
                    foreach ($option['values'] as $value) {
                        $optionValueList[] = [
                            'id' => str_replace(' ', '_', strtolower($value)),
                            'name' => $value
                        ];
                    }
                    $thisOption[$dataKey] = $optionValueList;
                    $options[] = $thisOption;
                } else {
                    foreach ($options as $key => $op) {
                        if (str_replace(' ', '_', strtolower($op['name'])) == str_replace(' ', '_', strtolower($option['name']))) {
                            $checkTerms = array();
                            foreach ($op['terms'] as $term) {
                                $checkTerms[] = $term['id'];
                            }
                            foreach ($option['values'] as $value) {
                                if (!in_array(str_replace(' ', '_', strtolower($value)), $checkTerms)) {
                                    $options[$key]['terms'][] = [
                                        'id' => str_replace(' ', '_', strtolower($value)),
                                        'name' => $value
                                    ];
                                }
                            }
                        }
                    }
                }
            }
        }
        return $options;
    }

    /**
     * Retrieve a single product's category(not used currently)
     *
     * @author inkXE
     * @date 18 Dec 2019
     * @param array|object Request parameters
     * @return string New product id
     */
    public function createDecoratedProduct($params)
    {
        $metaDataInfo = array();
        $imagesInfo = array();
        $colorVarArr = array();
        if (isset($params['type']) && $params['type'] != "") {
            $productType = $params['type'];
        }

        // Ask to send this paramaeter individually
        foreach ($params['attributes'] as $attribute) {
            if (strtolower($attribute['name']) == $this->colorAttrName) {
                $color = $attribute['attribute_options'];
                break;
            }
        }
        // get parent product details
        if (isset($params['product_id']) && $params['product_id'] > 0) {
            $thisProdImgs = $this->call('GET', '/admin/products/' . $params['product_id'] . '.json?fields=images');
        }

        if (isset($params['parent_product_id']) && $params['parent_product_id'] > 0) {
            $parentProduct = $this->call('GET', '/admin/products/' . $params['parent_product_id'] . '.json');
        } else {
            $parentProduct = array();
        }
        //get all variant colors
        if (isset($params['color_images']) && !empty($params['color_images'])) {
            foreach ($params['color_images'] as $colorImg) {
                $colorVarArr[] = $colorImg['id'];
            }
        }

        $imgArr = array();
        foreach ($params['product_image_url'] as $key => $img) {
            $key = $key + 1;
            array_push($imgArr, array("src" => $img, "position" => $key));
        }
        $imgCount = ($params['product_id'] == '' ? (count($imgArr) + 1) : (count($thisProdImgs) + 1));
        //get parent variant product images
        if ($params['product_id'] == '' && $params['ref_id'] > 0) {
            $selImgs = array();
            foreach ($parentProduct['images'] as $prodImg) {
                foreach ($colorVarArr as $eachVarColor) {
                    if (strpos(strtolower($prodImg['src']), strtolower($eachVarColor)) !== false) {
                        $selImgs[] = array("src" => $prodImg['src'], "position" => $imgCount);
                        $imgCount++;
                    }
                }
            }
            $imgArr = array_merge($imgArr, $selImgs);
        }
        if (isset($params['color_images'])) {
            $sides = array_column($params['color_images'], 'sides');
            $sides = array_merge(...$sides);
            $colorImage = [];
            $i = 1;
            foreach ($sides as $value) {
                if (!in_array($value, $params['product_image_url'])) {
                    $colorImage[] = ["src" => $value, "position" => sizeof($imgArr) + $i];
                    $i++;
                }
            }
            $imgArr = array_merge($imgArr, $colorImage);
        }

        $imagesInfo =  $imgArr;

        $optionArr = array();
        $variantArr = array();
        if (!empty($params['attributes'])) {
            foreach ($params['attributes'] as $key => $attribute) {
                $thisAttr = [];
                $key += 1;
                $thisAttr['name'] = $attribute['attribute_name'];
                $thisAttr['position'] = $key;
                $thisAttr['values'] = $attribute['attribute_options'];
                $optionArr[] = $thisAttr;
                $inputOptions[] = $attribute['attribute_options'];
            }
            $variantCombinations = $this->variantCombinations($inputOptions);

            foreach ($variantCombinations as $varKey => $optionSet) {
                $varKey++;
                $thisVariant = array(
                    "sku" => $params['sku'] . "_" . $varKey,
                    "price" => $params['price'],
                    "inventory_management" => "shopify",
                    "inventory_policy" => "deny"
                );
                if (is_array($optionSet)) {
                    foreach ($optionSet as $key => $option) {
                        $key++;
                        $thisVariant["option" . $key] = $option;
                    }
                } else {
                    $thisVariant["option1"] = $optionSet;
                }
                $variantArr[] = $thisVariant;
            }
        }
        // if ($params['product_id'] == ''){
        $productArray = array(
            "product" => array(
                "title" => $this->shopify_body(addslashes($params['name'])),
                "body_html" => $this->shopify_body(addslashes($params['description'])),
                "published" => true,
                "tags" => "Pre-Deco",
                "images" => $imagesInfo,
                "image" => array("src" => $params['product_image_url'][0], "position" => 1),
            ),
        );


        if (!empty($optionArr) && !empty($variantArr)) {
            $productArray['product']['options'] = $optionArr;
            $productArray['product']['variants'] = $variantArr;
        }


        $metaDataInfo = array(
            "metafield" => array(
                "namespace" => "Pre-Deco",
                "key" => "imprint_design_id",
                "value" => $params['ref_id'],
                "type" => "number_integer",
            ),
        );
        $newProduct = $this->call('POST', '/admin/products.json', $productArray);
        if (!empty($metaDataInfo)) {
            $this->call('POST', '/admin/products/' . $newProduct['id'] . '/metafields.json', $metaDataInfo);
        }


        if ($newProduct['variants'][0]['inventory_management'] == 'shopify') {
            $newProductVariants = $newProduct['variants'];
            $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
            foreach ($inventoryItems as $inventory) {
                $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventory);
                foreach ($invLevels as $level) {
                    $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
                    if (!$locationData['legacy']) {
                        $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" => $inventory, "available" => $params['quantity']);
                        $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                    }
                }
            }
        }

        // assign product to "Show in designer" collection if is customized is checked.
        if ($params['is_redesign'] == 1) {
            $customArr = array(
                "collect" => array(
                    "product_id" => $newProduct['id'],
                    "collection_id" => $this->getShowInDesignerId(),
                ),
            );
            $this->call('POST', '/admin/collects.json', $customArr);
        }

        // Add pre-designed-collection-id
        $customArr = array(
            "collect" => array(
                "product_id" => $newProduct['id'],
                "collection_id" => $this->getPreDesignedCollectionId(),
            ),
        );
        $this->call('POST', '/admin/collects.json', $customArr);

        // assign categories one by one (collections)
        $smartCollectionIds = $this->getSmartCollections();
        if (!empty($params['categories'])) {
            foreach ($params['categories'] as $category) {
                if (!in_array($category['category_id'], $smartCollectionIds)) {
                    $categoryArr = array(
                        "collect" => array(
                            "product_id" => $newProduct['id'],
                            "collection_id" => $category['category_id'],
                        ),
                    );
                    $this->call('POST', '/admin/collects.json', $categoryArr);
                } else {
                    continue;
                }
            }
        }
        // } else {
        //     // Get new product details
        //     $newProduct = $this->call('GET', '/admin/products/' . $params['product_id'] . '.json');
        //     // Get list of existing variants and images
        //     foreach ($newProduct['variants'] as $vrnt) {
        //         $oldVars[] = $vrnt;
        //     }
        //     foreach ($newProduct['images'] as $img) {
        //         $oldImgs[] = $img;
        //     }
        //     $allVarArr = array_merge($oldVars, $variantArr);
        //     $allImgsArr = array_merge($oldImgs, $selImgs);
        //     // assign variants to new product
        //     $productArray = array(
        //         "product" => array(
        //             "id" => $params['product_id'],
        //             "variants" => $allVarArr,
        //             "images" => $allImgsArr,
        //         ),
        //     );
        //     $editProduct = $this->call('PUT', '/admin/products/' . $params['product_id'] . '.json', $productArray);
        //     // update inventory
        //     if ($editProduct['variants'][0]['inventory_management'] == 'shopify') {
        //         $newProductVariants = $editProduct['variants'];
        //         $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
        //         foreach ($inventoryItems as $inventory) {
        //             $invLevels = $this->call('GET', '/admin/api/2019-04/inventory_levels.json?inventory_item_ids='.$inventory);
        //             foreach ($invLevels as $level) {
        //                 $locationData = $this->call('GET', '/admin/api/2019-04/locations/'.$level['location_id'].'.json');
        //                 if ($locationData['legacy'] == false) {
        //                     $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" =>$inventory, "available" => $params['quantity']);
        //                     $updateInventory = $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
        //                 }
        //             }
        //         }
        //     }
        //     // Get final product details
        //     $newProduct = $this->call('GET', '/admin/products/' . $params['product_id'] . '.json');
        // }

        // get images of new products
        $newProductImage = array();
        foreach ($newProduct['images'] as $img) {
            array_push($newProductImage, array("id" => $img['id'], "src" => $img['src']));
        }

        //assign image to all variants of the product
        foreach ($newProduct['variants'] as $variant) {
            if (!$variant['image_id']) {
                $color = str_replace(array('/', '_'), '', $variant['option1']);
                $imgID = 0;
                foreach ($newProductImage as $image) {
                    if (strpos(strtolower($image['src']), strtolower($color)) !== false && strpos(strtolower($image['src']), '_front') !== false) {
                        $imgID = $image['id'];
                        break;
                    }
                }
                if ($imgID > 0) {
                    $variantArr = array(
                        "variant" => array(
                            "id" => $variant['id'],
                            "image_id" => $imgID,
                        ),
                    );
                    $this->call('PUT', '/admin/variants/' . $variant['id'] . '.json', $variantArr);
                }
            }
        }
        return [
            'product_id' => $newProduct['id'],
        ];
    }

    private function variantCombinations($arrays, $i = 0)
    {
        if (!isset($arrays[$i])) {
            return array();
        }
        if ($i == count($arrays) - 1) {
            return $arrays[$i];
        }

        // get combinations from subsequent arrays
        $tmp = $this->variantCombinations($arrays, $i + 1);
        $combinations = array();
        // concat each array from tmp with each element from $arrays[$i]
        foreach ($arrays[$i] as $v) {
            foreach ($tmp as $t) {
                $combinations[] = is_array($t) ?
                    array_merge(array($v), $t) :
                    array($v, $t);
            }
        }
        return $combinations;
    }

    /**
     * Creating predecorated product for simple product
     * This method will be optimised in future enhancement
     * 
     * @author inkXE
     * @date 18 Dec 2019
     * @param array
     * @return array New product details
     */
    public function createSimpleDecoratedProduct($params)
    {
        $imgArr = array();
        $optionArr = array();
        $parentProductId = $params['parent_product_id'];
        $images = $params['product_image_url'];
        $productInfo = $this->call('GET', '/admin/products/' . $parentProductId . '.json?fields=variants,options,images');
        $thisProdImgs = $productInfo['images'];
        // get parent product images
        $imgArr = array();
        foreach ($images as $key => $img) {
            $key = $key + 1;
            array_push($imgArr, array("src" => $img, "position" => $key));
        }
        $imgCount = ($params['product_id'] == '' ? (count($imgArr) + 1) : (count($thisProdImgs) + 1));
        $selImgs = array();
        foreach ($thisProdImgs as $prodImg) {
            $selImgs[] = array("src" => $prodImg['src'], "position" => $imgCount);
            $imgCount++;
        }
        $imgArr = array_merge($imgArr, $selImgs);
        $productArray = array(
            "product" => array(
                "title" => self::shopify_body(addslashes($params['name'])),
                "body_html" => self::shopify_body(addslashes($params['description'])),
                "published" => true,
                "tags" => "Pre-Deco",
                "images" => $imgArr,
                "image" => array("src" => $params['product_image_url'][0], "position" => 1),
            ),
        );
        $isExtraAttr = false;
        if ($productInfo['variants'][0]['title'] !== 'Default Title') {
            $optionArr = array();
            $variantArr = array();
            foreach ($productInfo['options'] as $key => $opt) {
                array_push($optionArr, array("name" => $opt['name'], "position" => $opt['position']));
            }
            foreach ($productInfo['variants'] as $key => $variant) {
                array_push(
                    $variantArr,
                    array(
                        "option1" => $variant['option1'],
                        "option2" => $variant['option2'],
                        "option3" => $variant['option3'],
                        "sku" => $params['sku'],
                        "price" => $params['price'],
                        "inventory_management" => "shopify",
                        "inventory_policy" => "deny"
                    )
                );
            }
        }
        if (!empty($optionArr)) {
            $productArray['product']['options'] = $optionArr;
        }
        if (!empty($variantArr)) {
            $isExtraAttr = true;
            $productArray['product']['variants'] = $variantArr;
        }
        $newProduct = $this->call('POST', '/admin/products.json', $productArray);
        $newProductVariants = $newProduct['variants'];
        $variantArr = array(
            "variant" => array(
                "inventory_management" => "shopify",
                "inventory_policy" => "deny",
                "price" => $params['price'],
                "image_id" => $newProduct['images'][1]['id']
            )
        );
        if (!$isExtraAttr) {
            $this->call('PUT', '/admin/variants/' . $newProductVariants[0]['id'] . '.json', $variantArr);
        }
        if ($newProduct['variants'][0]['inventory_management'] == 'shopify') {
            $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
            foreach ($inventoryItems as $inventory) {
                $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventory);
                foreach ($invLevels as $level) {
                    $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
                    if (!$locationData['legacy']) {
                        $newInventory = array(
                            "location_id" => $level['location_id'],
                            "inventory_item_id" => $inventory,
                            "available" => $params['quantity'] - 1
                        );
                        $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                    }
                }
            }
        }
        // assign product to "Show in designer" collection if is customized is checked.
        if ($params['is_redesign'] == '1') {
            $customArr = array(
                "collect" => array(
                    "product_id" => $newProduct['id'],
                    "collection_id" => $this->getShowInDesignerId(),
                ),
            );
            $this->call('POST', '/admin/collects.json', $customArr);
        }
        // Add pre-designed-collection-id
        $customArr = array(
            "collect" => array(
                "product_id" => $newProduct['id'],
                "collection_id" => $this->getPreDesignedCollectionId(),
            ),
        );
        $this->call('POST', '/admin/collects.json', $customArr);

        // assign categories one by one (collections)
        $smartCollectionIds = $this->getSmartCollections();
        if (!empty($params['categories'])) {
            foreach ($params['categories'] as $category) {
                if (!in_array($category['category_id'], $smartCollectionIds)) {
                    $categoryArr = array(
                        "collect" => array(
                            "product_id" => $newProduct['id'],
                            "collection_id" => $category['category_id'],
                        ),
                    );
                    $this->call('POST', '/admin/collects.json', $categoryArr);
                } else {
                    continue;
                }
            }
        }
        return $newProduct;
    }

    /**
     * GET: Variant information by ID
     * 
     * @author inkXE
     * @date 18 Dec 2019
     * @param $variant variant ID 
     * @return array New product details
     */
    public function getVariantShortInfo($variant)
    {
        $variantID = $variant['variantID'];
        $productId = $variant['productID'];
        $product = $this->call('GET', '/admin/products/' . $productId . '.json');
        if ($variantID == $productId) {
            $variantID = $product['variants'][0]['id'];
        }
        $variant = $this->call('GET', '/admin/variants/' . $variantID . '.json');
        $colorPos = $this->getAttributePostion($this->colorAttrName, $productId);
        $this->getAttributePostion($this->sizeAttrName, $productId);
        // for tier price
        $metaData = $this->call('GET', '/admin/products/' . $productId . '/metafields.json');
        $tierPriceData = array();
        $commonTierPrice = array();
        $variantTierPrice = array();
        $sameforAllVariants = $isTier = false;
        if (!empty($metaData)) {
            foreach ($metaData as $meta) {
                if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') {
                    $tierContent = $meta['value'];
                    $tierPriceData = json_decode($tierContent, true);
                    $isTier = true;
                    break;
                }
            }
            // little confusion with the key here but was done as sent from admin form field
            if ($tierPriceData['pricing_per_variants'] == 'true') {
                $sameforAllVariants = true;
                foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                    $commonTierPrice[] = array(
                        "upper_limit" => $discount['upper_limit'],
                        "lower_limit" => $discount['lower_limit'],
                        "discount" => $discount['discount'],
                        "discountType" => $tierPriceData['discount_type']
                    );
                }
            } else {
                foreach ($tierPriceData['price_rules'] as $var) {
                    if ($var['id'] == $variantID) {
                        foreach ($var['discounts'] as $discount) {
                            $variantTierPrice[$variant['id']][] = array(
                                "upper_limit" => $discount['upper_limit'],
                                "lower_limit" => $discount['lower_limit'],
                                "discount" => $discount['discount'],
                                "discountType" => $tierPriceData['discount_type']
                            );
                        }
                    }
                }
            }
        }
        // for tier price
        foreach ($product['images'] as $pi) {
            $images[$pi['id']] = $pi;
        }
        $variantDetails = array();


        if (!empty($variant)) {

            $variantDetails['name'] = $product['title'];
            $variantDetails['price'] = $variant['price'];
            $variantDetails['attributes'] = array();
            // position value is increamented+1 while saving as json
            $attributeNames = array_column($product['options'], 'name');
            if (!empty($attributeNames)) {
                array_walk($attributeNames, function (&$name) {
                    $name = strtolower($name);
                });
                // keep the color at the begining
                $colorAttrPos = array_search($this->colorAttrName, $attributeNames);
                if ($colorAttrPos !== false) {
                    $colorOption = [$colorAttrPos => $attributeNames[$colorAttrPos]];
                    unset($attributeNames[$colorAttrPos]);
                    $attributeNames = $colorOption + $attributeNames;
                }
                foreach ($attributeNames as $position => $attributeName) {
                    $position++;
                    $variantOption = $variant['option' . $position];
                    if (strtolower($variantOption) != "default title" && SHOPIFY_VARIANTS_PLUGIN != 1) {
                        $variantDetails['attributes'][$attributeName] = [
                            'id' => strtolower($variantOption),
                            'name' => $variantOption,
                            'attribute_id' => $attributeName
                        ];
                    } else {
                        $productVarOption = $product['variants'][0]['option' . $position];
                        if(strtolower($productVarOption) != "default title") {
                        $variantDetails['attributes'][$attributeName] = [
                            'id' => strtolower($productVarOption),
                            'name' => $productVarOption,
                            'attribute_id' => $attributeName
                        ];
                      }
                    }
                }
            }

            if ($isTier) {
                $variantDetails['tier_prices'] = ($sameforAllVariants === true ? $this->createTierPrice($commonTierPrice, $variant['price']) : $this->createTierPrice($variantTierPrice[$variant['id']], $variant['price']));
            }
            $variantDetails['images'] = array();
            $variantDetails['categories'] = $this->getProductCategories($productId);
            $thisProdType = (strpos(strtolower($variant['title']), 'default title') !== false ? "simple" : "configurable");
            if ($variantID == $productId) {
                $thisProdType = "simple";
            }

            // if image is set on the custom variant (StoreProductController will handle the images )
            if (!empty($variant['imp_image_id'])) {
                $variantDetails['imp_image_id'] = $variant['imp_image_id'];
                return $variantDetails;
            }
            if ($colorPos > 0) {
                $color_look = $this->removeSPLChars($variant['option' . $colorPos]);
                $color_lookup = ($thisProdType == 'simple' ? '_' : $color_look);
            } else {
                // if color is not available first option would be searched in image name
                $color_lookup = "_" . $this->removeSPLChars($variant['option1']) . "_";
            }
            $variantDetails['color_lookup'] = $color_lookup;
            $variantImageID = ($variant['image_id'] ? $variant['image_id'] : $product['image']['id']);
            // get variant images
            if (isset($images) && $variantImageID && array_key_exists($variantImageID, $images)) {
                foreach ($images as $pimg) {
                    if (in_array($variantID, $pimg['variant_ids'])) {
                        $unassignedImgs['labels'] = "front";
                        $unassignedImgs['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $unassignedImgs['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    }
                    $thisImage = array();
                    if ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "front"))) {
                        $thisImage['labels'] = "Front";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "left"))) {
                        $thisImage['labels'] = "Left";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "right"))) {
                        $thisImage['labels'] = "Right";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "back"))) {
                        $thisImage['labels'] = "Back";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "top"))) {
                        $thisImage['labels'] = "Top";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "bottom"))) {
                        $thisImage['labels'] = "Bottom";
                        $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                        $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                    } else {
                        $thisImage = array();
                    }
                    if (!empty($thisImage)) {
                        $variantDetails['images'][] = $thisImage;
                    }
                }
            } else {
                if ($variant['request'] == 'quote') {
                    $thisImage['thumbnail'] = array($product['image']['src']);
                    $thisImage['src'] = array($product['image']['src']);
                    $thisImage['labels'] = "Front";
                    $variantDetails['images'][] = $thisImage;
                } else {
                    $thisImage['thumbnail'] = array();
                    $thisImage['src'] = array();
                    $thisImage['labels'] = "";
                    $variantDetails['images'][] = $thisImage;
                }
            }
            if (empty($unassignedImgs)) {
                $unassignedImgs['id'] = $product['images'][0]['id'];
                $unassignedImgs['src'] = self::shopify_image($product['images'][0]['src'], $this->fullSize);
                $unassignedImgs['thumbnail'] = self::shopify_image($product['images'][0]['src'], $this->thumbSize);
            }
            if (empty($variantDetails['images'])) {
                $variantDetails['images'][] = $unassignedImgs;
            }
        } else {
            $variantDetails = [];
        }
        return $variantDetails;
    }

    private function str_contains_all($haystack, array $needles)
    {
        foreach ($needles as $needle) {
            if (strpos($haystack, $needle) === false) {
                return false;
            }
        }
        return true;
    }

    /**
     * retrieve images from store
     *
     * @param string $src
     * @param string $size
     * @return string
     */
    private static function shopify_image($src, $size)
    {
        if (empty($size) || strpos($src, '/assets/products/') !== false) {
            return $src;
        }

        $img_path = pathinfo($src);
        return str_replace("." . $img_path['extension'], "_" . $size . "." . $img_path['extension'], $src);
    }

    /**
     * GET: product color options
     *
     * @author inkXE
     * @date 18 Dec 2019
     * @param $filter variant information
     * @return array colors details
     */
    public function getColorOptions($filter)
    {
        $productId = $filter['product_id'];
        $product = $this->call('GET', '/admin/products/' . $productId . '.json');
        foreach ($product['images'] as $pi) {
            $images[$pi['id']] = $pi;
        }
        $colorPos = $this->getAttributePostion($this->colorAttrName, $productId);
        if (empty($colorPos)) {
            $colorPos = 1;
        }
        $colorVariants = array();
        $thisProductColors = array();
        // for tier price
        $metaData = $this->call('GET', '/admin/products/' . $productId . '/metafields.json');
        $tierPriceData = array();
        $commonTierPrice = array();
        $variantTierPrice = array();
        $sameforAllVariants = $isTier = false;

        if (!empty($metaData)) {
            foreach ($metaData as $meta) {
                if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') {
                    $tierContent = $meta['value'];
                    $tierPriceData = json_decode($tierContent, true);
                    $isTier = true;
                    break;
                }
            }
            // little confusion with the key here but was done as sent from admin form field
            if ($tierPriceData['pricing_per_variants'] == 'true') {
                $sameforAllVariants = true;
                foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                    $commonTierPrice[] = array(
                        "upper_limit" => $discount['upper_limit'],
                        "lower_limit" => $discount['lower_limit'],
                        "discount" => $discount['discount'],
                        "discountType" => $tierPriceData['discount_type']
                    );
                }
            } else {
                foreach ($tierPriceData['price_rules'] as $variant) {
                    foreach ($variant['discounts'] as $discount) {
                        $variantTierPrice[$variant['id']][] = array(
                            "upper_limit" => $discount['upper_limit'],
                            "lower_limit" => $discount['lower_limit'],
                            "discount" => $discount['discount'],
                            "discountType" => $tierPriceData['discount_type']
                        );
                    }
                }
            }
        }

        if ($colorPos !== 1) {
            foreach ($product['options'] as $productOption) {
                if ($productOption['position'] == $colorPos) {
                    $attributeData = [
                        "name" => $productOption['name'],
                        "attribute_id" => strtolower($productOption['name'])
                    ];
                }
            }
        } else {
            $attributeData = [
                "name" => $product['options'][0]['name'],
                "attribute_id" => strtolower($product['options'][0]['name'])
            ];
        }

        $colorExist = 0;
        foreach ($product['options'] as $optionsValues) {
            if (strtolower($optionsValues['name']) == strtolower($this->colorAttrName)) {
                $colorExist = 1;
                break;
            }
        }

        // for tier price

        foreach ($product['variants'] as $variant) {
            $colorName = $variant['option' . $colorPos];
            if (!in_array($colorName, $thisProductColors) && $variant['title'] != "Default Title") {
                $thisProductColors[] = $colorName; // color should be unique
                $thisColorVar = [
                    'id' => str_replace([' ','/'], '_' ,strtolower($colorName)),
                    'name' => $colorName,
                    'variant_id' => $variant['id'],
                    'price' => $variant['price'],
                    'attrbitue_name' => $this->colorAttrName,
                    'inventory' => [
                        'stock' => ($variant['inventory_policy'] == "continue" || empty($variant['inventory_management']) ? 100000 : $variant['inventory_quantity']),
                        'min_quantity' => 1,
                        'max_quantity' => 10000,
                        'quantity_increments' => 1
                    ],
                    'sides' => [], // get variant images
                    "attribute" => $attributeData
                ];
                if ($isTier) {
                    $thisColorVar['tier_prices'] = ($sameforAllVariants === true ? $this->createTierPrice($commonTierPrice, $variant['price']) : $this->createTierPrice($variantTierPrice[$variant['id']], $variant['price']));
                }

                if (empty($variant['imp_image_id'])) {
                    if ($colorExist > 0) {
                        $color_look = $this->removeSPLChars($variant['option' . $colorPos]);
                        $color_lookup = ($colorName != '' ? $color_look : '_');
                    } else {
                        // if color is not available first option would be searched in image name
                        $color_lookup = "_" . $this->removeSPLChars($variant['option1']) . "_";
                    }
                    $thisColorVar['color_lookup'] = $color_lookup;

                    $variantImageID = ($variant['image_id'] ? $variant['image_id'] : $product['image']['id']);

                    if (isset($images) && $variantImageID && array_key_exists($variantImageID, $images)) {
                        foreach ($images as $pimg) {
                            if (in_array($variant['id'], $pimg['variant_ids'])) {
                                $unassignedImgs['labels'] = "front";
                                $unassignedImgs['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $unassignedImgs['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            }
                            $thisImage = array();
                            if ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "front"))) {
                                $thisImage['labels'] = "Front";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "left"))) {
                                $thisImage['labels'] = "Left";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "right"))) {
                                $thisImage['labels'] = "Right";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "back"))) {
                                $thisImage['labels'] = "Back";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "top"))) {
                                $thisImage['labels'] = "Top";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } elseif ($this->str_contains_all(strtolower($pimg['src']), array($color_lookup, "bottom"))) {
                                $thisImage['labels'] = "Bottom";
                                $thisImage['src'] = self::shopify_image($pimg['src'], $this->fullSize);
                                $thisImage['thumbnail'] = self::shopify_image($pimg['src'], $this->thumbSize);
                            } else {
                                $thisImage = array();
                            }
                            if (!empty($thisImage)) {
                                $thisColorVar['sides'][] = array('image' => $thisImage);
                            }
                        }
                    } else {
                        if ($variant['request'] == 'quote') {
                            $thisImage['thumbnail'] = array($product['image']['src']);
                            $thisImage['src'] = array($product['image']['src']);
                            $thisImage['labels'] = "Front";
                            $thisColorVar['sides'][] = array('image' => $thisImage);
                        } else {
                            $thisImage['thumbnail'] = array();
                            $thisImage['src'] = array();
                            $thisImage['labels'] = "";
                            $thisColorVar['sides'][] = array('image' => $thisImage);
                        }
                    }
                    if (empty($unassignedImgs)) {
                        $unassignedImgs['id'] = $product['images'][0]['id'];
                        $unassignedImgs['src'] = self::shopify_image($product['images'][0]['src'], $this->fullSize);
                        $unassignedImgs['thumbnail'] = self::shopify_image($product['images'][0]['src'], $this->thumbSize);
                    }
                    if (empty($thisColorVar['sides'])) {
                        $thisColorVar['sides'][] = array('image' => $unassignedImgs);
                    }
                } else {
                    $thisColorVar['imp_image_id'] = $variant['imp_image_id'];
                }
                $colorVariants[] = $thisColorVar;
            }
        }

        if ($colorExist == 1 && $filter['is_admin'] == 1) {
            return $colorVariants;
        } elseif ($filter['is_admin'] == 1) {
            return $colorVariants = [];
        }
        return $colorVariants;
    }

    /**
     * GET: product variant inventory
     * 
     * @author inkXE
     * @date 18 Dec 2019
     * @param $params variant information
     * @return array variant inventory
     */
    public function getVariantInventory($params)
    {
        $productId = $params['productId'];
        $variantId = $params['variantId'];
        $product = $this->call('GET', '/admin/products/' . $productId . '.json');
        if ($variantId == $productId) {
            $variantID = $product['variants'][0]['id'];
        }
        $variant = $this->call('GET', '/admin/variants/' . $variantId . '.json');
        $colorPos = $this->getAttributePostion($this->colorAttrName, $productId);
        $sizePos = $this->getAttributePostion($this->sizeAttrName, $productId);
        $targetColor = $variant['option' . $colorPos];
        $quantities = array();
        if ($variantID != $productId) {
            foreach ($product['variants'] as $variant) {
                if ($variant['option' . $colorPos] == $targetColor) {
                    $thisVariant = array();
                    $thisVariant['variant_id'] = $variant['id'];
                    $thisVariant['price'] = $variant['price'];
                    $thisVariant['inventory']['stock'] = ($variant['inventory_policy'] == "continue" || empty($variant['inventory_management']) ? 100000 : $variant['inventory_quantity']);
                    $thisVariant['inventory']['min_quantity'] = 1;
                    $thisVariant['inventory']['max_quantity'] = 10000;
                    $thisVariant['inventory']['quantity_increments'] = 1;
                    $thisVariant['attributes']['color_id'] = $variant['option' . $colorPos];
                    $thisVariant['attributes']['color'] = $variant['option' . $colorPos];
                    $thisVariant['attributes']['size_id'] = $variant['option' . $sizePos];
                    $thisVariant['attributes']['size'] = $variant['option' . $sizePos];
                    $thisVariant['tier_prices'] = array();
                    $quantities[] = $thisVariant;
                }
            }
        } else {
            $thisVariant = array();
            $thisVariant['variant_id'] = $variant['id'];
            $thisVariant['price'] = $variant['price'];
            $thisVariant['inventory']['stock'] = ($variant['inventory_policy'] == "continue" || empty($variant['inventory_management'])  ? 100000 : $variant['inventory_quantity']);
            $thisVariant['inventory']['min_quantity'] = 1;
            $thisVariant['inventory']['max_quantity'] = 10000;
            $thisVariant['inventory']['quantity_increments'] = 1;
            $thisVariant['attributes']['color_id'] = $variant['option' . $colorPos];
            $thisVariant['attributes']['color'] = $variant['option' . $colorPos];
            $thisVariant['attributes']['size_id'] = $variant['option' . $sizePos];
            $thisVariant['attributes']['size'] = $variant['option' . $sizePos];
            $thisVariant['tier_prices'] = array();
            $quantities[] = $thisVariant;
        }
        return $quantities;
    }

    /**
     * GET: parent variant details
     * 
     * @author inkXE
     * @date 18 Dec 2019
     * @param $productId requested product ID
     * @return string parent variant ID
     */
    public function getParentVariantID($variantID)
    {
        if (is_numeric($variantID)) {
            $variant = $this->call('GET', '/admin/variants/' . $variantID . '.json');
            $thisProductID = $variant['product_id'];
            $metaData = $this->call('GET', '/admin/products/' . $thisProductID . '/metafields.json?key=imprint_prod_rel');
            if (!empty($metaData)) {
                $idRelation = json_decode($metaData[0]['value'], true);
            }
            $parentVarID = $idRelation[$variantID];
            $productType = $idRelation['product_type'];
            if ($productType == 'simple') {
                $productDetails = $this->call('GET', '/admin/products/' . $parentVarID . '.json');
                $parentVarID = $productDetails['variants'][0]['id'];
            } else {
                if (!is_numeric($parentVarID)) {
                    $parentVarID = $variantID;
                }
            }
        } else {
            $parentVarID = $variantID;
        }
        return $parentVarID;
    }

    public function shopifyParentProductID($variantID)
    {
        $variant = $this->call('GET', '/admin/variants/' . $variantID . '.json');
        $thisProductData['pid'] = $variant['product_id'];
        $thisProductData['vid'] = $this->getParentVariantID($variantID);
        return json_encode($thisProductData);
    }

    /**
     * GET: all variant list
     *
     * @author inkXE
     * @date 23 July 2020
     * @param $productId requested product ID
     * @return arrau list of variants
     */
    public function getStoreVariants($productId)
    {
        $variants = [];
        $product = $this->call('GET', '/admin/products/' . $productId . '.json?fields=variants');
        if (!empty($product)) {
            $colorPos = $this->getAttributePostion($this->colorAttrName, $productId);
            $sizePos = $this->getAttributePostion($this->sizeAttrName, $productId);
            $catlogProductRel = new CatalogProductRel();
            $productStoredType = $catlogProductRel->where(['product_id' => $productId])->pluck('product_stored_type')->toArray();

            //if SHOPIFY_VARIANTS_PLUGIN is enabled then fetch variant/options from db 
            if ($productStoredType[0] != 'store') {
                $variantRelObj = new ProductVariantRel();
                $variantIds = $variantRelObj->where(['product_id' => $productId])->pluck('variant_id')->toArray();
                $variantObj = new ProductVariants();
                $variants = [];
                foreach ($variantIds as $varId) {
                    $variant = $variantObj->where(['xe_id' => $varId])->get()->toArray();
                    $thisVariants[] = $variant[0];
                }
                foreach ($thisVariants as $variant) {
                    $optionArr = explode('/', $variant['title']);
                    $thisVariant = [
                        'id' => $variant['xe_id'],
                        'title' => $variant['title'],
                        'price' => $variant['price'],
                        'sku' => $variant['sku'],
                        'inventory' => $variant['inventory'],
                        'color' => $optionArr[0],
                        'size' => $optionArr[1]
                    ];
                    $variants[] = $thisVariant;
                }
            } else {
            foreach ($product['variants'] as $variant) {
                if ($variant['title'] != "Default Title") {
                    $thisVariant = [
                        'id' => $variant['id'],
                        'title' => $variant['title'],
                        'price' => $variant['price'],
                        'sku' => $variant['sku'],
                        'inventory' => $variant['inventory_quantity'],
                        'color' => $variant['option' . $colorPos],
                        'size' => $variant['option' . $sizePos]
                    ];
                    $variants[] = $thisVariant;
                    }
                }
            }
        }
        return $variants;
    }

    public function getTierDiscounts($productId)
    {
        $tierDiscounts = array();
        $product = $this->call('GET', '/admin/products/' . $productId . '.json?key=title,variants');
        $metaFields = $this->call('GET', '/admin/products/' . $productId . '/metafields.json?key=tier_content');
        if (!empty($metaFields)) {
            $tierDiscounts = json_decode($metaFields[0]['value'], true);
        }
        $tierDiscounts['name'] = $product['title'];
        $tierDiscounts['price'] = $product['variants'][0]['price'];
        return $tierDiscounts;
    }

    public function saveProductTierPrice($tierData)
    {
        $saveStatus = false;
        $metaContent = json_encode($tierData);
        $metaFields = $this->call('GET', '/admin/products/' . $tierData['productID'] . '/metafields.json?key=tier_content');
        if (!empty($metaFields)) {
            foreach ($metaFields as $field) {
                $this->call('DELETE', '/admin/products/' . $tierData['productID'] . '/metafields/' . $field['id'] . '.json');
            }
        }
        $metaData = array(
            "metafield" => array(
                "namespace" => "imprint_data",
                "key" => "tier_content",
                "value" => addslashes($metaContent),
                "type" => "json"
            ),
        );
        $metaSave = $this->call('POST', '/admin/products/' . $tierData['productID'] . '/metafields.json', $metaData);
        if (is_array($metaSave) && is_numeric($metaSave['id'])) {
            $saveStatus = true;
        }
        return $saveStatus;
    }

    public function addCatalogProductToStore($priceInfo, $product, $price, $catalog_price, $storeCategories)
    {

        // get product images
        if ($product['old_product_id'] > 0) {
            $this->call('DELETE', '/admin/products/' . $product['old_product_id'] . '.json');
        }
        $imgArr = $optionArr = $variantArr = $newProductIds = array();
        foreach ($product['variations'] as $variant) {
            $key = 1;
            foreach ($variant['image_path'] as $img) {
                if (!empty($img) && !in_array(array("src" => $img, "position" => $key), $imgArr)) {
                    array_push($imgArr, array("src" => $img, "position" => $key));
                }
                $key++;
            }
            $thisVarPrice = $variant['piece_price'];

            if (!empty($priceInfo['is_margin'])) {
                if ($priceInfo['is_margin'] == 2) {
                    $thisVarPrice = $variant['piece_price'] + $priceInfo['product_margin'];
                } else {
                    $percentagePrice = $variant['piece_price'] * ($priceInfo['product_margin'] / 100);
                    $thisVarPrice = $variant['piece_price'] + $percentagePrice;
                }
            } else {
                $thisVarPrice = $price;
            }
            $thisVariant = array(
                "sku" => $variant['sku'],
                "barcode" => isset($variant['gtin']) ? $variant['gtin']  : '',
                "price" => number_format((float)$thisVarPrice, 2, '.', ''),
                "inventory_management" => "shopify",
                "inventory_policy" => "deny"
            );
            $optionKey = 1;
            foreach ($variant['attributes'] as $option) {
                $thisVariant["option" . $optionKey] = $option;
                $optionKey++;
            }
            $variantArr[] = $thisVariant;
        }

        if (!empty($product['attributes'])) {
            foreach ($product['attributes'] as $attrKey => $attribute) {
                // if(strtolower($attribute['name']) == 'color') {
                //     $color = $attribute['options'];
                // }                
                $thisAttr = [];
                if (strtolower($attribute['name']) != 'xe_is_designer') {
                    $thisAttr['name'] = $attribute['name'];
                    $thisAttr['position'] = ($attrKey + 1);
                    $thisAttr['values'] = $attribute['options'];
                    $optionArr[] = $thisAttr;
                }
            }
        }

        // if ($params['product_id'] == ''){
        $productArray = array(
            "product" => array(
                "title" => $this->shopify_body(addslashes($product['name'])),
                "body_html" => $this->shopify_body(addslashes($product['description'])),
                "published" => true,
                "tags" => "catalogue",
                "image" => array("src" => $product['images']['src'], "position" => 1),
            ),
        );


        if (!empty($optionArr) && !empty($variantArr)) {
            $productArray['product']['options'] = $optionArr;
            $productArray['product']['variants'] = $variantArr;
        }

        $metaDataInfo = array(
            "metafield" => array(
                "namespace" => "imprint_catalogue",
                "key" => "sanmar",
                "value" => "api",
                "value_type" => "string",
            ),
        );
        if (SHOPIFY_VARIANTS_PLUGIN == 1) {
            $newProductIds = $this->shopifyVariantManage($productArray,$product,$imgArr);
            return $newProductIds;
        }
        $variantChunks = array_chunk($variantArr, 99);
        foreach ($variantChunks as $variantSplitData) {
            $productArray['product']['variants'] = SHOPIFY_VARIANTS_PLUGIN == 1?array_unique($variantSplitData):$variantSplitData;
            $newProduct = $this->call('POST', '/admin/products.json', $productArray);
            foreach ($imgArr as $image) {
                $newImg['image'] = array("src" => $image['src']);
                $this->call('POST', '/admin/products/' . $newProduct['id'] . '/images.json', $newImg);
            }
            if (!empty($metaDataInfo)) {
                $this->call('POST', '/admin/products/' . $newProduct['id'] . '/metafields.json', $metaDataInfo);
            }

            if (!empty($storeCategories)) {
                foreach ($storeCategories as $key => $category) {
                    $categoryArr = array(
                        "collect" => array(
                            "product_id" => $newProduct['id'],
                            "collection_id" => $category,
                        ),
                    );
                    $this->call('POST', '/admin/collects.json', $categoryArr);
                }
            }
            if ($newProduct['variants'][0]['inventory_management'] == 'shopify') {
                $newProductVariants = $newProduct['variants'];
                $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
                foreach ($inventoryItems as $inventory) {
                    $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventory);
                    foreach ($invLevels as $level) {
                        $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
                        if (!$locationData['legacy']) {
                            $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" => $inventory, "available" => 100);
                            $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                        }
                    }
                }
            }
            // assign product to "Show in designer" collection if is customized is checked.
            $customArr = array(
                "collect" => array(
                    "product_id" => $newProduct['id'],
                    "collection_id" => $this->getShowInDesignerId(),
                ),
            );
            $this->call('POST', '/admin/collects.json', $customArr);
            // get images of new products
            $newProductImage = array();
            if (!empty($newProduct['id'])) {
                $thisProdImages = $this->call('GET', '/admin/products/' . $newProduct['id'] . '/images.json');
            }
            foreach ($thisProdImages as $img) {
                array_push($newProductImage, array("id" => $img['id'], "src" => $img['src']));
            }
            //assign image to all variants of the product
            $colorPos = $this->getAttributePostion($this->colorAttrName, $newProduct['id']);
            foreach ($newProduct['variants'] as $variant) {
                $imgID = 0;
                foreach ($newProductImage as $image) {
                    if (strpos(strtolower($image['src']), strtolower($this->removeSPLChars($variant['option' . $colorPos]))) !== false && strpos(strtolower($image['src']), '_front') !== false) {
                        $imgID = $image['id'];
                        break;
                    }
                }
                if ($imgID > 0) {
                    $variantArr = array(
                        "variant" => array(
                            "id" => $variant['id'],
                            "image_id" => $imgID,
                        ),
                    );
                    $this->call('PUT', '/admin/variants/' . $variant['id'] . '.json', $variantArr);
                }
            }
            $newProductIds[] = $newProduct['id'];
        }

        return $newProductIds;
    }

    private function removeSPLChars($string, $removeSpace=true)
    {
        $string = preg_replace('/[^A-Za-z0-9 \-]/', '', $string); // Removes special chars.
        $string = strtolower($string);
        $string = $removeSpace === true ? str_replace(' ', '', $string):str_replace(' ', '_', $string); // Replaces all spaces.
        return preg_replace('/-+/', '', $string); // Replaces multiple hyphens.
    }

    public function addCollection($catName)
    {
        $collectionArr['custom_collection'] = array(
            "title" => $catName
        );
        $addCustomCol = $this->call('POST', '/admin/custom_collections.json', $collectionArr);
        if (!empty($addCustomCol['id'])) {
            $storeResponse = [
                'status' => 1,
                'catatory_id' => $addCustomCol['id'],
                'message' => message('Catagories', 'saved'),
            ];
        } else {
            $storeResponse = [
                'status' => 0,
                'message' => 'Failed to create collection',
            ];
        }

        return $storeResponse;
    }

    public function GetProductMeta($productID)
    {
        $preDecoContent = ['custom_design_id' => 0, 'is_redesign' => 0];
        $productMetas = $this->call('GET', '/admin/products/' . $productID . '/metafields.json');
        foreach ($productMetas as $meta) {
            if ($meta['namespace'] == 'Pre-Deco' && $meta['key'] == 'imprint_design_id') {
                $preDecoContent['custom_design_id'] = $meta['value'];
                break;
            }
        }
        $customCats = $this->call('GET', '/admin/custom_collections.json?product_id=' . $productID);
        $smartCats = $this->call('GET', '/admin/smart_collections.json?product_id=' . $productID);
        $allCats = array_merge($customCats, $smartCats);
        $categories = array_column($allCats, 'handle');
        $preDecoContent['is_redesign'] = in_array('show-in-designer', $categories) ? 1 : 0;
        return $preDecoContent;
    }

    public function getShopifyProductInfo($productID)
    {
        return $this->call('GET', '/admin/products/' . $productID . '.json', array(), "", true);
    }

    /**
     * Update selected variant id inventory to out of stock
     *
     * @param $storeProductId  Product Id, $selectedVarIds Variant Id
     *
     * @author divya@imprintnext.com
     * @date  27 April 2022
     * @return true/false
     */
    public function variantStockUpdate($stockUpdateJson)
    {
        // group by product_id
        $newStockByProducts = array_reduce($stockUpdateJson, function ($accumulator, $element) {
            $accumulator[$element['productId']][] = $element;
            return $accumulator;
        }, []);
        $stockeUpdated = 0; // to mark stock is updated or not
        foreach ($newStockByProducts as $productId => $newStock) {
            $allVariantData = [];
            $newStockVariants = array_column($newStock, 'variantId');
            $productDetails = $this->call('GET', '/admin/products/' . $productId . '.json');
            $variants = $productDetails['variants'];
            $catlogProductRel = new CatalogProductRel();
            $productStoredType = $catlogProductRel->where(['product_id' => $productId])->pluck('product_stored_type')->toArray();

            //if SHOPIFY_VARIANTS_PLUGIN is enabled then update stock and price in Imprint db 
            if ($productStoredType[0] != 'store') {
                $variantRelObj = new ProductVariantRel();
                $variantIds = $variantRelObj->where(['product_id' => $productId])->pluck('variant_id')->toArray();
                $variantObj = new ProductVariants();
                foreach ($variantIds as $varId) {
                    $key = array_search($varId, $newStockVariants);
                    if ($key !== false && isset($newStock[$key]['price']) && isset($newStock[$key]['stock'])) {

                        $variantObj->where('xe_id', $varId)
                            ->update(['price' => $newStock[$key]['price'], 'inventory' => $newStock[$key]['stock']]);
                        $stockeUpdated = 1;
                    }
                }
            } else {
            foreach ($variants as $variant) {
                $variantUpdateData = [
                    'id' => $variant['id'],
                    'title' => $variant['title'],
                    'sku' => $variant['sku']
                ];
                // find the variant - then update the prices (all variant updated at a time)
                $key = array_search($variant['id'], $newStockVariants);
                if ($key !== false) {
                    if (isset($newStock[$key]['price'])) {
                        $variantUpdateData['price'] = $newStock[$key]['price'];
                    }
                    if (isset($newStock[$key]['stock'])) {
                        // update the inventory
                        $inventoryItemid = $variant['inventory_item_id'];
                        $inventoryLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventoryItemid);
                        foreach ($inventoryLevels as $invLevels) {
                            $locationData = $this->call('GET', '/admin/locations/' . $invLevels['location_id'] . '.json');
                            if (!$locationData['legacy']) {
                                $newInventory = [
                                    'location_id' => $invLevels['location_id'],
                                    'inventory_item_id' => $inventoryItemid,
                                    'available' => $newStock[$key]['stock']
                                ];
                                $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                                $stockeUpdated = 1;
                            }
                        }
                    }
                }
                // keep all variants and update with product at a time(price update)
                $allVariantData[] = $variantUpdateData;
            }

            // update product variant price
            // we are not keeping track of each variant's price update status
            if ($allVariantData) {
                $this->call('PUT', '/admin/products/' . $productId . '.json', [
                    'product' => [
                        'product' => $productId, 'variants' => $allVariantData
                    ]
                ]);
                }
            }
        }
        return [
            'status' => $stockeUpdated
        ];
    }

    /**
     * get all location for product
     *
     * @param $storeProductId  Product Id, $selectedVarIds Variant Id
     *
     * @author sonali@imprintnext.com
     * @date  19 july 2022
     * @return true/false 
     */
    public function getloation()
    {
        return $this->call('GET', '/admin/locations' . '.json');
    }

    /**
     * delete/archieve all abondoned cart Products from Shopify store
     *
     * @param $status,$days
     *
     * @author pansy@imprintnext.com
     * @date  06 Feb 2024
     * @return
     */
    public function deleteDuplicateProducts($params)
    {
        $response = [];
        $status = $params['status'];
        $days = $params['days'];
        $createdAfter = strtotime('-' . $days . 'days');
        $productCreated = date('Y-m-d\TH:i:sP', $createdAfter);
        $productParams['limit'] = 250;
        $productParams['created_at_max'] = $productCreated;
        $productParams['collection_id'] = $this->getCustomizedCollectionId();
        $duplicateProducts = $this->call('GET', '/admin/products.json', $productParams, "products");
        if (!empty($duplicateProducts)) {
            $removedProductArr = array(
                'action_performed_date' => date('Y-m-d'),
                'action_performed' => '',
                'product_id' => []
            );
            $temp = 0;
            $removedProdsJsonPath = ASSETS_PATH_W . 'products/duplicate_products.json';
            if ($status == 'delete') {
                foreach ($duplicateProducts as $eachDuplicateProduct) {
                    $deleteStatus = $this->call('DELETE', '/admin/products/' . $eachDuplicateProduct['id'] . '.json');
                    if ($deleteStatus['method'] == "DELETE") {
                        $temp += 1;
                    } else {
                        $removedProductArr['action_performed'] = $status;
                        $removedProductArr['product_id'][] += $eachDuplicateProduct['id'];
                        file_put_contents($removedProdsJsonPath, json_encode($removedProductArr));
                    }
                }
                $response = $temp > 0 ? array('status' => 0, 'message' => "Some Products could'nt be deleted") : $response = array('status' => 1, 'message' => "products deleted successfully");
            } elseif ($status == 'archive') {
                $update['product'] = array('status' => 'archived');
                foreach ($duplicateProducts as $eachDuplicateProduct) {
                    $archieveStatus = $this->call('PUT', '/admin/products/' . $eachDuplicateProduct['id'] . '.json', $update);
                    if ($archieveStatus['method'] == "PUT") {
                        $temp += 1;
                    } else {
                        $removedProductArr['action_performed'] = $status;
                        $removedProductArr['product_id'][] += $eachDuplicateProduct['id'];
                        file_put_contents($removedProdsJsonPath, json_encode($removedProductArr));
                    }
                }
                $response = $temp > 0 ? array('status' => 0, 'message' => "Some Products could'nt be archived") : $response = array('status' => 1, 'message' => "products archived successfully");
            }
        } else {
            $response = array('status' => 1, 'message' => "No duplicate products found");
        }
        return $response;
    }

    /** ************************* Product Module End **************************** **/

    /** ************************* Customer Module Start **************************** **/

    /**
     * Get list of Customer
     *
     * @author     debashrib@riaxe.com
     * @date       17 dec 2019
     * @param  API filter params
     * @response   Array of all customer list
     */
    public function getAllCustomers($callParams)
    {
        $users = array();
        $params['limit'] = $callParams['limit'];
        $page = $callParams['page'];
        $params['order'] = 'created_at%2Bdesc';
        $customersCount = $this->call('GET', '/admin/customers/count.json');
        if (!empty($callParams['from_date']) && !empty($callParams['to_date'])) {
            $params['created_at_min'] = date('Y-m-d h:i:s', strtotime($callParams['from_date']));
            $params['created_at_max'] = date('Y-m-d h:i:s', strtotime($callParams['to_date']));
        }
        if ($callParams['searchString'] != '') {
            $customers = $this->call('GET', '/admin/customers/search.json?query=' . $callParams['searchString']);
        } else {
            if ($page == 1 || empty($page)) {
                $customers = $this->call('GET', '/admin/customers.json', $params, "customers");
            } elseif ($page == 0) {
                $cursorURL = substr($_SESSION['customers']['previous'], strpos($_SESSION['customers']['previous'], "/admin"));
                $customers = $this->call('GET', $cursorURL, array(), "customers");
            } else {
                $cursorURL = substr($_SESSION['customers']['next'], strpos($_SESSION['customers']['next'], "/admin"));
                $customers = $this->call('GET', $cursorURL, array(), "customers");
            }
        }
        if (!empty($callParams['customer_no_order']) && $callParams['customer_no_order'] != 'false') {
            foreach ($customers as $cust) {
                if ($cust['orders_count'] < 1) {
                    $user = array();
                    $user['id'] = $cust['id'];
                    $user['first_name'] = $cust['first_name'];
                    $user['last_name'] = $cust['last_name'];
                    $user['email'] = $cust['email'];
                    $user['total_orders'] = $cust['orders_count'];
                    $user['last_order_id'] = str_replace('#', '', $cust['last_order_id']);
                    $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['created_at']));
                    $users['customer_list'][] = $user;
                }
            }
        } elseif (!empty($callParams['customer_no_order']) && $callParams['customer_no_order'] == 'false') {
            foreach ($customers as $cust) {
                if ($cust['orders_count'] > 0) {
                    $user = array();
                    $user['id'] = $cust['id'];
                    $user['first_name'] = $cust['first_name'];
                    $user['last_name'] = $cust['last_name'];
                    $user['email'] = $cust['email'];
                    $user['total_orders'] = $cust['orders_count'];
                    $user['last_order_id'] = str_replace('#', '', $cust['last_order_id']);
                    $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['created_at']));
                    $users['customer_list'][] = $user;
                }
            }
        } else {
            foreach ($customers as $cust) {
                $user = array();
                $user['id'] = $cust['id'];
                $user['first_name'] = $cust['first_name'];
                $user['last_name'] = $cust['last_name'];
                $user['email'] = $cust['email'];
                $user['total_orders'] = $cust['orders_count'];
                $user['last_order_id'] = str_replace('#', '', $cust['last_order_id']);
                $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['created_at']));
                $users['customer_list'][] = $user;
            }
        }
        $users['total_user'] = $customersCount;
        return $users;
    }

    /**
     * Get details of Customer
     *
     * @author     debashrib@riaxe.com
     * @date       17 dec 2019
     * @param  $customerID selected cutomer ID
     * @response   Array single customer details
     */
    public function getCustomerDetails($customerID, $getOrders = 0, $skipFields = [])
    {
        $customer = [];
        $billingAddress = [];
        $orderDetails = [];
        $customer = $this->call('GET', '/admin/customers/' . $customerID . '.json');
        
        // Getting Billing Address of perticular customer
        if (!empty($customer['default_address']['address1'])) {
            $billingAddress = [
                'first_name' => $customer['default_address']['first_name'],
                'last_name' => $customer['default_address']['last_name'],
                'email' => $customer['email'],
                'phone' => $customer['default_address']['phone'],
                'address_1' => $customer['default_address']['address1'],
                'address_2' => $customer['default_address']['address2'],
                'city' => $customer['default_address']['city'],
                'state' => $customer['default_address']['province'],
                'postcode' => $customer['default_address']['zip'],
                'country' => $customer['default_address']['country_code'],
            ];
        }

        // Getting all order details of perticular customer
        $customerOrder = $this->getOrderDetailsByCustomer($customerID);
        if (!empty($customerOrder)) {
            foreach ($customerOrder as $orderData) {
                $order = array();
                $order['id'] = $orderData['id'];
                $order['number'] = $orderData['order_number'];
                $order['quantity'] = array_sum(array_column($orderData['line_items'], 'quantity'));
                $order['currency'] = $orderData['currency'];
                $order['total_amount'] = (float)round($orderData['total_price'], 2);
                $order['created_date'] = date('Y-m-d h:i:s', strtotime($orderData['created_at']));
                if ($getOrders > 0 && !empty($orderData['line_items'])) {
                    // Getting line items of perticular order
                    foreach ($orderData['line_items'] as $lineItems) {
                        $refID = 0;
                        foreach ($lineItems['properties'] as $value) {
                            if ($value['name'] == "_refid") {
                                $refID = $value['value'];
                                break;
                            } 
                        }
                        if ($refID == 0 && $lineItems['vendor'] == "imprintNext") {
                            // get prod data $lineItems['product_id']
                            $productData = $this->call('GET', '/admin/products/' . $lineItems['product_id'] . '.json');
                            $tags = $productData['tags'];
                            $tags = explode('IMP_REF_ID_', $tags);
                            $refID = $tags[1];
                        }
                        if ($refID > 0 && !empty($lineItems['variant_id'])) {
                            $OriginalVarID = $this->getParentVariantID($lineItems['variant_id']);
                            $realProductData = $this->shopifyParentProductID($OriginalVarID);
                            $variant = $this->call('GET', '/admin/variants/' . $OriginalVarID . '.json');
                            $realProdID =  json_decode($realProductData, true);
                            $colorPos = $this->getAttributePostion($this->colorAttrName, $realProdID['pid']);
                            $sizePos = $this->getAttributePostion($this->sizeAttrName, $realProdID['pid']);
                            $lineItems = [
                                'product_id' => $realProdID['pid'],
                                'variant_id' => $OriginalVarID,
                                'name' => $lineItems['name'],
                                'price' => (float)round($lineItems['price'], 2),
                                'quantity' => $lineItems['quantity'],
                                'total' => (float)round($lineItems['price'], 2),
                                'sku' => $lineItems['sku'],
                                'custom_design_id' => (isset($refID) && $refID != '') ? $refID : '',
                                'size' => isset($variant['option' . $sizePos]) ? $variant['option' . $sizePos] : '',
                                'color' => isset($variant['option' . $colorPos]) ? $variant['option' . $colorPos] : '',
                                'images' => $this->getStoreProductImages($refID, $lineItems['variant_id']),
                            ];
                            $order['lineItems'][] = $lineItems;
                        }
                    }
                }
                if (!empty($order['lineItems'])) {
                    $orderDetails[] = $order;
                }
            }
        }

        $result = [
            "id" => $customer['id'],
            "first_name" => $customer['first_name'],
            "last_name" => $customer['last_name'],
            "email" => $customer['email'],
            "profile_pic" => '',
            "phone" => ($customer['phone'] != '') ? $customer['phone'] : '',
            "total_orders" => $customer['orders_count'],
            "total_order_amount" => (float)round($customer['total_spent'], 2),
            "average_order_amount" => (float)(
                ($customer['orders_count'] > 0)
                ? strval(round(($customer['total_spent'] / $customer['orders_count']), 2)) : '0:00'
            ),
            "date_created" => date('Y-m-d h:i:s', strtotime($customer['created_at'])),
            "billing_address" => $billingAddress,
            "last_order" => $orderDetails[0]['created_date'],
            "orders" => $orderDetails
        ];
        if (!in_array('shipping_address', $skipFields)) {
            $customerAddress = $this->call('GET', '/admin/customers/' . $customerID . '/addresses.json');
            if (!empty($customerAddress)) {
                foreach ($customerAddress as $address) {
                    if (!empty($address['address1'])) {
                        $shippingAddress[] = [
                            'id' => $address['id'],
                            'first_name' => $address['first_name'],
                            'last_name' => $address['last_name'],
                            'email' => $customer['email'],
                            'phone' => $address['phone'],
                            'address_1' => $address['address1'],
                            'address_2' => $address['address2'],
                            'city' => $address['city'],
                            'state_name' => $address['province'],
                            'state' => $address['province_code'],
                            'postcode' => $address['zip'],
                            'country_name' => $address['country_code'],
                            'country' => $address['country_code'],
                        ];
                    }
                }
            } else {
                $shippingAddress = array($billingAddress);
            }
            $result['shipping_address'] = $shippingAddress;
        }

        return $result;
    }

    public function storeCustomerCount()
    {
        return $this->call('GET', '/admin/customers/count.json');
    }

    public function getShopCountries()
    {
        $countries = $this->call('GET', '/admin/countries.json');
        $allCountries = array();
        foreach ($countries as $country) {
            if ($country['code'] != "*") {
                $allCountries[] = array('countries_code' => $country['code'], 'countries_name' => $country['name']);
            }
        }
        return $allCountries;
    }

    public function getProvibce($countryCode)
    {
        $countries = $this->call('GET', '/admin/countries.json');
        $states = array();
        foreach ($countries as $country) {
            if ($country['code'] == $countryCode) {
                foreach ($country['provinces'] as $key => $state) {
                    $states[$key]['state_code'] = $state['code'];
                    $states[$key]['state_name'] = $state['name'];
                }
            }
        }
        return $states;
    }

    public function deleteShopCustomer($customerID)
    {
        $response = array();
        foreach ($customerID as $cusID) {
            $deleteStatus = $this->call('DELETE', '/admin/customers/' . $cusID . '.json');
            if ($deleteStatus['method'] == "DELETE") {
                $response = array('status' => 0, 'message' => "Customer can not be deleted due to their active orders.");
            } else {
                $response = array('status' => 1, 'message' => "customer deleted");
            }
        }
        return $response;
    }

    public function newShopCustomer($custData)
    {
        $customerArr = array();
        $searchCustomer = $this->call('GET', '/admin/customers/search.json?query=email:"' . $custData['user_email'] . '"&fields=id,email');
        $searchnum = $this->call('GET', '/admin/customers/search.json?query=phone:"' . $custData['billing_phone'] . '"&fields=id,email');

        if (empty($searchCustomer) && empty($searchnum)) {
            if (!empty($custData)) {
                $customerInfo['first_name'] = $custData['first_name'];
                $customerInfo['last_name'] = $custData['last_name'];
                $customerInfo['password'] = $custData['user_password'];
                $customerInfo['password_confirmation'] = $custData['user_password'];
                $customerInfo['send_email_welcome'] = false;
                $customerInfo['email'] = $custData['user_email'];
                $customerInfo['phone'] = $custData['billing_phone'];
                $customerInfo['verified_email'] = true;
                $customerInfo['addresses'] = array(array(
                    'address1' => $custData['billing_address_1'],
                    'address2' => $custData['billing_address_2'],
                    'city' => $custData['billing_city'],
                    'province' => $custData['billing_state_code'],
                    'phone' => $custData['billing_phone'],
                    'zip' => $custData['billing_postcode'],
                    'last_name' => $custData['last_name'],
                    'first_name' => $custData['first_name'],
                    'country' => $custData['billing_country_code']
                ), array(
                    'address1' => $custData['shipping_address_1'],
                    'address2' => $custData['shipping_address_2'],
                    'city' => $custData['shipping_city'],
                    'province' => $custData['shipping_state_code'],
                    'phone' => $custData['shipping_phone'],
                    'zip' => $custData['shipping_postcode'],
                    'last_name' => $custData['last_name'],
                    'first_name' => $custData['first_name'],
                    'country' => $custData['shipping_country_code']
                ));
            }
            $customerArr['customer'] = $customerInfo;
            return $this->call('POST', '/admin/customers.json', $customerArr);
        } else {
            return array();
        }
    }

    public function changeDefaultAddress($newAddress)
    {
        $status = array();
        if (!empty($newAddress)) {
            $newAdd['address']['address1'] = $newAddress['address_1'];
            $newAdd['address']['address2'] = $newAddress['address_2'];
            $newAdd['address']['city'] = $newAddress['city'];
            $newAdd['address']['company'] = $newAddress['company'];
            $newAdd['address']['first_name'] = $newAddress['first_name'];
            $newAdd['address']['last_name'] = $newAddress['last_name'];
            $newAdd['address']['phone'] = $newAddress['mobile_no'];
            $newAdd['address']['zip'] = $newAddress['post_code'];
            $newAdd['address']['province_code'] = $newAddress['state'];
            $newAdd['address']['country_code'] = $newAddress['country'];
            $status = $this->call('POST', '/admin/customers/' . $newAddress['user_id'] . '/addresses.json', $newAdd);
        }
        return $status;
    }

    public function customerShortData($customerId, $isAddress = false)
    {
        $customer = [];
        $billingAddress = [];
        $customer = $this->call('GET', '/admin/customers/' . $customerId . '.json');
        $customerAddress = $this->call('GET', '/admin/customers/' . $customerId . '/addresses.json');
        $customerDetails = [
            "id" => $customer['id'],
            "email" => $customer['email'],
            "name" => $customer['first_name'] . ' ' . $customer['last_name'],
            "phone" => ($customer['phone'] != '') ? $customer['phone'] : '',
        ];
        // Getting Billing Address of perticular customer
        if ($isAddress && !empty($customer['addresses']) && !empty($customer['default_address']['address1'])) {
            $billingAddress = [
                'first_name' => $customer['default_address']['first_name'],
                'last_name' => $customer['default_address']['last_name'],
                'email' => $customer['email'],
                'phone' => $customer['default_address']['phone'],
                'address_1' => $customer['default_address']['address1'],
                'address_2' => $customer['default_address']['address2'],
                'city' => $customer['default_address']['city'],
                'state' => $customer['default_address']['province'],
                'postcode' => $customer['default_address']['zip'],
                'country' => $customer['default_address']['country_code'],
            ];
            if (!empty($customerAddress)) {
                foreach ($customerAddress as $address) {
                    if (!empty($address['address1'])) {
                        $shippingAddress[] = [
                            'id' => $address['id'],
                            'first_name' => $address['first_name'],
                            'last_name' => $address['last_name'],
                            'email' => $customer['email'],
                            'phone' => $address['phone'],
                            'address_1' => $address['address1'],
                            'address_2' => $address['address2'],
                            'city' => $address['city'],
                            'state_name' => $address['province_'],
                            'state' => $address['province'],
                            'postcode' => $address['zip'],
                            'country_name' => $address['country_code'],
                            'country' => $address['country_code'],
                        ];
                    }
                }
            } else {
                $shippingAddress = array($billingAddress);
            }
            $customerDetails['billing_address'] = $billingAddress;
            $customerDetails['shipping_address'] = $shippingAddress;
        }
        return array('customer' => $customerDetails);
    }
    public function updateCustomerAddressInShop($allPostPutVars, $id)
    {
        if (!empty($allPostPutVars)) {
            $address['address'] = array(
                "address1" => $allPostPutVars['address_1'],
                "address2" => $allPostPutVars['address_2'],
                "city" => $allPostPutVars['city'],
                "company" => $allPostPutVars['company'],
                "first_name" => $allPostPutVars['first_name'],
                "last_name" => $allPostPutVars['last_name'],
                "phone" => $allPostPutVars['mobile_no'],
                "zip" => $allPostPutVars['post_code'],
                "name" => $allPostPutVars['first_name'] . " " . $allPostPutVars['last_name'],
                "province_code" => $allPostPutVars['state'],
                "country_code" => $allPostPutVars['country']
            );
            $hideProd = $this->call('PUT', '/admin/customers/' . $allPostPutVars['user_id'] . '/addresses/' . $id . '.json', $address);
        }
        if ($hideProd['method'] !== "PUT") {
            return array("status" => 1, "message" => "address updated");
        } else {
            return array();
        }
    }

    /**
     * get customer saved search as groups 
     *
     * @author     debashisd@riaxe.com
     * @date       3 June 2021
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function getCustomerSavedSearches($filters)
    {
        $CustomerGroupCount = $this->call('GET', '/admin/customer_saved_searches/count.json');

        if (!empty($filters['savedSearchID'])) {
            $group = $this->call('GET', '/admin/customer_saved_searches/' . $filters['savedSearchID'] . '.json');
            $customerGroups = array("id_group" => $group['id'], 'name' => $group['name']);
        } else {
            $limit = !empty($filters['perpage']) ? $filters['perpage'] : 250;
            $page = $filters['page'];
            $params = array("limit" => $limit);
            if (!empty($filters['name'])) {
                $params['name'] = $filters['name'];
            }
            if ($page == 1 || empty($page)) {
                $savedSearches = $this->call('GET', '/admin/customer_saved_searches.json', $params, "customer_saved_searches");
            } elseif ($page == 0) {
                $cursorURL = substr($_SESSION['customer_saved_searches']['previous'], strpos($_SESSION['customer_saved_searches']['previous'], "/admin"));
                $savedSearches = $this->call('GET', $cursorURL, array(), "customer_saved_searches");
            } else {
                $cursorURL = substr($_SESSION['customer_saved_searches']['next'], strpos($_SESSION['customer_saved_searches']['next'], "/admin"));
                $savedSearches = $this->call('GET', $cursorURL, array(), "customer_saved_searches");
            }
            $customerGroups = array();
            if ($savedSearches["method"] != "GET") {
                foreach ($savedSearches as $group) {
                    $thisGroup = array("id_group" => $group['id'], 'name' => $group['name']);
                    $customerGroups[] = $thisGroup;
                }
            }
        }
        return  array("data" => $customerGroups, "total_records" => $CustomerGroupCount);
    }

    public function getStoreCustomerByGroupId($savedSearchID, $returnCount = false)
    {
        $getCustomers = $this->call('GET', '/admin/customer_saved_searches/' . $savedSearchID . '/customers.json?limit=250');
        if ($returnCount) {
            return count($getCustomers);
        } else {
            $thisGroupCustomers = array();
            if (!empty($getCustomers)) {
                foreach ($getCustomers as $customer) {
                    $thisGroupCustomers[] = array('id' => $customer['id'], 'name' => $customer['first_name'] . " " . $customer['last_name'], 'email' => $customer['email']);
                }
            }
            return $thisGroupCustomers;
        }
    }

    public function updateCustomerSavedSearch($updateData)
    {
        $returnData = array('status' => 0, 'message' => 'Shopify terminated the request');
        $customerGroupID = $updateData['id'];
        $groupName = $updateData['name'];
        $tagFilter = "tag:" . $groupName;
        $groupData = $this->call('GET', '/admin/customer_saved_searches/' . $customerGroupID . '.json');
        $thisGroupQueries = explode(" ", $groupData['query']);
        if (!empty($thisGroupQueries)) {
            foreach ($thisGroupQueries as $filter) {
                if (strpos($filter, 'tag:') === false) {
                    $tagFilter .= " " . $filter;
                }
            }
        }
        if (!empty($tagFilter)) {
            $tagFilterForGroup = array('customer_saved_search' => array(
                'id' => $customerGroupID,
                'name' => $groupName,
                'query' => $tagFilter
            ));
        }
        $updateGroup = $this->call('PUT', '/admin/customer_saved_searches/' . $customerGroupID . '.json', $tagFilterForGroup);
        // clean existing customers assignment to groups
        $allCustomers =  $this->call('GET', '/admin/customers.json');
        foreach ($allCustomers as $cust) {
            if (strpos($cust['tags'], $groupName) !== false) {
                $thisUpdateData = array('customer' => array(
                    'id' => $cust['id'],
                    'tags' => str_replace($groupName, '', str_replace("imp_grp_" . $customerGroupID, '', $cust['tags']))
                ));
                $this->call('PUT', '/admin/customers/' . $cust['id'] . '.json', $thisUpdateData);
            }
        }
        if (!isset($updateGroup['method'])) {
            $this->updateTagsToCustomers($updateData);
            $returnData = array('status' => 1, 'message' => 'CustomerGroup is updated in Shopify store');
        } else {
            print_r($updateGroup);
            exit();
        }
        return $returnData;
    }

    private function updateTagsToCustomers($updateData)
    {
        $customers = $updateData['customer_id'];
        $customerGroupID = $updateData['id'];
        $groupName = $updateData['name'];
        foreach ($customers as $customerID) {
            $this->call('GET', '/admin/customers/' . $customerID . '.json?fields=tags');
            $newTags = $groupName . ", imp_grp_" . $customerGroupID;
            $thiscustTags = $newTags;
            $updateData = array('customer' => array(
                'id' => $customerID,
                'tags' => $thiscustTags
            ));
            $this->call('PUT', '/admin/customers/' . $customerID . '.json', $updateData);
        }
    }

    public function createcustomerSavedSearch($name, $customers)
    {
        $newCustGrpID = "";
        $tagFilter = "tag:" . $name;
        $groupPostData = array('customer_saved_search' => array(
            'name' => $name,
            'query' => $tagFilter
        ));
        $addGroup = $this->call('POST', '/admin/customer_saved_searches.json', $groupPostData);
        if (!isset($addGroup['method'])) {
            $newCustGrpID = $addGroup['id'];
        }
        $updateCustomers = array('customer_id' => $customers, 'id' => $newCustGrpID, 'name' => $name);
        $this->updateTagsToCustomers($updateCustomers);
        return $newCustGrpID;
    }

    public function deleteCustomerSavedSearch($customerGroupId)
    {
        $groupDeleted = false;
        $deleteGroup = $this->call('DELETE', '/admin/customer_saved_searches/' . $customerGroupId . '.json');
        if (!isset($deleteGroup['method'])) {
            $groupDeleted = true;
        }
        return $groupDeleted;
    }

    public function savedSearchIDFromCustID($customerID)
    {
        $customer = $this->call('GET', '/admin/customers/' . $customerID . '.json');
        $tags = explode(', ', $customer['tags']);
        $groupName = '';
        foreach ($tags as $tag) {
            if (strpos($tag, 'imp_grp_') !== false) {
                $groupName = str_replace('imp_grp_', '', $tag);
                break;
            }
        }
        return $groupName;
    }

    /*     * ************************* Customer Module  End **************************** */

    /*     * ***************************** Order Module Start ***************************** */
    /**
     * creat order quote request 
     *
     * @author     debashisd@riaxe.com
     * @date       11 Aug 2019
     * @param  quotation data Array
     * @response   Array of new order id
     */
    public function createShopOrder($quoteData)
    {

        $lineItems = array();
        foreach ($quoteData['product_data'] as $item) {
            $isStickerProduct = $item['is_sticker_product'];
            $sheetName = $item['sheet_name'];
            if ($item['variant_id'] == 0) {
                $productData = $this->call('GET', '/admin/products/' . $item['product_id'] . '.json');
                $variantId = $productData['variants'][0]['id'];
            }
            if (isset($isStickerProduct) && $isStickerProduct == 1) {
                $optionArr = [
                    "name" => "Sheets",
                    "values" => $sheetName
                ];
                $variantID = $variantId;
                $variant = $this->call('GET', '/admin/products/' . $item['product_id'] . '/variants/' . $variantID . '.json');
                $newPrice = $item['updated_price'];
                $thisVariant = array(
                    "sku" => $variant['sku'],
                    "price" => $newPrice,
                    "option1" => $sheetName,
                    "taxable" => $variant['taxable'],
                    "weight" => $variant['weight'],
                    "weight_unit" => $variant['weight_unit'],
                    "inventory_management" => $variant['inventory_management'],
                    "inventory_policy" => $variant['inventory_policy'],
                );
                $variantArr[] = $thisVariant;
                $productArray = array(
                    "product" => array(
                        "title" => addslashes($productData['title']),
                        "vendor" => "imprintNext",
                        "published_scope" => "web",
                        "template_suffix" => "imprint_dummy_product",
                        "options" => $optionArr,
                        "variants" => $variantArr
                    ),
                );
                $thisProductArray[] = $productArray;
            }
            if ($item['design_cost'] > 0 || ($item['unit_price'] !== $item['updated_price'])) {
                $product_data = array(
                    "product_id" => $item['product_id'],
                    "variant_id" => ($item['variant_id'] == 0) ? $variantId : $item['variant_id'],
                    "options" => array(),
                    "custom_price" => $item['design_cost'],
                    "ref_id" => $item['custom_design_id'],
                    "qty" => ($item['quantity'] != 0) ? $item['quantity'] : $item['overall_quantity'],
                    "product_price" => $item['updated_price']
                );

                $dupItem = $this->addCustomProduct($product_data);
                $thisItem['variant_id'] =  $dupItem['new_variant_id'];
                $thisItem['quantity'] =  !empty($item['quantity']) ? $item['quantity'] : $item['overall_quantity'];
            } else {
                $thisItem['variant_id'] =  ($item['variant_id'] == 0) ? $variantId : $item['variant_id'];
                $thisItem['quantity'] =  ($item['quantity'] != 0) ? $item['quantity'] : $item['overall_quantity'];
            }
            $thisItem['properties'] = array();
            if ($item['custom_design_id'] > 0 && !empty($item['custom_design_id'])) {
                $thisItem['properties'][] = array("name" => "_refid", "value" => $item['custom_design_id']);
            } else {
                $thisItem['properties'][] = array("name" => "_refid", "value" => -1);
                foreach ($item['files'] as $key => $file) {
                    $side = $key + 1;
                    $thisItem['properties'][] = array("name" => "design_url_" . $side, "value" => $file['decoration_area'][0]['upload_design_url']);
                    $thisItem['properties'][] = array("name" => "preview_url_" . $side, "value" => $file['decoration_area'][0]['upload_preview_url']);
                }
            }
            $lineItems[] = $thisItem;
        }

        //for multisticker product creation
        $addProduct = $this->call('POST', '/admin/products.json', $thisProductArray);
        $thisItem['variant_id'] =  $addProduct['variants'][0]['id'];
        $thisItem['quantity'] =  $item['quantity'];
        $thisItem['price'] = $addProduct['variants'][0]['price'];


        $customer = $this->call('GET', '/admin/customers/' . $quoteData['customer_id'] . '.json');
        $customerAddress = $this->call('GET', '/admin/customers/' . $quoteData['customer_id'] . '/addresses/' . $quoteData['shipping_id'] . '.json');
        $customerInfo = array('first_name' => $customer['first_name'], 'last_name' => $customer['last_name'], 'email' => $customer['email']);
        $address = array(
            'first_name' => $customerAddress['first_name'],
            'last_name' => $customerAddress['last_name'],
            'address1' => $customerAddress['address1'],
            'address2' => $customerAddress['address2'],
            'phone' => $customerAddress['phone'],
            'city' => $customerAddress['city'],
            'province' => $customerAddress['province'],
            'country' => $customerAddress['country'],
            'zip' => $customerAddress['zip']
        );
        $discount = $shipping = $tax = array();
        $orderAmount = $quoteData['design_total'];
        if ($quoteData['discount_amount'] > 0) {
            $discount = array(
                'type' => "script",
                'description' => "Discount set during quptation in ImprintNext",
                'value' => $quoteData['discount_amount'],
                'value_type' => ($quoteData['discount_type'] == "flat" ? "fixed_amount" : "percentage"),
                'allocation_method' => "across",
                'target_selection' => "all",
                'target_type' => "line_item",
            );
            $discountAmnt = ($quoteData['discount_type'] == "flat" ? $quoteData['discount_amount'] : ($quoteData['discount_amount'] / 100) * $quoteData['design_total']);
            $orderAmount = $quoteData['design_total'] - $discountAmnt;
        }
        if ($quoteData['shipping_amount'] > 0) {
            $shipping = array(
                'price' => $quoteData['shipping_amount'],
                'title' => "manual",
            );
        }
        if ($quoteData['tax_amount'] > 0) {
            $tax[] = array(
                'price' => ($quoteData['tax_amount'] / 100) * $orderAmount,
                'rate' => $quoteData['tax_amount'] / 100,
                'title' => "OrderTax"
            );
            $tax[] = array(
                'price' => ($quoteData['rush_type'] == "flat" ? $quoteData['rush_amount'] : ($quoteData['rush_amount'] / 100) * $quoteData['design_total']),
                'rate' => $quoteData['shipping_amount'],
                'title' => "Rush charge"
            );
        }

        $tax1 = (($quoteData['tax_amount'] / 100) * $orderAmount);
        $tax2 = ($quoteData['rush_type'] == "flat" ? $quoteData['rush_amount'] : ($quoteData['rush_amount'] / 100) * $orderAmount);
        $totalTax =  $tax1 + $tax2;


        // order details
        $thisOrder['email'] = $customer['email'];
        $thisOrder['taxes_included'] = false;
        $thisOrder['total_tax'] = $totalTax;
        $thisOrder['total_discounts'] = ($quoteData['discount_type'] == "flat" ? $quoteData['discount_amount'] : (($quoteData['discount_amount'] / 100) * $quoteData['design_total']));
        $thisOrder['send_receipt'] = true;
        $thisOrder['send_fulfillment_receipt'] = true;
        $thisOrder['tags'] = "Quotation, " . $quoteData['quote_id'];
        $thisOrder['line_items'] = $lineItems;
        $thisOrder['customer'] = $customerInfo;
        $thisOrder['billing_address'] = $address;
        $thisOrder['shipping_address'] = $address;
        $thisOrder['note'] = ($quoteData['note']) ? $quoteData['note'] : "";
        if (!empty($tax)) {
            $thisOrder['tax_lines'] = $tax;
        }
        if (!empty($discount)) {
            $thisOrder['discount_applications'] = $discount;
        }
        if (!empty($shipping)) {
            $thisOrder['shipping_lines'] = array($shipping);
        }
        // $thisOrder['transactions'] = $address;
        // $thisOrder['financial_status'] = $address;
        $orderData['order'] = $thisOrder;
        $addOrder = $this->call('POST', '/admin/orders.json', $orderData);
        return array("id" => $addOrder['id'], "order_number" => $addOrder['order_number']);
    }

    /**
     * Get Order w.r.t Customer
     *
     * @author     debashrib@riaxe.com
     * @date       17 dec 2019
     * @param  customer id
     * @response   Array of order details
     */
    public function getOrderDetailsByCustomer($customerID)
    {
        $customerOrder = [];
        if ($customerID != '') {
            $customerOrder = $this->call('GET', '/admin/customers/'.$customerID.'/orders.json?status=any&limit=20');
        }
        return $customerOrder;
    }

    /**
     * retrieve orders from store
     *
     * @author debashrib@riaxe.com
     * @date  18 dec 2019
     * @param array $callParams
     * @param string lastOrderId
     * @param int store
     * @param int range
     * @param date fromDate
     * @param date toDate
     * @return json
     */
    public function ordersAll($callParams)
    {
        $sortOrder = $callParams['order'];
        $sortOrderBy = $callParams['orderby'];
        $customerID = $callParams['customer_id'];

        // set server timezone
        $shop = $this->call('GET', '/admin/shop.json?fields=iana_timezone');
        if (!empty($shop['iana_timezone'])) {
            date_default_timezone_set($shop['iana_timezone']);
        }

        // geting orders with filter
        if (!empty($customerID)) {
            $orders = $this->call('GET', '/admin/customers/' . $customerID . '/orders.json?status=any');
        } else {
            $orders = $this->orders_get($callParams);
        }
        $cancelIDs = [];
        $closedIDs = [];
        $canceledOrders = $this->call('GET', '/admin/orders.json?status=cancelled');
        $cancelIDs = array_column($canceledOrders, 'id');

        $archivedOrders = $this->call('GET', '/admin/orders.json?status=closed');
        $closedIDs = array_column($archivedOrders, 'id');

        // Get total order count
        $ordersCount = (!empty($orders)) ? count($orders) : 0;

        $finalResult = [];
        $result = [];
        // Get total order count
        if ($ordersCount > 0) {
            foreach ($orders as $order) {
                $res = [];
                if (in_array($order['id'], $cancelIDs)) {
                    $status = "Canceled";
                } elseif (in_array($order['id'], $closedIDs)) {
                    $status = "Archived";
                } else {
                    $status = self::shopify_order_status($order);
                }
                $res = [
                    "id" => $order['id'],
                    "order_number" => $order['order_number'],
                    "customer_id" => $order['customer']['id'],
                    "customer_first_name" => $order['customer']['first_name'],
                    "customer_last_name" => $order['customer']['last_name'],
                    "created_date" => date('Y-m-d H:i:s', strtotime($order['created_at'])),
                    "total_amount" => (float)round($order['total_price'], 2),
                    "currency" => $order['currency'],
                    "order_total_quantity" => array_sum(array_column($order['line_items'], 'quantity')),
                    "production" => '',
                    "is_customize" => $order['is_customize'],
                    "status" => $status
                ];
                $result[] = $res;
            }
            // Sorting Order result
            if (isset($sortOrder) && isset($sortOrderBy) && $sortOrder != '' && $sortOrderBy != '') {
                // sort by price
                if ($sortOrderBy == 'price') {
                    if ($sortOrder == 'desc') {
                        $finalResult = $this->array_sort_by_column($result, 'total_amount', SORT_DESC);
                    } else {
                        $finalResult = $this->array_sort_by_column($result, 'total_amount', SORT_ASC);
                    }
                    // sort by customer
                } elseif ($sortOrderBy == 'name') {
                    if ($sortOrder == 'desc') {
                        $finalResult = $this->array_sort_by_column($result, 'customer_first_name', SORT_DESC);
                    } else {
                        $finalResult = $this->array_sort_by_column($result, 'customer_first_name', SORT_ASC);
                    }
                    // sort by date
                } elseif ($sortOrderBy == 'date') {
                    if ($sortOrder == 'desc') {
                        $finalResult = $this->array_sort_by_column($result, 'created_date', SORT_DESC);
                    } else {
                        $finalResult = $this->array_sort_by_column($result, 'created_date', SORT_ASC);
                    }
                } else {
                    $finalResult = $result;
                }
            } else {
                $finalResult = $result;
            }
        }

        return $finalResult;
    }

    /**
     * retrieve order details from store
     *
     * @param array $callParams
     * @param string orderIncrementId
     * @param int store
     * @return json
     */
    public function orderDetails($args)
    {
        $order_id = $args['id'];
        $order = $this->call('GET', '/admin/orders/' . $order_id . '.json');
        $billingAddress = [];
        $shippingAddress = [];
        $lineItems = [];
        $colorPos = "";
        $sizePos = "";

        // set server timezone
        $shop = $this->call('GET', '/admin/shop.json?fields=iana_timezone');
        if (!empty($shop['iana_timezone'])) {
            date_default_timezone_set($shop['iana_timezone']);
        }

        // Getting Billing Address of perticular order
        $billingAddress = [
            'first_name' => $order['billing_address']['first_name'],
            'last_name' => ($order['billing_address']['last_name'] != '') ? $order['billing_address']['last_name'] : '',
            'email' => $order['email'],
            'phone' => ($order['billing_address']['phone'] != '') ? $order['billing_address']['phone'] : '',
            'address_1' => ($order['billing_address']['address1'] != '') ? $order['billing_address']['address1'] : '',
            'address_2' => ($order['billing_address']['address2'] != '') ? $order['billing_address']['address2'] : '',
            'country' => $order['billing_address']['country'],
            'state' => $order['billing_address']['province'],
            'city' => $order['billing_address']['city'],
            'postcode' => $order['billing_address']['zip'],
            'company' => ($order['billing_address']['company'] != '') ? $order['billing_address']['company'] : '',
            'country_code' => $order['billing_address']['country_code'],
            'state_code' => $order['billing_address']['province_code'],
        ];

        // Getting Shipping Address of perticular order
        $shippingAddress = [
            'first_name' => $order['shipping_address']['first_name'],
            'last_name' => $order['shipping_address']['last_name'],
            'email' => $order['email'],
            'phone' => ($order['shipping_address']['phone'] != '') ? $order['shipping_address']['phone'] : '',
            'address_1' => ($order['shipping_address']['address1'] != '') ? $order['shipping_address']['address1'] : '',
            'address_2' => ($order['shipping_address']['address2'] != '') ? $order['shipping_address']['address2'] : '',
            'country' => $order['shipping_address']['country'],
            'state' => $order['shipping_address']['province'],
            'city' => $order['shipping_address']['city'],
            'postcode' => $order['shipping_address']['zip'],
            'company' => ($order['shipping_address']['company'] != '') ? $order['shipping_address']['company'] : '',
            'country_code' => $order['shipping_address']['country_code'],
            'state_code' => $order['shipping_address']['province_code'],
        ];

        // Getting line items of perticular order
        $orderItems = [];
        if (isset($args['ui'])) {
            if (!empty($order['line_items'])) {
                foreach ($order['line_items'] as $lineItems) {
                    $refID = '';
                    foreach ($lineItems['properties'] as $value) {
                        if ($value['name'] == "_refid") {
                            $refID = $value['value'];
                            break;
                        }
                    }
                    if ($refID == 0 && $lineItems['vendor'] == "imprintNext") {
                        // get prod data $lineItems['product_id']
                        $productData = $this->call('GET', '/admin/products/' . $lineItems['product_id'] . '.json');
                        $tags = $productData['tags'];
                        $tags = explode('IMP_REF_ID_', $tags);
                        $tags = $tags[1];
                        $tags = explode(',',$tags);
                        $refID = $tags[0];
                    }

                    $customDesignId = (isset($refID) && $refID != '') ? $refID : '';

                    $tag = (explode(", ", $order['tags']));
                    if (in_array("Quotation", $tag)) {
                        $quote = 1;
                    }

                    $variant = $this->call('GET', '/admin/variants/' . $lineItems['variant_id'] . '.json');

                    $colorPos = $this->getAttributePostion($this->colorAttrName, $lineItems['product_id']);
                    $sizePos = $this->getAttributePostion($this->sizeAttrName, $lineItems['product_id']);
                    $stitchColorPos = $this->getAttributePostion('stitch_color', $lineItems['product_id']);
                    $orgVarId = $this->getParentVariantID($lineItems['variant_id']);
                    $realProdData = json_decode($this->shopifyParentProductID($orgVarId), true);
                    $thisLineItem =  [
                        'id' => $lineItems['id'],
                        'product_id' => $lineItems['product_id'],
                        'variant_id' => $lineItems['variant_id'],
                        'parent_prod_id' => $realProdData['pid'],
                        'parent_var_id' => $realProdData['vid'],
                        'name' => $lineItems['title'] == "Sticker" ? $lineItems['title'] : $lineItems['name'],
                        'price' => (float)round($lineItems['price'], 2),
                        'quantity' => $lineItems['quantity'],
                        'total' => (float)round(($lineItems['price'] * $lineItems['quantity']), 2),
                        'sku' => $lineItems['sku'],
                        'custom_design_id' => $customDesignId,
                        'size' => isset($variant['option' . $sizePos]) ? $variant['option' . $sizePos] : '',
                        'color' => isset($variant['option' . $colorPos]) ? $variant['option' . $colorPos] : '',
                        'stitch_color' => isset($variant['option' . $stitchColorPos]) ? $variant['option' . $stitchColorPos] : '',
                        'images' => $refID > 0 ? $this->getStoreProductImages($refID, $lineItems['variant_id']) : $this->getQuoteImages($lineItems['properties'], $lineItems['variant_id']),
                        'productDecorationSettingData' => '',
                        'weight' => $variant['weight'],
                        'weight_unit' => $variant['weight_unit'],
                        'is_quote_order' => isset($quote) ? $quote : 0
                    ];
                    if (empty($thisLineItem['images'])) {
                        $rawVariant = $this->call('GET', '/admin/variants/' . $lineItems['variant_id'] . '.json');
                        $thisProductID = $rawVariant['product_id'];
                        $rawProduct = $this->call('GET', '/admin/products/' . $thisProductID . '.json?fields=images,image');
                        if (!empty($rawVariant['image_id'])) {
                            $thisVariantImg = $this->call('GET', '/admin/products/' . $thisProductID . '/images/' . $rawVariant['image_id'] . '.json');
                            $thisLineItem['images'][] = array(
                                "src" => $thisVariantImg['src'],
                                "thumbnail" => $thisVariantImg['src']
                            );
                        } else {
                            $thisLineItem['images'][] = array(
                                "src" => $rawProduct['image']['src'],
                                "thumbnail" => $rawProduct['image']['src']
                            );
                        }
                    }
                    $orderItems[$customDesignId]['line_items'][] = $thisLineItem;
                }
                $orderItems = array_values($orderItems);
            }
        } else {
            if (!empty($order['line_items'])) {
                foreach ($order['line_items'] as $lineItems) {
                    $refID = '';
                    foreach ($lineItems['properties'] as $value) {
                        if ($value['name'] == "_refid") {
                            $refID = $value['value'];
                            break;
                        }
                    }
                    if ($refID == 0 && $lineItems['vendor'] == "imprintNext") {
                        // get prod data $lineItems['product_id']
                        $productData = $this->call('GET', '/admin/products/' . $lineItems['product_id'] . '.json');
                        $tags = $productData['tags'];
                        $tags = explode('IMP_REF_ID_', $tags);
                        $refID = $tags[1];
                    }
                    $tag = (explode(", ", $order['tags']));
                    if (in_array("Quotation", $tag)) {
                        $quote = 1;
                    }

                    $variant = $this->call('GET', '/admin/variants/' . $lineItems['variant_id'] . '.json');

                    $colorPos = $this->getAttributePostion($this->colorAttrName, $lineItems['product_id']);
                    $sizePos = $this->getAttributePostion($this->sizeAttrName, $lineItems['product_id']);
                    $stitchColorPos = $this->getAttributePostion('stitch_color', $lineItems['product_id']);
                    $thisLineItem = [
                        'id' => $lineItems['id'],
                        'product_id' => $lineItems['product_id'],
                        'variant_id' => $lineItems['variant_id'],
                        'name' => $lineItems['name'],
                        'price' => (float)round($lineItems['price'], 2),
                        'quantity' => $lineItems['quantity'],
                        'total' => (float)round(($lineItems['price'] * $lineItems['quantity']), 2),
                        'sku' => $lineItems['sku'],
                        'custom_design_id' => (isset($refID) && $refID != '') ? $refID : '',
                        'size' => isset($variant['option' . $sizePos]) ? $variant['option' . $sizePos] : '',
                        'color' => isset($variant['option' . $colorPos]) ? $variant['option' . $colorPos] : '',
                        'stitch_color' => isset($variant['option' . $stitchColorPos]) ? $variant['option' . $stitchColorPos] : '',
                        'images' => $refID > 0 ? $this->getStoreProductImages($refID, $lineItems['variant_id']) : $this->getQuoteImages($lineItems['properties'], $lineItems['variant_id']),
                        'productDecorationSettingData' => '',
                        'weight' => $variant['weight'],
                        'weight_unit' => $variant['weight_unit'],
                        'is_quote_order' => isset($quote) ? $quote : 0
                    ];
                    if (empty($thisLineItem['images'])) {
                        $rawVariant = $this->call('GET', '/admin/variants/' . $lineItems['variant_id'] . '.json');
                        $thisProductID = $rawVariant['product_id'];
                        $rawProduct = $this->call('GET', '/admin/products/' . $thisProductID . '.json?fields=images,image');
                        if (!empty($rawVariant['image_id'])) {
                            $thisVariantImg = $this->call('GET', '/admin/products/' . $thisProductID . '/images/' . $rawVariant['image_id'] . '.json');
                            $thisLineItem['images'][] = array(
                                "src" => $thisVariantImg['src'],
                                "thumbnail" => $thisVariantImg['src']
                            );
                        } else {
                            $thisLineItem['images'][] = array(
                                "src" => $rawProduct['image']['src'],
                                "thumbnail" => $rawProduct['image']['src']
                            );
                        }
                    }
                    $orderItems[] = $thisLineItem;
                }
            }
        }

        return [
            "id" => $order['id'],
            "order_number" => $order['order_number'],
            "customer_first_name" => $order['customer']['first_name'],
            "customer_last_name" => $order['customer']['last_name'],
            "customer_email" => $order['email'],
            "customer_id" => $order['customer']['id'],
            "created_date" => date('Y-m-d H:i:s', strtotime($order['created_at'])),
            "total_amount" => (float)($order['taxes_included'] ? round(($order['total_line_items_price'] - $order['total_tax']), 2) : round($order['total_line_items_price'], 2)),
            "total_tax" => (float)round($order['total_tax'], 2),
            "total_shipping" => (float)round($order['shipping_lines'][0]['price'], 2),
            "total_discounts" => (float)round($order['total_discounts'], 2),
            "currency" => $order['currency'],
            "note" => ($order['note'] != '') ? $order['note'] : '',
            "production" => '',
            "status" => self::shopify_order_status($order),
            "total_orders" => $order['customer']['orders_count'],
            "billing" => $billingAddress,
            "shipping" => $shippingAddress,
            "payment" => $order['gateway'] . " gateway",
            "store_url" => "https://{$this->shopDomain}/",
            "orders" => $orderItems
        ];
    }

    private function getQuoteImages($properties, $variantID="")
    {
        $images = array();
        $propNames = array_column($properties, "name");
        if (empty($properties) || !in_array("_refid", $propNames)) {
            $variantImage = $this->call('GET', '/admin/variants/' . $variantID . '.json');
            if (!empty($variantImage['image_id'])) {
                $thisBlankImage = $this->call('GET', '/admin/products/' . $variantImage['product_id'] . '/images/'.$variantImage['image_id'].'.json');
                $images[] =  array("src" => $thisBlankImage['src'], "thumbnail" => $thisBlankImage['src']);
                
            }else{
                $productImage = $this->call('GET', '/admin/products/' . $variantImage['product_id'] . '/images.json');
                $images[] =  array("src" => $productImage[0]['src'], "thumbnail" => $productImage[0]['src']);
            }
        }else{
            foreach ($properties as $prop) {
                if (strpos($prop['name'], 'preview_url_') !== false) {
                    $images[] =  array("src" => $prop['value'], "thumbnail" => $prop['value']);
                }
            }
        }
        return $images;
    }
    
    /**
     * GET: Shopify order details
     *
     * @param $orderID requested order ID
     * @return Array order details
     */
    public function orderInfo($orderID)
    {
        $order = $this->call('GET', '/admin/orders/' . $orderID . '.json');
        $orderDetails = $orderItems = [];
        $lineItems = [];
        foreach ($order['line_items'] as $lineItems) {
            foreach ($lineItems['properties'] as $value) {
                if ($value['name'] == "_refid") {
                    $refID = $value['value'];
                    break;
                } else {
                    $refID = '';
                }
            }
            if ($refID == 0 && $lineItems['vendor'] == "imprintNext") {
                // get prod data $lineItems['product_id']
                $productData = $this->call('GET', '/admin/products/' . $lineItems['product_id'] . '.json');
                $tags = $productData['tags'];
                $tags = explode('IMP_REF_ID_', $tags);
                $tags = $tags[1];
                $tags = explode(',', $tags);
                $refID = $tags[0];
            }
            $thisLineItem = [
                'item_id' => $lineItems['id'],
                'print_status' => "",
                'product_id' => $lineItems['product_id'],
                'variant_id' => $lineItems['variant_id'],
                'product_sku' => $lineItems['sku'],
                'product_name' => $lineItems['title'],
                'quantity' => $lineItems['quantity'],
                'images' => $refID > 0 ? $this->getStoreProductImages($refID, $lineItems['variant_id']) : $this->getQuoteImages($lineItems['properties'], $lineItems['variant_id']),
                'categories' => array_column($this->getProductCategories($lineItems['product_id']), 'id'),
                'ref_id' => (isset($refID) && $refID != '') ? $refID : ''
            ];
            try {
                $this->adjustProductInventory($lineItems['variant_id'], $lineItems['quantity'], 0);
            } catch (Exception $e) {
                $thisLineItem['inventory_update'] = "failed";
            }
            $orderItems[] = $thisLineItem;
        }
        $orderDetails['order_details'] = [
            "order_id" => $order['id'],
            "customer_id" => $order['customer']['id'],
            "store_id" => 1,
            "order_incremental_id" => $order['order_number'],
            "order_items" => $orderItems
        ];
        return json_encode($orderDetails);
    }

    /**
     * Adjust inventory of items after the order is placed with duplicate products.
     *
     * @param params order item variant id and item quantity
     */
    private function adjustProductInventory($dupVariantID, $quantity, $isUpdate)
    {
        $status = 0;
        $VariantID = $this->getParentVariantID($dupVariantID);
        $variant = $this->call('GET', '/admin/variants/' . $VariantID . '.json');
        if (!empty($variant['imp_image_id'])) {
            $upProductVar = new ProductVariants();
            $catlogVar = $upProductVar->where('xe_id', '=', $variant['id'])->first()->toArray();
            $finalQty = $catlogVar['inventory'] - $quantity;
            $updateList = ['inventory' => $finalQty];
            $upProductVar->where('sku', '=', $variant['sku'])->update($updateList);
            $status = 1;
        } else {
            $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $variant['inventory_item_id']);
            if (!empty($invLevels)) {
                $newInventory = array(
                    "location_id" => $invLevels[0]['location_id'],
                    "inventory_item_id" => $variant['inventory_item_id'],
                    "available_adjustment" => ($isUpdate == 1) ? $quantity : -$quantity,
                );
                $this->call('POST', '/admin/inventory_levels/adjust.json', $newInventory);
                $status = 1;
            }
        }
        return $status;
    }

    /**
     * get orders from store
     *
     * @param params order filter parameters
     */
    private function orders_get($params)
    {
        $ordersData = [];
        $orders = [];
        $finalOrder = [];
        $QRYparams['limit'] = $params['per_page'];
        $customize = $params['is_customize'];
        $fromDate = ($params['fromDate'] != '') ? date("Y-m-d\TH:i:s", strtotime($params['fromDate'])) : '';
        $toDate = ($params['toDate'] != '') ? date("Y-m-d\TH:i:s", strtotime($params['toDate'] . ' +1 day')) : '';
        $sku = ($params['sku'] != '') ? $params['sku'] : '';
        $orderStatus = $params['order_status'];
        $QRYparams['status'] = (!empty($orderStatus) && $orderStatus == 'archive') ? "closed" : "open";
        if ($fromDate && $toDate) {
            $QRYparams['created_at_min'] = $fromDate;
            $QRYparams['created_at_max'] = $toDate;
        }
        if ($params['search'] != '') {
            $QRYparams['name'] = $params['search'];
        }
        $orderCount = 0;
        $ordersData = 0;
        if (empty($QRYparams['limit'])) {
            $QRYparams['limit'] = 250;
            $params['page'] = 1;
        }
        do {
            if ($orderCount == 0 && $ordersData == 0 && $params['page'] == 1) {
                $ordersData = $this->call('GET', '/admin/orders.json', $QRYparams, "orders");
            } elseif ($params['page'] == 0) {
                $cursorURL = substr($_SESSION['orders']['previous'], strpos($_SESSION['orders']['previous'], "/admin"));
                $ordersData = $this->call('GET', $cursorURL, array(), "orders");
            } else {
                $cursorURL = substr($_SESSION['orders']['next'], strpos($_SESSION['orders']['next'], "/admin"));
                $ordersData = $this->call('GET', $cursorURL, array(), "orders");
            }
            // return data based on customize filter
            $dataCount = (!empty($ordersData)) ? count($ordersData) : 0;
            if ($dataCount > 0) {
                foreach ($ordersData as $order) {
                    $getProp = array();
                    foreach ($order['line_items'] as $lt) {
                        $refID = '';
                        foreach ($lt['properties'] as $vaalue) {
                            if ($vaalue['name'] == "_refid") {
                                $refID = $lt['properties'][0]['value'];
                                break;
                            } 
                        }
                        if ($refID == '' && $lt['vendor'] == "imprintNext") {
                            // get prod data $lineItems['product_id']
                            $productData = $this->call('GET', '/admin/products/' . $lt['product_id'] . '.json');
                            $tags = $productData['tags'];
                            $tags = explode('IMP_REF_ID_', $tags);
                            $refID = $tags[1];
                        }
                        if ($refID) {
                            $getProp[] = $refID;
                        }
                    }
                    if (!empty($getProp)) {
                        $order['is_customize'] = 1;
                        $customOrders[] = $order;
                    } else {
                        $order['is_customize'] = 0;
                        $blankOrders[] = $order;
                    }
                    $allOrders[] = $order;
                }
            }
            if (isset($customize) && $customize == 1) {
                $orders = $customOrders;
            } else {
                $orders = $allOrders;
            }
            if (isset($sku) && $sku != '') {
                if (!empty($orders)) {
                    foreach ($orders as $skuOrder) {
                        $getSKU = array();
                        if (!empty($getSKU)) {
                            $finalOrder[] = $skuOrder;
                        }
                    }
                }
            } else {
                $finalOrder = $orders;
            }
            $orderCount = (!empty($finalOrder)) ? count($finalOrder) : 0;
        } while ($orderCount < $params['per_page'] && $dataCount == $params['per_page']);
        return $finalOrder;
    }

    /**
     * get order status from store
     *
     * @param array $order
     */
    private function shopify_order_status($order)
    {
        if ($order['financial_status'] == "refunded") {
            return "Refunded";
        }
        if ($order['fulfillment_status'] && $order['financial_status'] != "refunded") {
            if ($order['fulfillment_status'] == "partial") {
                return "Processing";
            } else {
                return "Complete";
            }
        }
        return "Pending";
    }

    /**
     * GET: Shopify store images
     *
     * @param $productId selected product ID
     * @return Array product details
     */
    public function getStoreProductImages($refID, $variantID)
    {
        $parentVarID = $this->getParentVariantID($variantID);
        $stateDesignPath = path('abs', 'design_state') . 'carts/' . $refID . '.json';
        $productImages = [];
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'predecorators/' . $refID . '.json';
        }
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'quotes/' . $refID . '.json';
        }
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'artworks/' . $refID . '.json';
        }
        if (file_exists($stateDesignPath)) {
            $jsonData = json_clean_decode(file_get_contents($stateDesignPath), true);
            if (!empty($jsonData)) {
                foreach ($jsonData['design_product_data'] as $designProductData) {
                    if ((is_array($designProductData)
                            && in_array($parentVarID, $designProductData['variant_id']))
                        || $designProductData['variant_id'] == $productId
                    ) {
                        $customImageUrl = $designProductData['design_urls'];
                        break;
                    }
                }
                if (empty($customImageUrl)) {
                    $customImageUrl = $jsonData['design_product_data'][0]['design_urls'];
                }
                foreach ($customImageUrl as $key => $customImage) {
                    $img = [
                        "id" => ($key + 1),
                        "src" => $customImage,
                        "thumbnail" => $customImage,
                    ];
                    $productImages[] = $img;
                }
            }
        }
        return $productImages;
    }

    /**
     * retrieve attribute position
     *
     * @param int product id
     * @param string attribute
     * @return attribute position
     */
    public function getAttributePostion($attribute, $productId)
    {
        $product = $this->call('GET', '/admin/products/' . $productId . '.json');
        $pos = "";
        foreach ($product['options'] as $option) {
            if (strtolower($option['name']) == strtolower($attribute)) {
                $pos = $option['position'];
            }
        }
        return $pos;
    }

    /**
     * retrieve sorted array
     *
     * @param array
     * @param string array column name short by
     * @param string asc/desc
     * @return sortted array
     */
    public function array_sort_by_column($arr, $col, $dir)
    {
        $sortCol = array();
        $sortCol = array_column($arr, $col);
        array_multisort($sortCol, $dir, SORT_NATURAL | SORT_FLAG_CASE, $arr);
        return $arr;
    }

    /**
     * hide or delete duplicate products
     *
     * @param product id
     * @param string delete_status
     * @return status
     */
    public function editCustomCartProduct($call_params)
    {
        $dupVariantID = $call_params['variant_id'];
        $quantity = $call_params['isQuantity'];
        if(!empty($quantity) && !empty($dupVariantID)) {
            $this->adjustProductInventory($dupVariantID, $quantity, 1);
        }
        $isCustom = false;
        $checkProd = $this->call('GET', '/admin/smart_collections.json?product_id=' . $call_params['product_id']);
        if (!empty($checkProd) || isset($checkProd['Tags'])) {
            foreach ($checkProd as $prod) {
                if ($prod['handle'] == "customized") {
                    $isCustom = true;
                }
            }
        }
        if ($call_params['isDelete'] == 0 && $isCustom) {
            $product_array = array(
                "product" => array(
                    "id" => $call_params['product_id'],
                    "published" => false,
                )
            );
            return $this->call('PUT', '/admin/products/' . $call_params['product_id'] . '.json', $product_array);
        } elseif ($call_params['isDelete'] == 1 && $isCustom) {
            return $this->call('DELETE', '/admin/products/' . $call_params['product_id'] . '.json');
        }
    }

    /**
     * retrieve order logs
     *
     * @param string orderID
     * @return array of order log from Shopify
     */
    public function getOrderLog($orderId)
    {
        $order = $this->call('GET', '/admin/orders/' . $orderId . '.json');
        $orderLogs = [];
        $storeLogs = [];
        if ($order['cancelled_at'] && $order['cancelled_at'] != "") {
            $storeLogs[] = array("log_type" => "order_status", "message" => "Order cancelled because " . $order['cancel_reason'] . " .", "created_at" => date("Y-m-d h:i:s", strtotime($order['cancelled_at'])), "updated_at" => date("Y-m-d h:i:s", strtotime($order['cancelled_at'])));
        }
        if ($order['fulfillment_status'] && $order['cancelled_at'] != "") {
            $storeLogs[] = array("log_type" => "order_status", "message" => "Order Fulfilled.", "created_at" => date("Y-m-d h:i:s", strtotime($order['cancelled_at'])), "updated_at" => date("Y-m-d h:i:s", strtotime($order['cancelled_at'])));
        }
        $storeLogs[] = array("log_type" => "order_status", "message" => "Order confirmation sent at " . $order['email'] . ".", "created_at" => date("Y-m-d h:i:s", strtotime($order['created_at'])), "updated_at" => date("Y-m-d h:i:s", strtotime($order['updated_at'])));
        $storeLogs[] = array("log_type" => "payment_status", "message" => $order['total_price'] . " " . $order['currency'] . " through " . $order['gateway'] . " is " . $order['financial_status'] . ".", "created_at" => date("Y-m-d h:i:s", strtotime($order['created_at'])), "updated_at" => date("Y-m-d h:i:s", strtotime($order['updated_at'])));
        $storeLogs[] = array("log_type" => "order_status", "message" => "Order placed through " . $order['source_name'] . " channel.", "created_at" => date("Y-m-d h:i:s", strtotime($order['created_at'])), "updated_at" => date("Y-m-d h:i:s", strtotime($order['updated_at'])));
        foreach ($storeLogs as $log) {
            if (is_array($log) && !empty($log)) {
                $log['agent_type'] = "admin";
                $log['agent_id'] = "";
                $log['status'] = "new";
                $orderLogs[] = $log;
            }
        }

        return json_encode($orderLogs);
    }

    /**
     * update order status
     *
     * @param string orderID
     * @return boolean status
     */
    public function updateOrderStatuses($orderId, $orderData)
    {
        $orderStatus = $orderData['statusKey'];
        switch ($orderStatus) {
            case 'closed':
                $orderResponse = $this->call('POST', '/admin/orders/' . $orderId . '/close.json');
                break;
            case 'cancelled':
                $orderResponse = $this->call('POST', '/admin/orders/' . $orderId . '/cancel.json');
                break;
            case 'refunded':
                $orderResponse = $this->call('POST', '/admin/orders/' . $orderId . '/cancel.json');
                break;
            case 'reopened':
                $orderResponse = $this->call('POST', '/admin/orders/' . $orderId . '/open.json');
                break;
            default:
                $orderResponse = array("id" => $orderId);
                break;
        }
        return array('id' => $orderResponse['id']);
    }

    public function archiveShopOrders($params)
    {
        $orderIDs = explode(",", str_replace(array('[', ']'), '', $params['order_id']));
        foreach ($orderIDs as $orderID) {
            $this->call('POST', '/admin/orders/' . $orderID . '/close.json');
        }
        return array("status" => 1);
    }

    public function getOrderLineItemData($orderID, $orderItemId, $is_customer)
    {
        $order = $this->call('GET', '/admin/orders/' . $orderID . '.json');
        $orderData = array();
        $orderData['order_id'] = $order['id'];
        $orderData['order_number'] = $order['order_number'];
        $orderData['item_id'] = $orderItemId;
        foreach ($order['line_items'] as $item) {
            if(!empty($item['properties'])) {
                foreach ($item['properties'] as $value) {
                    if ($value['name'] == "_refid") {
                        $refID = $value['value'];
                    } else {
                        continue;
                    }
                }
            } 
            
            if ($item['id'] == $orderItemId) {
                $parentProdID = $this->shopifyParentProductID($item['variant_id']);
                $parentProdIDs = json_decode($parentProdID, true);
                $orderData['product_id'] = $parentProdIDs['pid'];
                $orderData['variant_id'] = $parentProdIDs['vid'];
                $orderData['name'] = $item['title'];
                $orderData['quantity'] = $item['quantity'];
                $orderData['sku'] = $item['sku'];
                $orderData['price'] = $item['price'];
                $orderData['total'] = $item['price_set']['shop_money']['amount'];
                $images = $refID > 0 ? $this->getStoreProductImages($refID, $item['variant_id']) : $this->getQuoteImages($item['properties'], $parentProdIDs['vid']);
                $orderData['images'] = $images;
                $orderData['categories'] = $this->getProductCategories($parentProdID);
                $orderData['attributes'] = [];
                $orderData['custom_design_id'] = (isset($refID) && $refID != '') ? $refID : '';
                $orderData['customer_id'] = $item['customer']['id'];
                $orderData['customer_email'] = $item['customer']['email'];
                $orderData['customer_first_name'] = $item['customer']['first_name'];
                $orderData['customer_last_name'] = $item['customer']['last_name'];
                $orderData['billing'] = array(
                    'first_name' => $order['billing_address']['first_name'],
                    'last_name' => $order['billing_address']['last_name'],
                    'company' => $order['billing_address']['company'],
                    'address_1' => $order['billing_address']['address1'],
                    'address_2' => $order['billing_address']['address2'],
                    'city' => $order['billing_address']['city'],
                    'state' => $order['billing_address']['province'],
                    'country' => $order['billing_address']['country'],
                    'postcode' => $order['billing_address']['zip']
                );
                $orderData['shipping'] = array(
                    'first_name' => $order['billing_address']['first_name'],
                    'last_name' => $order['shipping_address']['last_name'],
                    'company' => $order['shipping_address']['company'],
                    'address_1' => $order['shipping_address']['address1'],
                    'address_2' => $order['shipping_address']['address2'],
                    'city' => $order['shipping_address']['city'],
                    'state' => $order['shipping_address']['province'],
                    'country' => $order['shipping_address']['country'],
                    'postcode' => $order['shipping_address']['zip']
                );
                $orderData['country'] = $order['shipping_address']['country'];
                $orderData['postcode'] = $order['shipping_address']['zip'];
            }
        }

        return $orderData;
    }
    /**
     * Get Order short info for purchase order
     *
     * @author     sonali@imprintnext.com
     * @date       08 nov 2022
     * @param  customer id
     * @response   Array of order details
     */

    public function getOrderShotInfo($orderIds, $storeId)
    {
        foreach ($orderIds as $orderId) {
            $orderDetails = $this->call('GET', '/admin/orders/' . $orderId . '.json');
            $shop = $this->call('GET', '/admin/shop.json?fields=iana_timezone');
            if (!empty($shop['iana_timezone'])) {
                date_default_timezone_set($shop['iana_timezone']);
            }
            $orderData[] = [
                "id" => $orderDetails['id'],
                "order_number" => $orderDetails['order_number'],
                "order_total_quantity" => 1,
                "customer_first_name" => $orderDetails['billing_address']['first_name'],
                "customer_last_name" => $orderDetails['billing_address']['last_name'],
                "created_date" => date('Y-m-d H:i:s', strtotime($orderDetails['created_at'])),
                "status" => self::shopify_order_status($orderDetails),
            ];
        }
        return $orderData;
    }

    public function createStoreReorder($OrderID)
    {
        $newCartItems = array();
        $orderDetails = $this->call('GET', '/admin/orders/' . $OrderID . '.json?fields=line_items');
        foreach ($orderDetails['line_items'] as $lineItem) {
            //calling product API again to check if exists as order api response is not reliable
            $thisProdData = $this->call('GET', '/admin/products/' . $lineItem['product_id'] . '.json?fields=id', [], "" . true);
            $productExists = (!array_key_exists("method", $thisProdData)) ? true : false;
            $customProduct = $this->checkCustomLineItem($lineItem['properties']);
            //if custom/non custom product exists, good to go
            //if product is not there but custom one then create on the fly
            if ($productExists && !$customProduct['is_custom']) {
                $newCartItems[] = $customProduct['custom_design_id'] . '-' . $lineItem['variant_id'] . '-' . $lineItem['quantity'] . '-' . "0";
            } elseif ($customProduct['is_custom']) { //
                $thisCustomDesignID = $customProduct['custom_design_id'];
                $parentVariantID = $customProduct['orig_variant_id'];
                //create a duplicate prod
                // prepare cart addition array for duplication
                $variant = $this->call('GET', '/admin/variants/' . $parentVariantID . '.json');
                $parentProdID = $variant['product_id'];
                $cartItemArray = array();
                if (is_numeric($parentProdID)) {
                    $cartItemArray[] = array(
                        "product_id" => $parentProdID,
                        "qty" => $lineItem['quantity'],
                        "variant_id" => $parentVariantID,
                        "options" => [],
                        "added_price" => $customProduct['custom_price_per_unit'],
                        "custom_design_id" => $thisCustomDesignID,
                    );
                }
                $cartItemArr = $this->createAddToCartLink($cartItemArray, $thisCustomDesignID);
                $newCartItems[] = $cartItemArr[0];
            } else {
                continue;
            }
        }
        //send for add to cart
        if (!empty($newCartItems)) {
            $getShopDomain = $this->shopDomain();
            $thisShopDomain = !empty($getShopDomain['domain']) ? $getShopDomain['domain'] : SHOPIFY_SHOP . ".myshopify.com";
            $CartURL = 'https://' . $thisShopDomain . '/cart?view=refitem&ref=' . implode('--', $newCartItems);
            $response = array(
                "status" => 1,
                "cart_link" => $CartURL,
            );
        } else {
            $response = array(
                "status" => 0,
                "message" => "Items of the selected Orders are not available",
            );
        }
        return $response;
    }

    private function checkCustomLineItem($properties)
    {
        $customData = array(
            'is_custom' => false,
            'custom_design_id' => 0
        );
        foreach ($properties as $prop) {
            if ($prop['name'] == "_refid") {
                $customData = array(
                    'is_custom' => true,
                    'custom_design_id' => $prop['value']
                );
            }
            if ($prop['name'] == "_parent_var_id") {
                $customDataProd = explode("||", $prop['value']);
                $customData['orig_variant_id'] = $customDataProd[0];
                $customData['custom_price_per_unit'] = $customDataProd[1];
            }
        }
        return $customData;
    }

    /*     * ***************************** Order Module End ***************************** */
    /*     * ***************************** Cart Module Start ***************************** */
    /**
     * Previously used for add to cart but now only for quote order creation.
     * 
     * @author inkXE
     * @date 20 Dec 2018
     * @param product details
     * @return Cart link
     */
    public function addCustomProduct($callParams)
    {

        $pid = $callParams['product_id'];
        $variantID = $callParams['variant_id'];
        $quantity = $callParams['qty'];
        $productUpdatedPrice = $callParams['product_price'];
        $tierPriceData = array();
        $product = $this->call('GET', '/admin/products/' . $pid . '.json');
        if ($variantID == $pid) {
            $variantID = $product['variants'][0]['id'];
        }
        $variant = $this->call('GET', '/admin/products/' . $pid . '/variants/' . $variantID . '.json');
        // get store location id. Currently set for main store location.

        $variantPrice = $variant['price'];
        $inventoryQty = $variant['inventory_quantity'];
        // get option array for new product
        $optionArr = array();
        foreach ($product['options'] as $opt) {
            array_push($optionArr, array("name" => $opt['name'], "position" => $opt['position']));
            // push the third attribute to the variant details. if it has extra one
            if (strtolower($opt['name']) !== 'title') {
                $newAttribute['option' . $opt['position']] = $variant['option' . $opt['position']];
            } else {
                $newAttribute['option1'] = '';
            }
            // change price in case of tier pricing
            if (strtolower($opt['name']) == "quantity") {
                $tierPriceData = $this->getTierPrice($product);
                $isTier = true;
            }
        }
        $metaData = $this->call('GET', '/admin/products/' . $pid . '/metafields.json');
        $tierPriceData = array();
        $commonTierPrice = array();
        $variantTierPrice = array();
        $sameforAllVariants = $isTier = false;
        if (!empty($metaData)) {
            foreach ($metaData as $meta) {
                if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') {
                    $tierContent = $meta['value'];
                    $tierPriceData = json_decode($tierContent, true);
                    $isTier = true;
                    break;
                }
            }
            // little confusion with the key here but was done as sent from admin form field
            if ($tierPriceData['pricing_per_variants'] == 'true') {
                $sameforAllVariants = true;
                foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                    $commonTierPrice[] = array(
                        "upper_limit" => $discount['upper_limit'],
                        "lower_limit" => $discount['lower_limit'],
                        "discount" => $discount['discount'],
                        "discountType" => $tierPriceData['discount_type']
                    );
                }
            } else {
                foreach ($tierPriceData['price_rules'] as $variantRule) {
                    foreach ($variantRule['discounts'] as $discount) {
                        $variantTierPrice[$variantRule['id']][] = array(
                            "upper_limit" => $discount['upper_limit'],
                            "lower_limit" => $discount['lower_limit'],
                            "discount" => $discount['discount'],
                            "discountType" => $tierPriceData['discount_type']
                        );
                    }
                }
            }
        }

        if ($isTier) {
            $variantPrice = ($sameforAllVariants === true ? $this->getPriceAfterTierDiscount($commonTierPrice, $variant['price'], $quantity) : $this->getPriceAfterTierDiscount($variantTierPrice[$callParams['variant_id']], $variant['price'], $quantity));
        }
        if ($callParams['is_variable_decoration']) {
            $newPrice = $callParams['custom_price'];
        } else {
            $newPrice = !empty($productUpdatedPrice) ? $productUpdatedPrice + $callParams['custom_price'] : $variantPrice + $callParams['custom_price'];
        }
        $refID = $callParams['ref_id'];
        // fetch the png image url from server //
        $imgPngArr = $this->getDesignPreviewImages($variantID, $pid, $refID, "artworks");
        $pngArr = array();
        if (is_countable($imgPngArr)) {
            $kount = 1;
            foreach ($imgPngArr as $pngUrl) {
                $pngArr[] = array("src" => $pngUrl, "position" => $kount);
                $kount++;
            }
        }
        // get variant details
        $variantArr = array(
            "sku" => $variant['id'] . "_" . $variant['sku'],
            "price" => $newPrice,
            "taxable" => $variant['taxable'],
            "weight" => $variant['weight'],
            "weight_unit" => $variant['weight_unit'],
            "inventory_management" => $variant['inventory_management'],
            "inventory_policy" => $variant['inventory_policy'],
        );
        $parentVarInvItemIds[] = $variant['inventory_item_id'];
        $variantArr = array_merge($variantArr, $newAttribute);
        $productArray = array(
            "product" => array(
                "title" => addslashes($product['title']),
                "vendor" => "imprintNext",
                "tags" => "customized",
                "published" => true,
                "published_scope" => "web",
                "options" => $optionArr,
                "variants" => array($variantArr),
                "images" => $pngArr,
                "image" => $pngArr['0'],
            ),
        );

        $addProduct = $this->call('POST', '/admin/products.json', $productArray);
        if ($addProduct['variants'][0]['inventory_management'] == 'shopify') {
            $newProductVariants = $addProduct['variants'];
            $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
            if (!empty($parentVarInvItemIds[0])) {
                foreach ($parentVarInvItemIds as $invL => $inventory) {
                    $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventory);
                    foreach ($invLevels as $level) {
                        $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
                        if (!$locationData['legacy']) {
                            $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" => $inventoryItems[$invL], "available" => $inventoryQty);
                            $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                        }
                    }
                }
            }
        }
        $thisVar = $addProduct['variants'][0];
        return array('new_product_id' => $addProduct['id'], 'new_variant_id' => $thisVar['id']);
    }

    /**
     * new addtocart api for fast response.
     * 
     * @author inkXE
     * @date 20 Dec 2018
     * @param product details
     * @return Cart link
     */
    public function createAddToCartLink($cartProduct, $designID)
    {
        $pid = $cartProduct[0]['product_id'];
        $tierPriceData = array();
        $product = $this->call('GET', '/admin/products/' . $pid . '.json');
        $cartItemsArr = $metaDataInfo = array();

        //do not create duplicate product for blank product add to cart
         if ($designID > 0) {
            $imgPngArr = $this->getDesignPreviewImages($cartProduct[0]['variant_id'], $cartProduct[0]['product_id'], $designID);
        }else {
            foreach ($cartProduct as $blankItem) {
                $cartItemsArr[] = $designID . '-' . $blankItem['variant_id'] . '-' . $blankItem['qty'] . '-' ."00";
            }
            return $cartItemsArr;
        }

        // Get tier price details
        $metaData = $this->call('GET', '/admin/products/' . $pid . '/metafields.json');
        $tierPriceData = array();
        $commonTierPrice = array();
        $variantTierPrice = array();
        $sameforAllVariants = $isTier = false;
        if (!empty($metaData)) {
            foreach ($metaData as $meta) {
                if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') {
                    $tierContent = $meta['value'];
                    $tierPriceData = json_decode($tierContent, true);
                    $isTier = true;
                    break;
                }
            }
            // little confusion with the key here but was done as sent from admin form field
            if ($tierPriceData['pricing_per_variants'] == 'true') {
                $sameforAllVariants = true;
                foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                    $commonTierPrice[] = array(
                        "upper_limit" => $discount['upper_limit'],
                        "lower_limit" => $discount['lower_limit'],
                        "discount" => $discount['discount'],
                        "discountType" => $tierPriceData['discount_type']
                    );
                }
            } else {
                foreach ($tierPriceData['price_rules'] as $variantRule) {
                    foreach ($variantRule['discounts'] as $discount) {
                        $variantTierPrice[$variantRule['id']][] = array(
                            "upper_limit" => $discount['upper_limit'],
                            "lower_limit" => $discount['lower_limit'],
                            "discount" => $discount['discount'],
                            "discountType" => $tierPriceData['discount_type']
                        );
                    }
                }
            }
        }
        // get option array for new product
        $optionArr = array();
        $parentAttributes = $cartProduct[0]['options'];
        $isSticker = $cartProduct[0]['is_sticker_product'];
        if (!empty($parentAttributes)) {
            $position = 0;
            foreach ($parentAttributes as $key => $opt) {
                if (!strpos($key, "_id")) {
                    $position++;
                    array_push($optionArr, array("name" => $key, "position" => $position));
                }
            }
        }
        $sheetData = array_column($cartProduct, 'sheet_name');
        if ($isSticker == 1 && !empty($sheetData)) {
            $optionArr = [
                "name" => "Sheets",
                "values" => $sheetData

            ];
        }
        // prepare variant array
        foreach ($cartProduct as $key => $cartVariant) {
            $variantOpt = $cartVariant['options'];
            $variantID = $cartVariant['variant_id'];
            $quantity = $cartVariant['qty'];
            if ($cartVariant['total_qty'] > 0) {
                $quantity = $cartVariant['total_qty'];
            }
            if ($variantID == $pid) {
                $variantID = $product['variants'][0]['id'];
            }
            $variant = $this->call('GET', '/admin/products/' . $pid . '/variants/' . $variantID . '.json');
            $variantPrice = $variant['price'];
            if ($isTier) {
                $variantPrice = ($sameforAllVariants === true ? $this->getPriceAfterTierDiscount($commonTierPrice, $variant['price'], $quantity) : $this->getPriceAfterTierDiscount($variantTierPrice[$variantID], $variant['price'], $quantity));
            }
            $newPrice = $variantPrice + $cartVariant['added_price'];
            // get variant details
            $thisVariant = array(
                "sku" => $variant['sku'],
                "price" => $newPrice,
                "option1" => $cartVariant['sheet_name'],
                "taxable" => $variant['taxable'],
                "weight" => $variant['weight'],
                "weight_unit" => $variant['weight_unit'],
                "inventory_management" => $variant['inventory_management'],
                "inventory_policy" => $variant['inventory_policy'],
            );
            $parentVarInvItemIds[] = $variant['inventory_item_id'];
            $parentItemQty[] = $variant['inventory_quantity'];
            if (!empty($variantOpt)) {
                $position = 0;
                foreach ($variantOpt as $key => $opt) {
                    if (!strpos($key, "_id")) {
                        $position++;
                        $thisVariant['option' . $position] = addslashes($opt);
                    }
                }
            }
            $variantArr[] = $thisVariant;
        }

        $productArray = array(
            "product" => array(
                "title" => addslashes($product['title']),
                "vendor" => "imprintNext",
                "tags" => $product['tags'] . ", " . ($isTier ? "customized, imp_" . $pid . ",volumediscount, IMP_REF_ID_" . $designID : "customized, imp_" . $pid . ", IMP_REF_ID_" . $designID),
                "published" => true,
                "published_scope" => "web",
                "template_suffix" => "imprint_dummy_product, imp_noindex",
                "options" => $optionArr,
                "variants" => $variantArr,
                "images" => !empty($imgPngArr[0]) ? array(array("src" => $imgPngArr[0], "position" => 1)) : array(array("src" => $product['image']['src'], "position" => 1)),
                "image" => !empty($imgPngArr[0]) ? array($imgPngArr[0]) : array($product['image']['src']),
            ),
        );
        $addProduct = $this->call('POST', '/admin/products.json', $productArray);
        if ($addProduct['variants'][0]['inventory_management'] == 'shopify') {
            $newProductVariants = $addProduct['variants'];
            $inventoryItems = array_column($newProductVariants, 'inventory_item_id');
            if(SHOPIFY_VARIANTS_PLUGIN == 1){
                $parentVarInvItemIds = array_column($addProduct['variants'], 'inventory_item_id');
            }
            foreach ($parentVarInvItemIds as $invL => $inventory) {
                $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids='.$inventory);
                $newInventory = array("location_id" => $invLevels[0]['location_id'], "inventory_item_id" =>$inventoryItems[$invL], "available" =>$parentItemQty[$invL]);
                $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
            }
        }
        foreach ($addProduct['variants'] as $key => $newVariant) {
            $cartItemsArr[] = $designID . '-' . $newVariant['id'] . '-' . $cartProduct[$key]['qty'] . '-' . $cartProduct[$key]['variant_id'] . '||' . sprintf("%.2f", $cartProduct[$key]['added_price']);
            $relationOfIDs[$newVariant['id']] = $cartProduct[$key]['variant_id'];
        }
        $relationOfIDs['product_type'] = ($pid == $relationOfIDs[$newVariant['id']]) ? "simple" : "variable";
        $metaDataInfo = array(
            "metafield" => array(
                "namespace" => "imprintnext",
                "key" => "imprint_prod_rel",
                "value" => addslashes(json_encode($relationOfIDs)),
                "type" => "json",
            ),
        );
        $metaDataInfoHide = array(
            "metafield" => array(
                "namespace" => "seo",
                "key" => "hidden",
                "value" => 1,
                "type" => "number_integer",
            ),
        );
        if ($isTier) {
            //save the tier price details in deuplicate product
            $saveTierMataDetails = [
                "pricing_per_variants" => "true",
                "discount_type" => $commonTierPrice[0]['discountType'],
                "price_rules" => [
                    [
                        "id" => $addProduct['id'],
                        "discounts" => !empty($commonTierPrice) ? $commonTierPrice : $variantTierPrice
                    ]
                ],
                "store_id" => "1",
                "productID" =>  $addProduct['id']
            ];
            $this->saveProductTierPrice($saveTierMataDetails);
        }
        if (!empty($metaDataInfo)) {
            $this->call('POST', '/admin/products/' . $addProduct['id'] . '/metafields.json', $metaDataInfo);
            $this->call('POST', '/admin/products/' . $addProduct['id'] . '/metafields.json', $metaDataInfoHide);
        }

        return $cartItemsArr;
    }

    private function getPriceAfterTierDiscount($tierPriceRule, $price, $quantity)
    {
        $newPrice = $price;
        if (!empty($tierPriceRule)) {
            $upperLimits = max(array_column($tierPriceRule, 'upper_limit'));
            if ($quantity > $upperLimits) {
                $quantity = $upperLimits;
            }
            foreach ($tierPriceRule as $tier) {
                if ($quantity >= $tier['lower_limit'] && $quantity <= $tier['upper_limit']) {
                    $newPrice = ($tier['discountType'] == "flat" ? ($price - $tier['discount']) : ($price - (($tier['discount'] / 100) * $price)));
                    break;
                }
            }
        }
        return $newPrice;
    }

    /**
     * GET custom preview images of cart items
     * 
     * @author inkXE
     * @date 20 Dec 2018
     * @param $refids design reference ID
     * @return Cart link
     */
    public function getDesignPreviewImages($variantID, $pid, $refids = 0, $source = "carts")
    {
        try {
            if ($refids) {
                $regidArr = explode(',', $refids);
                $jsonData = '';
                $desStateDIR = path('abs', 'design_state') . $source;
                $baseImagePath = $desStateDIR . SEPARATOR;
                foreach ($regidArr as $values) {
                    $stateDesignFile = $baseImagePath . $values . '.json';
                    $jsonData = json_decode(file_get_contents($stateDesignFile), true);
                    if ($jsonData != '') {
                        foreach ($jsonData['design_product_data'] as $capture) {
                            if ($capture['variant_id'][0] == $variantID || $pid == $variantID) {
                                $images = $capture['design_urls'];
                                break;
                            }
                        }
                    } else {
                        $msg = array("status" => "nodata");
                        return json_encode($msg);
                    }
                }
                return $images;
            }
        } catch (Exception $e) {
            $result = array('Caught exception:' => $e->getMessage());
            return json_encode($result);
        }
    }
    public function shopDomain()
    {
        return $this->call('GET', '/admin/shop.json?fields=domain');
    }


    /**
     * Shopify GraphQl API curl function
     *
     * @param $method  GraphQl API call method (POST)
     * @param $payload  Shopify API parameters if any
     *
     * @author dan@imprintnext.com
     * @date   28-10-2022
     * @return Curlresponse
     */
    private function graphQlRequest($method, $payload)
    {

        $apiUrl = "https://{$this->shopDomain}/" . "admin/api/" . $this->apiVersion . "/graphql.json";
        $authToken = base64_encode($this->apiKey . ":" . $this->token);
        $requestHeaders = [
            'Content-Type: application/json; charset=utf-8',
            'Authorization: Bearer ' . $authToken,
            'X-Shopify-Access-Token: ' . $this->token
        ];
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $apiUrl);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_USERAGENT, 'ohShopify-php-api-client');
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 90);
        curl_setopt($ch, CURLOPT_TIMEOUT, 500);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if (!empty($requestHeaders)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $requestHeaders);
        }
        if ($method != 'GET' && !empty($payload)) {
            if (is_array($payload)) {
                $payload = http_build_query($payload);
            }
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        }
        $response = curl_exec($ch);
        $errno = curl_errno($ch);
        $error = curl_error($ch);
        curl_close($ch);
        if ($errno) {
            create_log('clipart', 'error', [
                'message' => 'Shopify GraphQl Request Error!',
                'extra' => ['errno' => $errno, 'error' => $error]
            ]);
            return false;
        }
        $respData = preg_split("/\r\n\r\n|\n\n|\r\r/", $response, 2);
        return  json_clean_decode($respData[1], true);
    }

    /**
     * Copy old product delivery profile to new product.
     *
     * @param $newProductId
     *
     * @author dan@imprintnext.com
     * @date   28-10-2022
     * @return Integer : Delivery update count
     */
    public function copyDeliveryProfile($varIDs)
    {
        // get the delivery profile of the product variants
        $payload = '{"query":"{productVariant(id: \"gid:\/\/shopify\/ProductVariant\/' .
            $varIDs['oldVarID'] . '\") {deliveryProfile {id}}}"}';

        $result =  $this->graphQlRequest('POST', $payload);

        // update the delivery profile for the new product crated
        if (
            isset($result['data'], $result['data']['productVariant']) &&
            !empty($result['data']['productVariant']['deliveryProfile'])
        ) {
            $deliveryProfileId = $result['data']['productVariant']['deliveryProfile']['id'];

            // update delivery profile using the graphQl
            $payload = '{"query":"mutation deliveryProfileUpdate {deliveryProfileUpdate(id:\"' .
                $deliveryProfileId . '\", profile: {variantsToAssociate:[\"gid:\/\/shopify\/ProductVariant\/' .
                $varIDs['newVarID'] . '\"]}) {profile {id} userErrors {field message}}}"}';

            $result =  $this->graphQlRequest('POST', $payload);
        }

        return $result;
    }

    public function addVariantToShopProd($prodData)
    {
        $productData = $this->getShopifyProductInfo($prodData['productID']);
        $variantOptionExists = true;
        $requestedVarID = "";
        $productArray = array(
            "product" => array(
                "status" => "active",
                "tags" => $productData['tags'] . ", imprint_options",
            ),
        );
        if (count($productData['options']) > 1 || $productData['options'][0]['name'] != "variant") {
            $variantOptionExists = false;
            $productArray['product']['options'] = array(
                'name' => "variant",
                'values' => !empty($productData['options'][0]['values']) ? array_push($optValues, $prodData['variantName']) : array($prodData['variantName']),
                'position' => 1
            );
            // foreach ($productData['variants'] as $variant) {
            //     $this->call('DELETE', '/admin/products/'. $prodData['productID'] .'/variants/'.$variant['id'].'.json');
            // }
        }
        $oldTags = explode(', ', $productData['tags']);
        if (!$variantOptionExists || !in_array("imprint_options", $oldTags) || empty($productData['published_at'])) {

            $updateProd = $this->call('PUT', '/admin/products/' . $prodData['productID'] . '.json', $productArray);
        }
        foreach ($productData['variants'] as $variant) {
            if ($variant['option1'] == $prodData['variantName']) {
                $requestedVarID = $variant['id'];
                break;
            }
        }

        if (empty($requestedVarID)) {
            $variantArray = array(
                "variant" => array(
                    "product_id" => $prodData['productID'],
                    "option1" => $prodData['variantName'],
                    "price" => $prodData['variantPrice']
                ),
            );
            $variant = $this->call('POST', '/admin/products/' . $prodData['productID'] . '/variants.json', $variantArray);
            $requestedVarID = $variant['id'];
        }
        // update inventory for this variant
        if ($variant['inventory_quantity'] != $prodData['inventory'] && $variant['inventory_management'] == 'shopify') {
            $inventoryItemID = $variant['inventory_item_id'];
            $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventoryItemID);
            foreach ($invLevels as $level) {
                $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
                if (!$locationData['legacy']) {
                    $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" => $inventoryItemID, "available" => $prodData['inventory']);
                    $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
                }
            }
        }
        return $requestedVarID;
    }

    public function tierPriceOfDuplicateProduct($price, $itemDetails)
    {
        $jsonResponse = [
            'status' => 0,
            'message' => "Product tierPrice couldn't updated in store.",
        ];
        $success = 0;

        foreach ($itemDetails as $key => $item) {
           // $variantDetails = $this->call('GET', '/admin/variants/' . $item['variant_id'] . '.json');

            $updateVariantPrice = [
                "variant" => [
                    "id" => $item['variant_id'],
                    "price" => $price[$key]
                ]
            ];
            $this->call('PUT', '/admin/variants/' . $item['variant_id'] . '.json', $updateVariantPrice);
            $success++;
        }
        if ($success > 0) {
            $jsonResponse = [
                'status' => 1,
                'message' => "Product tierPrice updated in store.",
            ];
        }
        return $jsonResponse;
    }

    /*     * ***************************** Cart Module End ***************************** */
    public function shopifyVariantManage($productArray,$product,$imgArr)
    {
        
        $price = $productArray['product']['variants'][0]['price'];
        $productArray['product']['options'] = [];
        $productArray['product']['variants'] = [];
        $newProduct = $this->call('POST', '/admin/products.json', $productArray);
        // $newProduct['id'] = 9165174767908;
        $variantData = [
            "variant" => [
                "id" => $newProduct['variants'][0]['id'],
                "price" => $price
            ]
        ];

        $this->call('PUT', '/admin/variants/' . $newProduct['variants'][0]['id'] . '.json', $variantData);
        $inventoryId = $newProduct['variants'][0]['inventory_item_id'];
        $invLevels = $this->call('GET', '/admin/inventory_levels.json?inventory_item_ids=' . $inventoryId);
        $tracked = [
            "inventory_item" => [
                "id" => $inventoryId,
                "tracked" => "true"
            ]
        ];
        $this->call('PUT', '/admin/inventory_items/' . $inventoryId . '.json', $tracked);
        foreach ($invLevels as $level) {
            $locationData = $this->call('GET', '/admin/locations/' . $level['location_id'] . '.json');
            if (!$locationData['legacy']) {
                $newInventory = array("location_id" => $level['location_id'], "inventory_item_id" => $inventoryId, "available" => 100);
                $this->call('POST', '/admin/inventory_levels/set.json', $newInventory);
            }
        }
        $customArr = array(
            "collect" => array(
                "product_id" => $newProduct['id'],
                "collection_id" => $this->getShowInDesignerId(),
            ),
        );
        $this->call('POST', '/admin/collects.json', $customArr);
        $newImg['image'] = array("src" => $imgArr[0]['src']);
        $this->call('POST', '/admin/products/' . $newProduct['id'] . '/images.json', $newImg);
        $this->saveVariantToDB($product, $newProduct['id']);
        $newProductIds[] = $newProduct['id'];
        return $newProductIds;
    }

    private function saveVariantToDB($productInfo,$productId){
        $allAttributes = $productInfo['attributes'];
       
        $getAttr = array_column($allAttributes, 'name');
        foreach($allAttributes as $key => $singleAtt){
            $productOptionObj = new ProductOptions();
			$getOptionObj = $productOptionObj->where(['name' => $singleAtt['name']]);
            if ($getOptionObj->count()) {
				$productOptionObj = $getOptionObj->get()->first();
            } else {
				$productOptionObj->fill($singleAtt);
				// skip if unable to save
				if (!$productOptionObj->save()) {
					continue;
				}
			}
            if(!empty($singleAtt['options'])){
                
                $optionRelData = [
					'product_id' => strval($productId),
					'option_values' => $singleAtt['options'],
					'position' => $key
				];
               
				$optionRelObj = new ProductOptionRel();
				$checkOptionRel = $optionRelObj->where(['option_id' => $productOptionObj->xe_id, 'product_id' => $productId]);
				if ($checkOptionRel->count()) {
					$optionRelObj = $checkOptionRel->get()->first();
				} else {
					$optionRelData['option_id'] = $productOptionObj->xe_id;
				}
				$optionRelObj->fill($optionRelData);
				$optionRelObj->save();
            }
        }
        $getAllVariants = $productInfo['variations'];
        $uniqueVar = [];
        foreach ($getAllVariants as $singleVar) {
            $option1 = strtolower($getAttr[0]);
            $option2 = strtolower($getAttr[1]);
            $imagename = $singleVar['attributes'][$option1];
            $addedImage = [
                'name' => $imagename,
                'is_disable' => 0,
                'store_id' => 1,
                'cloud_storage' => 0,
                'is_catalog' => 1
            ];
            $getImageObj = new ProductImage($addedImage);
            $getImageObj->save();
            foreach ($singleVar['image_path'] as $imgKey => $imgValue) {
                $addedSideImage = [
                    'product_image_id' => $getImageObj->xe_id,
                    'side_name' => $singleVar['side_name'][$imgKey],
                    'sort_order' => $imgKey,
                    'file_name' => $imgValue
                ];
                $newSideImage = new ProductImageSides($addedSideImage);
                $newSideImage->save();
            }
            $title = str_replace('/', '', $singleVar['attributes'][$option1]) . '/' . $singleVar['attributes'][$option2];
            $key = $singleVar['attributes'][$option1] . '/' . $singleVar['attributes'][$option2];
            if (!in_array($key, $uniqueVar)) {
                $uniqueVar[] = $key;
                $addedVariants = [
                    'title' => $title,
                    'price' => $singleVar['piece_price'],
                    'sku' => $singleVar['sku'],
                    'image_id' => $getImageObj->xe_id,
                    'inventory' => $singleVar['quantity'],
                    'barcode' => $singleVar['gtin'],
                    'weight' => $singleVar['unit_weight'],
                ];
                $newProductVar = new ProductVariants($addedVariants);
                $newProductVar->save();
                $varRel = [
                    'product_id' => strval($productId),
                    'variant_id' => $newProductVar->xe_id
                ];
                $newProductVarRel = new ProductVariantRel($varRel);
                $newProductVarRel->save();
            }
            
        }
    }
}
