<?php
/**
 * Manage Background Things to change
 *
 * PHP version 5.6
 *
 * @category  Backgrounds
 * @package   Assets
 * @author    Satyabrata <satyabratap@riaxex.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */

namespace App\Modules\Backgrounds\Controllers;

use App\Components\Controllers\Component as ParentController;
use App\Modules\Backgrounds\Models\Background;
use App\Modules\Backgrounds\Models\BackgroundCategory as Category;
use App\Modules\Backgrounds\Models\BackgroundCategoryRelation;
use App\Modules\Backgrounds\Models\BackgroundTagRelation;

/**
 * Backgrounds Controller
 *
 * @category Backgrounds
 * @package  Assets
 * @author   Satyabrata <satyabratap@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */

class BackgroundController extends ParentController
{
    /**
     * POST: Save Background
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is saved or any error occured
     */
    public function saveBackgrounds($request, $response)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Backgrounds', 'error')
        ];
        try {

            // Get Store Specific Details from helper
            $getStoreDetails = get_store_details($request);

            $allPostPutVars = $request->getParsedBody();
            $allFileNames = $saveBackgroundList = $recordBackgroundIds = [];
            $uploadingFilesNo = $success = 0;
            $err = true; // for validation of fields

            $isS3Enabled = $this->checkS3Settings($getStoreDetails['store_id']);

            if (empty($allPostPutVars['name'])) {
                $jsonResponse['message'] = 'Name should not be empty!';
            } elseif ($allPostPutVars['type'] == 1 && !$request->getUploadedFiles()) {
                $jsonResponse['message'] = 'File Required!';
            } elseif ($allPostPutVars['type'] == 0 && empty($allPostPutVars['upload'])) {
                $jsonResponse['message'] = 'Color code Required!';
            } elseif (empty($allPostPutVars['categories'])) {
                $jsonResponse['message'] = "Please select category";
            } else {
                $err = false;
            }

            if (!$err) {
                // type = 1 : with file as background and check if any file received
                if ($allPostPutVars['type'] == 1) {
                    // Save file
                    $allFileNames = do_upload('upload', path('abs', 'background'), [150], 'array');
                    if (empty($allFileNames)) {
                        throw new \Exception('File upload error');
                    }
                    if (is_string($allFileNames) && $allFileNames == "limit_error") {
                        $jsonResponse = [
                            'status' => 1,
                            'message' => "The upload limit is exceeded"
                        ];
                    } else {
                        // remove blank value : skipped emptycheck in the loop
                        $allFileNames = array_filter($allFileNames);
                        // number of uploaded files
                        $uploadingFilesNo = count($allFileNames);

                        foreach ($allFileNames as $eachFile) {
                            $saveBackgroundList = [
                                    'store_id' => $getStoreDetails['store_id'],
                                    'name' => $allPostPutVars['name'],
                                    'price' => $allPostPutVars['price'],
                                    'type' => $allPostPutVars['type'],
                                    'value' => $eachFile,
                                    'cloud_storage' => 0
                                ];
                            if ($isS3Enabled) {
                                $thumbnailFile = "thumb_" . $eachFile;
                                $backgroundFile = path('abs', 'background') . $eachFile;
                                $thumbUpload =  path('abs', 'background') . $thumbnailFile;
                                $this->uploadFileToS3("background", $thumbUpload, $getStoreDetails['store_id']);
                                $this->uploadFileToS3("background", $backgroundFile, $getStoreDetails['store_id']);
                                $saveBackgroundList['cloud_storage'] = 1;
                            }
                            $saveEachBackground = new Background($saveBackgroundList);
                            if ($saveEachBackground->save()) {
                                $backgroundLastId = $saveEachBackground->xe_id;
                                $recordBackgroundIds[] = $backgroundLastId;
                                /**
                                 * Save category and subcategory data
                                 * Category id format. [4,78,3]
                                 */
                                if (!empty($allPostPutVars['categories'])) {
                                    $categoryIds = $allPostPutVars['categories'];
                                    // Save Background Categories
                                    $this->saveBackgroundCategories($backgroundLastId, $categoryIds);
                                }
                                /**
                                 * - Save tags with respect to the Backgrounds
                                 * - Tag Names format.: tag1,tag2,tag3
                                 */
                                if (!empty($allPostPutVars['tags'])) {
                                    $tags = $allPostPutVars['tags'];
                                    $this->saveBgTags(
                                        $getStoreDetails['store_id'],
                                        $backgroundLastId,
                                        $tags
                                    );
                                }
                                $success++;
                            }
                        }
                    }
                } else {
                    $allPostPutVars += [
                        'value' => $allPostPutVars['upload'],
                        'store_id' => $getStoreDetails['store_id']
                    ];
                    $saveBackgroundColor = new Background($allPostPutVars);
                    if ($saveBackgroundColor->save()) {
                        $backgroundLastId = $saveBackgroundColor->xe_id;
                        /**
                         * Save category and subcategory
                         * Category id format: [4,78,3]
                         */
                        if (!empty($allPostPutVars['categories'])) {
                            $categoryIds = $allPostPutVars['categories'];
                            // Save Background Categories
                            $this->saveBackgroundCategories($backgroundLastId, $categoryIds);
                        }
                        /**
                         * - Save tags
                         * - Tag Names format: tag1,tag2,tag3
                         */
                        if (
                            isset($allPostPutVars['tags'])
                            && $allPostPutVars['tags'] != ""
                        ) {
                            $tags = $allPostPutVars['tags'];
                            $this->saveBgTags(
                                $getStoreDetails['store_id'],
                                $backgroundLastId,
                                $tags
                            );
                        }
                        $success++;
                        $uploadingFilesNo = 1;
                    }
                }
            }
        
            if (!empty($success)) {
                $jsonResponse = [
                    'status' => 1,
                    'message' => $success . ' out of ' . $uploadingFilesNo
                        . ' Background(s) uploaded successfully'
                ];
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Backgrounds'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * Save Categories/Sub-categories w.r.t Background
     *
     * @param $backgroundId Background ID
     * @param $categoryIds  (in  an array with comma separated)
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return boolean
     */
    protected function saveBackgroundCategories($backgroundId, $categoryIds)
    {
        $getAllCategoryArr = json_clean_decode($categoryIds, true);
        foreach($getAllCategoryArr as $bagCatId){
            $newCategory = new Category();
			$categoriesData = $newCategory->where('xe_id', '=', $bagCatId)->get()->toArray();
            if($bagCatId == $categoriesData[0]['xe_id']){
				$flagList = [ 'asset_available_flag' => 1];
                $newCategory->where('xe_id', '=', $bagCatId)->update($flagList);
			}
        }
        //SYNC Categories to the Background_Category Relationship Table
        $backgroundInit = new Background();
        $findBackground = $backgroundInit->find($backgroundId);
        if ($findBackground->categories()->sync($getAllCategoryArr)) {
            return true;
        }
        return false;
    }

    /**
     * Save tags w.r.t Background
     *
     * @param $storeId      Store Id
     * @param $backgroundId Background ID
     * @param $tags         Multiple Tags
     *
     * @author satyabratap@riaxe.com
     * @author tanmayap@riaxe.com
     * @date   4th Nov 2019
     * @return boolean
     */
    protected function saveBgTags($storeId, $backgroundId, $tags)
    {
        // Save Backgrounds and tags relation
        if (!empty($tags)) {
            $getTagIds = $this->saveTags($storeId, $tags);
            // SYNC Tags into Backgrounds Tag Relationship Table
            $backgroundInit = new Background();
            $findBackground = $backgroundInit->find($backgroundId);
            if ($findBackground->tags()->sync($getTagIds)) {
                return true;
            }
        } else {
            // Clean relation in case no tags supplied
            $tagRelInit = new BackgroundTagRelation();
            $backgroundTags = $tagRelInit->where('background_id', $backgroundId);
            if ($backgroundTags->delete()) {
                return true;
            }
        }

        return false;
    }

    /**
     * GET: Single Background
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author dan@imprintnext.com
     * @date   19 July 2022
     * @return Single Background
     */
    public function getSingleBackground($request, $response, $args){
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 1,
            'data' => [],
            'message' => message('Background', 'not_found')
        ];

        $getStoreDetails = get_store_details($request);
        $backgroundId = to_int($args['id']);

       
        if (!empty($backgroundId)) {
            $backgroundInit = new Background();
            // get single background matching with the id
            $getBackgrounds = $backgroundInit->where([
                ['xe_id', '>', 0],
                ['store_id', '=', $getStoreDetails['store_id']],
                ['xe_id', '=', $backgroundId]
            ])->select([
                'xe_id', 'name', 'value', 'price', 'type', 'cloud_storage'
            ]);

            // remove the unused appends (category_names) and get the data
            $backgroundData = $getBackgrounds->first()->setAppends(['thumbnail'])->toArray();

            // get the file / thumbnail
            if($backgroundData['cloud_storage'] == 1){
                $fileName =  $this->getS3URL($backgroundData['value'], $getStoreDetails['store_id']);
                $thumbnail = $this->getS3URL($backgroundData['thumbnail'], $getStoreDetails['store_id']);    
                $backgroundData['value'] = $fileName;
                $backgroundData['thumbnail'] = $thumbnail;
            }

            // Get Category Ids
            if (!empty($backgroundData)) {
                $getCategories = $this->getCategoriesById(
                    'Backgrounds',
                    'BackgroundCategoryRelation',
                    'background_id',
                    $backgroundId
                );
                $getTags = $this->getTagsById(
                    'Backgrounds',
                    'BackgroundTagRelation',
                    'background_id',
                    $backgroundId
                );
                $backgroundData['categories'] = !empty($getCategories)
                    ? $getCategories : [];
                $backgroundData['tags'] = !empty($getTags) ? $getTags : [];
            }

            $jsonResponse = [
                'status' => 1, 
                'records' => 1, 
                'data' => [$backgroundData]
            ];
        }
        return response(
            $response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * GET: List of Backgrounds
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return All Background List
     */
    public function getBackgrounds($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 1,
            'data' => [],
            'message' => message('Background', 'not_found')
        ];

        $offset = 0;
        $getStoreDetails = get_store_details($request);

        $backgroundInit = new Background();
        $getBackgrounds = $backgroundInit->where([
            ['xe_id', '>', 0],
            ['store_id', '=', $getStoreDetails['store_id']]
        ]);

        // Total records irrespectable of filters
        $totalCounts = $getBackgrounds->count();
        if ($totalCounts > 0) {
            // Collect all Filter columns from url
            $page = $request->getQueryParam('page');
            $perpage = $request->getQueryParam('perpage');
            $categoryId = !empty($request->getQueryParam('category')) ?
            json_clean_decode($request->getQueryParam('category'), true) : [];
            $sortBy = $request->getQueryParam('sortby');
            $order = $request->getQueryParam('order');
            $name = $request->getQueryParam('name');
            $type = $request->getQueryParam('type');
            $printProfileKey = $request->getQueryParam('print_profile_id');
            $page = ($request->getQueryParam('page') != '') ? $request->getQueryParam('page') : 1;
			$perpage = ($request->getQueryParam('perpage') != '') ? $request->getQueryParam('perpage') : 10;

            // For multiple Background data
            $getBackgrounds->select([
                'xe_id', 'name', 'value', 'price', 'type', 'cloud_storage'
            ]);

            // Filter Search as per type
            if (isset($type) && $type != "") {
                $getBackgrounds->where(function ($query) use ($type) {
                    $query->where('type', $type);
                });
            }

            // Multiple Table search for name attribute
            if (isset($name) && $name != "") {
                $name = '\\' . $name;
                $getBackgrounds->where(function ($query) use ($name) {
                    $query->where('name', 'LIKE', '%' . $name . '%')
                        ->orWhereHas('backgroundTags.tag', function ($q) use ($name) {
                            return $q->where('name', 'LIKE', '%' . $name . '%');
                        })
                        ->orWhereHas('backgroundCategory.category', function ($q) use ($name) {
                            return $q->where('name', 'LIKE', '%' . $name . '%');
                        });
                });
            }

            // Filter By Print Profile's category Id
            if (!empty($printProfileKey)) {
                $profileCatRelObj = new \App\Modules\PrintProfiles\Models\PrintProfileAssetsCategoryRel();
                $assetTypeArr = $this->assetsTypeId('backgrounds');
                $profileCatRelDetails = $profileCatRelObj->where([
                    ['asset_type_id', '=', $assetTypeArr['asset_type_id']],
                    ['print_profile_id', '=', $printProfileKey]
                ]);
                // get the print profile category ids
                $profileCatIds = $profileCatRelDetails->get()->pluck('category_id')->toArray();
                if ($profileCatIds) {
                    $categoryId = array_merge($categoryId, $profileCatIds);
                }
            }

            // Filter by Category ID
            if (!empty($categoryId)) {
                $getBackgrounds->whereHas('backgroundCategory', function ($q) use ($categoryId) {
                    return $q->whereIn('category_id', $categoryId);
                });
            }

            // Total records including all filters
            $getTotalPerFilters = $getBackgrounds->count();

            // Pagination is not implemented for configurator tool section.
            if (!isset($args['category'])) {
                // Get pagination data
                if (isset($page) && $page != "") {
                    $totalItem = empty($perpage) ? PAGINATION_MAX_ROW : $perpage;
                    $offset = $totalItem * ($page - 1);
                    $getBackgrounds->skip($offset)->take($totalItem);
                }
            }
            // Sorting by column name and sord order parameter
            if (!empty($sortBy) && !empty($order)) {
                $getBackgrounds->orderBy($sortBy, $order);
            }

            $backgroundData = $getBackgrounds->get()->toArray();
            foreach ($backgroundData as $key => $value) {
                if ($value['cloud_storage'] == 1) {
                    $thisFileName = $value['value'];
                    $thisThumbnail = $value['thumbnail'];
                    $backgroundData[$key]['value'] =  $this->getS3URL($thisFileName, $getStoreDetails['store_id']);
                    $backgroundData[$key]['thumbnail'] = $this->getS3URL($thisThumbnail, $getStoreDetails['store_id']);
                }
            }

            $jsonResponse = [
                'status' => 1,
                'records' => count($backgroundData),
                'total_records' => $getTotalPerFilters,
                'data' => $backgroundData
            ];
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * PUT: Update a single background
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is updated or not
     */
    public function updateBackground($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Background', 'error'),
        ];
        try {
            $getStoreDetails = get_store_details($request);
            $storeId = $getStoreDetails['store_id'];
            $allPostPutVars = $updateData = $request->getParsedBody();
            $backgroundId = to_int($args['id']);
            $err = true; // validation flag for fields
            $backgroundInit = new Background();
            if (!empty($backgroundId)) {
                $getOldBG = $backgroundInit->where('xe_id', $backgroundId)->get()->first()->toArray();
            }

            if (empty($getOldBG)) {
                $jsonResponse['message'] = 'Background not found!';
            } elseif (empty($allPostPutVars['name'])) {
                $jsonResponse['message'] = 'Name should not be empty!';
            } elseif (empty($allPostPutVars['categories'])) {
                $jsonResponse['message'] = "Please select category";
            } else {
                $err = false;
            }

            if (!$err) {
                $oldCategoryArr = explode(",", $getOldBG['category_id']);
                $newCategory = str_replace(array( '\'', '"','[' , ']', '<', '>' ), ' ', $allPostPutVars['categories']);
				$newCategoryArr = explode(",", $newCategory);
                $result = array_diff($oldCategoryArr, $newCategoryArr);
                foreach ($result as $res) {
					$backGroundCount = new BackgroundCategoryRelation();
					$countData = $backGroundCount->where('category_id', '=', $res)->get()->toArray();
                    $newCategory = new Category();
					if (count($countData) == 1) {
						$flagList = [ 'asset_available_flag' => 0];
						$newCategory->where('xe_id', '=', $res)->update($flagList);
						
					} else {
						$flagList = [ 'asset_available_flag' => 1];
						$newCategory->where('xe_id', '=', $res)->update($flagList);
					}
				}

                if (!empty($getOldBG)) {
                    unset(
                        $updateData['id'],
                        $updateData['tags'],
                        $updateData['categories'],
                        $updateData['upload'],
                        $updateData['value'],
                        $updateData['backgroundId']
                    );

                    if (isset($updateData['type']) && $updateData['type'] == 0 && !empty($allPostPutVars['upload'])) {
                        $updateData['value'] = $allPostPutVars['upload'];
                    } else {
                        if ($request->getUploadedFiles()) {
                            $getUploadedFileName = do_upload(
                                'upload',
                                path('abs', 'background'),
                                [100],
                                'string'
                            );
                            if ($getOldBG['cloud_storage'] == 1) {
                                $updateData['cloud_storage'] = 1;
                                $thumbUpload = path('abs', 'background') . "thumb_" . $getUploadedFileName;
                                $fileToUpload = path('abs', 'background') . $getUploadedFileName;
                                $this->uploadFileToS3("background", $fileToUpload, $storeId);
                                $this->uploadFileToS3("background", $thumbUpload, $storeId);
                            }
                            $updateData['value'] = $getUploadedFileName;
                        }
                    }
                    if (
                        $getOldBG['type'] == 1 &&
                        ((isset($updateData['type']) && $updateData['type'] == 0) || !empty($getUploadedFileName))
                    ) {
                        if ($getOldBG['cloud_starage'] == 1) {
                            $this->deleteS3File($getOldBG['value'], $storeId);
                            $this->deleteS3File($getOldBG['thumbnail'], $storeId);
                        } else {
                            // Delete old file
                            $this->deleteOldFile(
                                "backgrounds",
                                "value",
                                ['xe_id' => $backgroundId],
                                path('abs', 'background'),
                                false
                            );
                        }
                    }
                    $updateData['store_id'] = $storeId;

                    // Update record
                    $backgroundInit->where('xe_id', '=', $backgroundId)->update($updateData);

                    // Save category
                    if (isset($allPostPutVars['categories']) && $allPostPutVars['categories'] != "") {
                        $categoryIds = $allPostPutVars['categories'];
                        $this->saveBackgroundCategories($backgroundId, $categoryIds);
                    }

                    // Save tags
                    $tags = !empty($allPostPutVars['tags']) ? $allPostPutVars['tags'] : "";
                    $this->saveBgTags($storeId, $backgroundId, $tags);

                    $jsonResponse = [
                        'status' => 1,
                        'message' => message('Background', 'updated')
                    ];
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Backgrounds'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * DELETE: Delete single/multiple background
     *
     * @param $request  Slim's Argument parameters
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   4th Nov 2019
     * @return json response wheather data is deleted or not
     */
    public function deleteBackground($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Background', 'error'),
        ];
        try {
            // get the store id
            $getStoreDetails = get_store_details($request);
            $storeId = $getStoreDetails['store_id'];
            // deletable ids
            $getDeleteIdsToArray = !empty($args['id']) ? json_clean_decode($args['id'], true) : [];

            if ($getDeleteIdsToArray) {
                // total number of requested for delete
                $totalCount = count($getDeleteIdsToArray);
                // get the backgrounds
                $backgroundInit = new Background();
                $backgrounds = $backgroundInit->whereIn('xe_id', $getDeleteIdsToArray);
                // go through and delete files + background
                if ($backgrounds->count() > 0) {
                    $getBackgroundDetails = $backgrounds->get()->toArray();
                    $success = 0; // total success delete
                    foreach ($getBackgroundDetails as $bg) {
                        $categoryIdArr = explode(",",$bg['category_id']);
                        foreach($categoryIdArr as $bgCatId){
                            $bgCount = new BackgroundCategoryRelation();
							$countData = $bgCount->where('category_id', '=', $bgCatId)->get()->toArray();
                            $newCategory = new Category();
							if(count($countData) == 1){
								$flagList = [ 'asset_available_flag' => 0];
								$newCategory->where('xe_id', '=', $bgCatId)->update($flagList);
							}else{
								$flagList = [ 'asset_available_flag' => 1];
								$newCategory->where('xe_id', '=', $bgCatId)->update($flagList);
                            }
						}
                        if ($bg['type'] == 1) {
                            if ($bg['cloud_storage'] == 1) {
                                $fileName = $bg['value'];
                                $thumbnail = $bg['thumbnail'];
                                $this->deleteS3File($fileName, $storeId);
                                $this->deleteS3File($thumbnail, $storeId);
                            } else {
                                $this->deleteOldFile(
                                    "backgrounds",
                                    "value",
                                    ['xe_id' => $bg['xe_id']],
                                    path('abs', 'background'),
                                    false
                                );
                            }
                        }
                        $bgCount->where('background_id', $bg['xe_id'])->delete();
                        $backgroundInit->where('xe_id', $bg['xe_id'])->delete();
                        $success++;
                    }
                    if ($success > 0) {
                        $jsonResponse = [
                            'status' => 1,
                            'message' => $success . ' out of ' . $totalCount
                                . ' Background(s) deleted successfully',
                        ];
                    }
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Backgrounds'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * Delete a category from the table
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   20 Jan 2020 Satyabrata
     * @return Delete Json Status
     */
    public function deleteCategory($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $getStoreDetails = get_store_details($request);
        $storeId = $getStoreDetails['store_id'];
        $jsonResponse = [
            'status' => 0,
            'message' => message('Category', 'error')
        ];
        if (!empty($args['id'])) {
            $categoryId = $args['id'];
            $jsonResponse = $this->deleteCat(
                $storeId,'backgrounds', $categoryId,
                'Backgrounds', 'BackgroundCategoryRelation'
            );
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }
}
