<?php
/**
 * Manage Graphic Fonts
 *
 * PHP version 5.6
 *
 * @category  Graphic_Font
 * @package   Eloquent
 * @author    Satyabrata <satyabratap@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */

namespace App\Modules\GraphicFonts\Controllers;

use App\Components\Controllers\Component as ParentController;
use App\Modules\GraphicFonts\Models\GraphicFont;
use App\Modules\GraphicFonts\Models\GraphicFontLetter;
use App\Modules\GraphicFonts\Models\GraphicFontTagRelation;

/**
 * Graphic Font Controller
 *
 * @category Class
 * @package  Graphic_Fonts
 * @author   Satyabrata <satyabratap@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */
class GraphicFontController extends ParentController
{

    /**
     * Html dom object
     */
    public $domObj;

    /**
     * Initialize Constructor
     */
    public function __construct()
    {
        $this->domHtmlPathInclue();
    }
    
    /**
     * POST: Save Graphic Fonts
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return json response wheather data is saved or any error occured
     */
    public function saveGraphicFonts($request, $response)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Graphic Fonts', 'error'),
        ];

        try {
            // Get Store Specific Details from helper
            $getStoreDetails = get_store_details($request);
            $allPostPutVars = $request->getParsedBody();
            $graphicFontsJson = $allPostPutVars['data'];
            $graphicFontsArray = json_clean_decode($graphicFontsJson, true);

            if (!empty($graphicFontsArray['name'])) {
                $nameCheck = (new GraphicFont())->where(['name' => $graphicFontsArray['name']])
                ->count();
                // create new font if not existed
                if ($nameCheck == 0) {
                    $isS3Enabled = $this->checkS3Settings($getStoreDetails['store_id']);
                    $fontData = [
                        'name' => $graphicFontsArray['name'],
                        'price' => $graphicFontsArray['price'],
                        'store_id' => $getStoreDetails['store_id'],
                        'cloud_storage' => ($isS3Enabled ? 1 : 0),
                        'is_letter_style' => $graphicFontsArray['is_letter_style'],
                        'is_number_style' => $graphicFontsArray['is_number_style'],
                        'is_special_character_style' => $graphicFontsArray['is_special_character_style']
                    ];

                    // save the graphic font
                    $saveGraphicFont = new GraphicFont($fontData);
                    $saveGraphicFont->save();
                    $graphicFontInsertId = $saveGraphicFont->xe_id;

                    /**
                     * Save tags
                     * Tag Names format : tag1,tag2,tag3
                     */
                    $tags = !empty($graphicFontsArray['tags']) ? $graphicFontsArray['tags'] : "";
                    $this->saveGraphicFontTags(
                        $getStoreDetails['store_id'],
                        $graphicFontInsertId,
                        $tags
                    );

                    if (!empty($graphicFontsArray['letter_style'])) {
                        $this->_saveFontDesigns(
                            $graphicFontsArray['letter_style'],
                            $graphicFontInsertId,
                            'letter',
                            $getStoreDetails['store_id']
                        );
                    }
                    if (!empty($graphicFontsArray['number_style'])) {
                        $this->_saveFontDesigns(
                            $graphicFontsArray['number_style'],
                            $graphicFontInsertId,
                            'number',
                            $getStoreDetails['store_id']
                        );
                    }
                    if (!empty($graphicFontsArray['special_character_style'])) {
                        $this->_saveFontDesigns(
                            $graphicFontsArray['special_character_style'],
                            $graphicFontInsertId,
                            'special_character',
                            $getStoreDetails['store_id']
                        );
                    }
                    $jsonResponse = [
                        'status' => 1,
                        'message' => message('Graphic Font', 'saved')
                    ];
                } else {
                    $jsonResponse['message'] = message('Graphic Fonts', 'exist');
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Graphic Fonts'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * Save Graphic-Font-Designs
     *
     * @param $fontdetails Font Details Array
     * @param $fontId      Graphic Font Id
     * @param $slug        Type of graphic font
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return boolean
     */
    private function _saveFontDesigns($fontdetails, $fontId, $slug, $getStoreDetails)
    {
        if (!empty($fontdetails)) {
            foreach ($fontdetails as $fontData) {
                $getUploadedFileName = do_upload(
                    $fontData['image_upload_data'],
                    path('abs', 'graphicfont'),
                    [],
                    'string'
                );
                $isS3Enabled = $this->checkS3Settings($getStoreDetails);
                if ($isS3Enabled){
                    $fileToUpload = path('abs', 'graphicfont') . $getUploadedFileName;
                    $this->uploadFileToS3("graphicfont", $fileToUpload, $getStoreDetails);
                }
                $designData = [
                    'graphic_font_id' => $fontId,
                    'name' => $fontData['name'],
                    'font_type' => $slug,
                ];
                if (!empty($getUploadedFileName)) {
                    $designData['file_name'] = $getUploadedFileName;
                }
                $saveGraphicFonts = new GraphicFontLetter($designData);
                $saveGraphicFonts->save();
                
                //Update SVG file height and width
                $imageUrl = path('abs', 'graphicfont').$getUploadedFileName;
                if (file_exists($imageUrl)) {
                    $ext = strtolower(pathinfo($imageUrl, PATHINFO_EXTENSION));
                    if ($ext =='svg') {
                        $this->updateSvgHeightWidth($imageUrl);
                    }
                }
            }
        }
        return false;
    }

    /**
     * Save Tags and Graphic-Font-Tag Relations
     *
     * @param $storeId      Store ID
     * @param $fontInsertId Graphic Font ID
     * @param $tags         Multiple Tags
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return boolean
     */
    protected function saveGraphicFontTags($storeId, $fontInsertId, $tags)
    {
        // Save Graphic Font and tags relation
        if (!empty($tags)) {
            $getTagIds = $this->saveTags($storeId, $tags);
            // SYNC Tags into Relationship Table
            $graphicFontInit = new GraphicFont();
            $findGraphicFont = $graphicFontInit->find($fontInsertId);
            if ($findGraphicFont->tags()->sync($getTagIds)) {
                return true;
            }
        } else {
            // Clean relation in case no tags supplied
            $tagRelInit = new GraphicFontTagRelation();
            $graphicFontTags = $tagRelInit->where('graphic_font_id', $fontInsertId);
            if ($graphicFontTags->delete()) {
                return true;
            }
        }
        return false;
    }

    /**
     * GET: Single Graphic Font
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author dan@riaxe.com
     * @date   18-07-2022
     * @return All/Single Graphic Font List
     */
    public function getSingleGraphicFont($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'data' => [],
            'message' => message('Graphic Font', 'not_found')
        ];

        $getStoreDetails = get_store_details($request);
        $graphicFontId = $args['id']; // graphic font id

        if (!empty($graphicFontId)) {
            $graphicFontsInit = new GraphicFont();
            $getGraphicFonts = $graphicFontsInit->with([
                'letter_style', 'number_style', 'special_character_style'
            ])
                ->select([
                    'xe_id', 'name', 'price', 'is_letter_style',
                    'is_number_style', 'is_special_character_style', 'cloud_storage'
                ])
                ->where([
                    ['xe_id', '>', 0],
                    ['xe_id', '=', $graphicFontId]
                ]);

            $graphicFontData = $getGraphicFonts->first()->toArray();
               
            if ($graphicFontData['cloud_storage'] == 1) {
                $graphicFile = $graphicFontData['file_name'];
                $value = $this->getS3URL($graphicFile, $getStoreDetails['store_id']);
                $graphicFontData['file_name'] = $value;

                foreach ($graphicFontData['letter_style'] as $key => $letterValue) {
                    $letterFile = $letterValue['file_name'];
                    $url = $this->getS3URL($letterFile, $getStoreDetails['store_id']);
                    $graphicFontData['letter_style'][$key]['file_name'] = $url;
                }
                foreach ($graphicFontData['number_style'] as $key => $numberValue) {
                    $numberFile = $numberValue['file_name'];
                    $url = $this->getS3URL($numberFile, $getStoreDetails['store_id']);
                    $graphicFontData['number_style'][$key]['file_name'] = $url;
                }
                foreach ($graphicFontData['special_character_style'] as $key => $specialValue) {
                    $specialFile = $specialValue['file_name'];
                    $url = $this->getS3URL($specialFile, $getStoreDetails['store_id']);
                    $graphicFontData['special_character_style'][$key]['file_name'] = $url;
                }
            }
            
            $getTags = $this->getTagsById(
                'GraphicFonts',
                'GraphicFontTagRelation',
                'graphic_font_id',
                $graphicFontId
            );
            $graphicFontData['tags'] = $getTags;

            $jsonResponse = [
                'status' => 1,
                'data' => [$graphicFontData]
            ];
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }
    /**
     * GET: List of Graphic Font
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return All/Single Graphic Font List
     */

    public function getGraphicFonts($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;

        // Get Store Specific Details from helper
        $getStoreDetails = get_store_details($request);
        $offset = 0;
        $graphicFontsInit = new GraphicFont();

        //All Filter columns from url
        $page = $request->getQueryParam('page');
        $perpage = $request->getQueryParam('perpage');
        $sortBy = !empty($request->getQueryParam('sortby'))
        && $request->getQueryParam('sortby') != ""
        ? $request->getQueryParam('sortby') : 'xe_id';
        $order = !empty($request->getQueryParam('order'))
        && $request->getQueryParam('order') != ""
        ? $request->getQueryParam('order') : 'desc';
        $name = $request->getQueryParam('name');

        $getGraphicFonts = $graphicFontsInit->where([
            ['xe_id', '>', 0],
            ['store_id', '=', $getStoreDetails['store_id']]
        ])->select([
            'xe_id', 'name', 'price', 'is_letter_style', 'is_number_style',
            'is_special_character_style', 'cloud_storage'
        ]);

        // Searching as per name, category name & tag name
        if (isset($name) && $name != "") {
            $name = '\\' . $name;
            $getGraphicFonts->where(function ($query) use ($name) {
                $query->where('name', 'LIKE', '%' . $name . '%')
                    ->orWhereHas('graphicFontTags.tag', function ($q) use ($name) {
                        return $q->where('name', 'LIKE', '%' . $name . '%');
                    });
            });
        }

        // Total records including all filters
        $getTotalPerFilters = $getGraphicFonts->count();

        // Pagination Data
        if (!empty($page)) {
            $totalItem = empty($perpage) ? PAGINATION_MAX_ROW : $perpage;
            $offset = $totalItem * ($page - 1);
            $getGraphicFonts->skip($offset)->take($totalItem);
        }

        // Sorting All records by column name and sord order parameter
        if (!empty($sortBy) && !empty($order)) {
            $getGraphicFonts->orderBy($sortBy, $order);
        }

        $graphicFontData = $getGraphicFonts->get()->toArray();

        foreach ($graphicFontData as $fontKey => $fontValue) {
            if ($fontValue['cloud_storage'] == 1 && !empty($fontValue['file_name'])) {
                $thisFileName = $fontValue['file_name'];
                $graphicFontData[$fontKey]['file_name'] = $this->getS3URL($thisFileName, $getStoreDetails['store_id']);
            }
        }

        $jsonResponse = [
            'status' => 1,
            'records' => count($graphicFontData),
            'total_records' => $getTotalPerFilters,
            'data' => $graphicFontData
        ];

        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * PUT: Update a Single Graphic Font
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return json response wheather data is updated or not
     */
    public function updateGraphicFonts($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Graphic Fonts', 'not_found')
        ];
        try {
            // Get Store Specific Details from helper
            $getStoreDetails = get_store_details($request);

            $allPostPutVars = $request->getParsedBody();
            $graphicFontArray =  !empty($allPostPutVars['data']) ?
            json_clean_decode($allPostPutVars['data'], true) : [];
            $graphicFontId = to_int($args['id']);

            // check exising record with same name
            $graphicFontInit = new GraphicFont();
            $nameCheck = $graphicFontInit->where([
                ['name', '=', $graphicFontArray['name']],
                ['xe_id', '<>', $graphicFontId]
            ]);

            if (empty($graphicFontArray['name'])) {
                $jsonResponse['message'] = 'Graphic font name required!';
            } elseif ($nameCheck->count() > 0) {
                $jsonResponse['message'] = message('Graphic Fonts', 'exist');
            } else {
                // Save Graphic Font Details
                $graphicFontInit = new GraphicFont();
                $graphicFonts = $graphicFontInit->where('xe_id', $graphicFontId);

                // get the font record
                $graphicFontArr = $graphicFonts->firstOrFail();
              
                // Update record
                $updateDetail = [
                    'name' => $graphicFontArray['name'],
                    'price' => $graphicFontArray['price'],
                    'store_id' => $getStoreDetails['store_id'],
                    'is_letter_style' => $graphicFontArray['is_letter_style'],
                    'is_number_style' => $graphicFontArray['is_number_style'],
                    'is_special_character_style' => $graphicFontArray['is_special_character_style']
                ];

                $graphicFonts->update($updateDetail);

                // Update Graphic Font Designs
                if (!empty($graphicFontArray['letter_style'])) {
                    $this->_updateFontDesigns(
                        $graphicFontArray['letter_style'],
                        $graphicFontId,
                        'letter',
                        $getStoreDetails['store_id'],
                        $graphicFontArr
                    );
                }
                if (!empty($graphicFontArray['number_style'])) {
                    $this->_updateFontDesigns(
                        $graphicFontArray['number_style'],
                        $graphicFontId,
                        'number',
                        $getStoreDetails['store_id'],
                        $graphicFontArr
                    );
                }
                if (!empty($graphicFontArray['special_character_style'])) {
                    $this->_updateFontDesigns(
                        $graphicFontArray['special_character_style'],
                        $graphicFontId,
                        'special_character',
                        $getStoreDetails['store_id'],
                        $graphicFontArr
                    );
                }
                /**
                 * Update tags
                 * Tag Names format : tag1,tag2,tag3
                 */
                $tags = !empty($graphicFontArray['tags']) ? $graphicFontArray['tags'] : "";
                $this->saveGraphicFontTags(
                    $getStoreDetails['store_id'],
                    $graphicFontId,
                    $tags
                );

                $jsonResponse = [
                    'status' => 1,
                    'message' => message('Graphic Fonts', 'updated')
                ];
            }
        } catch (\Exception $e) {
            $serverStatusCode = EXCEPTION_OCCURED;
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Graphic Fonts'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * Update Graphic-Font-Designs
     *
     * @param $fontdetails Font Details Array
     * @param $fontId      Graphic Font Id
     * @param $slug        Type of graphic font
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return boolean
     */
    private function _updateFontDesigns($fontdetails, $fontId, $slug, $storeId, $graphicFontArr)
    {
        // do not proceed if font detail is empty
        if (empty($fontdetails)) {
            return false;
        }

        $initGraphicFonts = new GraphicFontLetter();
        $isCloudStorage = $graphicFontArr['cloud_storage'];
       
        foreach ($fontdetails as $fontData) {
            // get the old record if present
            $getStyle = NULL;
            if (!empty($fontData['xe_id'])) {
                $getStyle = $initGraphicFonts->where('xe_id', '=', $fontData['xe_id'])->get()->first();
            } 
              
            // delete the font marked as trash
            if (!empty($fontData['is_trash'])) {

                // data is only present at the front-end, skip check further
                if (!$getStyle) {
                    continue;
                }
                // delete the file
                $this->_deleteLetterFile($getStyle->xe_id, $isCloudStorage, $getStyle->file_name, $storeId);

                // delete the record
                $initGraphicFonts->where('xe_id', '=', $getStyle->xe_id)->delete();
                continue;
            }
           
            
            // set the style data
            $designData = [
                'graphic_font_id' => $fontId,
                'name' => $fontData['name'],
                'font_type' => $slug
            ];

            $getUploadedFileName = false;
            if (!empty($fontData['image_upload_data'])) {
                // upload the file if present
                $requestedFileKey = $fontData['image_upload_data'];

                $getUploadedFileName = do_upload(
                    $requestedFileKey,
                    path('abs', 'graphicfont'),
                    [],
                    'string'
                );
            }
            
            if ($getUploadedFileName) {
                // move file to the s3
                if ($isCloudStorage) {
                    $fileToUpload = path('abs', 'graphicfont') . $getUploadedFileName;
                    $this->uploadFileToS3("graphicfont", $fileToUpload, $storeId);
                }
                $designData['file_name'] = $getUploadedFileName;

                 // delete the old file: for existing record
                if ($getStyle) {
                    $this->_deleteLetterFile($getStyle->xe_id, $isCloudStorage, $getStyle->file_name, $storeId);
                }
            }

            if ($getStyle) {
                // update the exising style
                $initGraphicFonts->where('xe_id', '=', $getStyle->xe_id)->update($designData);
            } else {
                // create new style
                $initGraphicFonts->fill($designData)->save();
            }
        }
        return true;
    }

     /**
     * Internal: Delete the old files of letter style
     *
     * @param $letterStyleId   Letter style ID
     * @param $fileName        Letter style file name 
     * @param $isCloudStorage  Cloud Storage flag (1=true)
     *
     * @date   11 August 2022
     * @return boolean
     */
    private function _deleteLetterFile($letterStyleId, $isCloudStorage = 0, $fileName = NULL, $storeId = NULL)
    {
        // delete the file from the cloud
        if ($isCloudStorage && $fileName && $storeId) {
            $this->deleteS3File($fileName, $storeId);
            return true;
        } 
        // delete the file from the server
        return $this->deleteOldFile(
            "graphic_font_letters",
            "file_name",
            ['xe_id' => $letterStyleId],
            path('abs', 'graphicfont'),
            false
        );
    }

    /**
     * DELETE: Delete single/multiple Graphic Font(s)
     *
     * @param $request  Slim's Argument parameters
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return json response wheather data is deleted or not
     */
    public function deleteGraphicFonts($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => message('Graphic Fonts', 'not_found')
        ];
        try {
            $getStoreDetails = get_store_details($request);
            $storeId = $getStoreDetails['store_id'];
            $isS3Enabled = $this->checkS3Settings($storeId);
            $getDeleteIdsToArray = !empty($args['id']) ? json_clean_decode($args['id'], true) : [];

            if (!empty($getDeleteIdsToArray)) {
                $totalCount = count($getDeleteIdsToArray);

                $graphicFontDelInit = new GraphicFont();
                $getGraphicFonts = $graphicFontDelInit->whereIn('xe_id', $getDeleteIdsToArray)->get();

                $success = 0;
                foreach ($getGraphicFonts as $graphicFont) {
                    // get the cloud storage flag
                    $isS3Enabled = $graphicFont->cloud_storage;

                    // get the graphic font letter
                    $getFontLetters = (new GraphicFontLetter())->where('graphic_font_id', $graphicFont->xe_id);
                    if ($getFontLetters->count() > 0) {
                        $fontLetters = $getFontLetters->get();
                        // delete the graphic letter file
                        foreach ($fontLetters as $fontLetter) {
                            if ($isS3Enabled) {
                                $fileName = $fontLetter->file_name;
                                $this->deleteS3File($fileName, $storeId);
                            } else {
                                $this->deleteOldFile(
                                    "graphic_font_letters",
                                    "file_name",
                                    ['xe_id' => $fontLetter->xe_id],
                                    path('abs', 'graphicfont'),
                                    false
                                );
                            }
                        }
                        // delete the graphic letter record
                        $getFontLetters->delete();
                    }

                    // delete the relationship table
                    (new GraphicFontTagRelation())->where('graphic_font_id', $graphicFont->xe_id)->delete();

                    // delete the font record
                    $graphicFontDelInit->where('xe_id', $graphicFont->xe_id)->delete();
                    $success++;
                }
                $jsonResponse = [
                    'status' => 1,
                    'message' => $success . ' out of ' . $totalCount
                        . ' Graphic Fonts(es) deleted successfully'
                ];
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('Assets', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Graphic Fonts'
                ]
            ]);
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     * GET: List of Graphic Font
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author satyabratap@riaxe.com
     * @date   31st Jan 2019
     * @return All Graphic Font List
     */
    public function getAllGraphicFonts($request, $response)
    {
        $serverStatusCode = OPERATION_OKAY;
        $graphicFontData = [];
        $jsonResponse = [
            'status' => 0,
            'data' => [],
            'message' => message('Graphic Font', 'not_found'),
        ];
        // Get Store Specific Details from helper
        $getStoreDetails = get_store_details($request);
        $graphicFontsInit = new GraphicFont();
        $getGraphicFonts = $graphicFontsInit->with('characters')
            ->select(
                'xe_id', 'name', 'price', 'is_letter_style',
                'is_number_style', 'is_special_character_style'
            )
            ->where('xe_id', '>', 0)
            ->where('store_id', $getStoreDetails['store_id']);
        $graphicFontData = $getGraphicFonts->get();
        if (!empty($graphicFontData) && $graphicFontData->count() > 0) {
            // Unset file_name Key in case of single record fetch
            $graphicFontData = json_clean_decode($graphicFontData, true);
            unset($graphicFontData['file_name']);
            $jsonResponse = [
                'status' => 1,
                'records' => count($graphicFontData),
                'data' => $graphicFontData,
            ];
        }

        return response(
            $response, ['data' => $jsonResponse, 'status' => $serverStatusCode]
        );
    }

    /**
     * GET: include dom html file
     *
     * @author robert@imprintnext.com
     * @date   17 Nov 2020
     * @return nothing
     */
    private function domHtmlPathInclue()
    {
        include_once dirname(__FILE__) . '/../../../Dependencies/simple_html_dom.php';
    }

    /**
     * Update SVG height and width
     *
     * @param $imageUrl SVG file URL
     *
     * @author robert@imprintnext.com
     * @date   17 Nov 2020
     * @return boolean
     */
    private function updateSvgHeightWidth($imageUrl)
    {
        $fileContent = read_file($imageUrl);
        $html = new \simple_html_dom();
        $html->load($fileContent, false);
        $viewBoxSvg = $html->find('svg[viewBox]', 0);
        if (isset($viewBoxSvg) && !empty($viewBoxSvg)) {
            $viewBox = $viewBoxSvg->viewBox;
            if (isset($viewBox) && !empty($viewBox)) {
                $viewBox = explode(' ', $viewBox);
                $vBwidth = $viewBox[2];
                $vBheight = $viewBox[3];
                if ($vBwidth &&  $vBheight) {
                    $viewBoxSvg->height = $vBheight;
                    $viewBoxSvg->width =  $vBwidth;
                }
                write_file( $imageUrl, $viewBoxSvg );
            }
        } 
    }
}
