<?php
/**
 * Manage logs for the catalog
 *
 * PHP version 5.6
 *
 * @category  Products
 * @package   Log
 * @author    Debasis Rath <dan@imprintnext.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */
namespace App\Modules\Products\Controllers;
use App\Modules\Products\Models\CatalogProductRel;
/**
 * CatalogLog Controller
 *
 * @category Class
 * @package  Log
 * @author   Debasis Rath <dan@imprintnext.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */
class CatalogLogController
{
    public $maxLog = 3; // max kept on the server
    public $logDir = ASSETS_PATH_W . 'catalog/logs/'; // log path
    public $ext = 'json';
    public $activeLogTime = 24; // validity of active log time (hours)
    public $fileName = ['sanmar' => '', 'alp' => '', 'sns' => ''];
    public $logData = [];

    public function __construct($config = [])
    {
        if (!is_dir($this->logDir)) {
            mkdir($this->logDir, 0777, true);
        }
        $this->logData['log_stamp'] = time();
    }
    /**
     * Internal: Get the last log file, create new if it is expired
     */
    private function getLogFileName($provider)
    {
        // default log file name
        $fileName = $this->logDir . $provider . '_' . time() . '.' . $this->ext;

        // get log order by name descending
        $fileList = array_reverse(glob($this->logDir . $provider . '*.' . $this->ext));

        if (!empty($fileList)) {
            // last created file
            $lastFile = $fileList[0];
            // get the last file if created within 24 hours
            $fileTimeStamp = ltrim(pathinfo($lastFile, PATHINFO_FILENAME), "{$provider}_");
            if ((time() - $fileTimeStamp) < $this->activeLogTime * 3600) {
                $fileName = $lastFile;
            } else {
                array_unshift($fileList, $fileName);
            }
        }

        // keep logs upto max-log
        while (count($fileList) > $this->maxLog) {
            $delFile = end($fileList);
            unlink($delFile);
            array_pop($fileList);
        }

        return $fileName;
    }

    /**
     * Internal: set common fields for log
     */
    public function setCommonData($logData = [])
    {
        $this->logData = array_merge($this->logData, $logData);
    }
    /**
     * Internal: create log based on povider
     */
    public function createLog($message, $logData = [])
    {
        // keep the common data with dynamic fields
        $logData = array_merge($this->logData, $logData);
        $currProvider =  !empty($logData['provider']) ? $logData['provider'] : 'catalog' ;
        // new file is created when expiry time (maxLog) has passed
        if (!$this->fileName[$currProvider]) {
            $this->fileName[$currProvider] = $this->getLogFileName($currProvider);
        }
        $logData['message'] = $message;
        $logData = json_encode($logData);

        if (isset($logData['status'])) {
            $logData .= PHP_EOL;
        }
        // write the log and back to caller
        return file_put_contents($this->fileName[$currProvider], $logData . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    /**
     *  Get: the log file list
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author dan@imprintnext.com
     * @date   15-02-2023
     * @return json
     */
    public function getLogFiles($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'data' => 'No log found!'
        ];

        // get log order by name descending
        $fileList = glob($this->logDir . '*.' . $this->ext);

        if ($fileList) {
            $logFiles = array_map(function ($file) {
                $modified = filemtime($file);
                $file = pathinfo($file, PATHINFO_BASENAME);
                $provider = strtok($file, '_');
                $logStamp = ltrim(pathinfo($file, PATHINFO_FILENAME), "{$provider}_");
    
                return [
                    'file' => $file,
                    'provider' => $provider,
                    'created' => date('Y-m-d H:i:s', $logStamp),
                    'modified' =>  date('Y-m-d H:i:s', $modified)
                ];
                
            }, $fileList);
            
            // sort by modified
            array_multisort(array_column($logFiles, 'modified'), SORT_DESC, $logFiles);

            $jsonResponse = [
                'status' => 1,
                'data' => $logFiles
            ];
        }
       
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    /**
     *  Get: read log file details
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author dan@imprintnext.com
     * @date   15-02-2023
     * @return json
     */
    public function readLogs($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'message' => 'Some error occurred!'
        ];
        // get the parameters
        $filters = $request->getQueryParams();
        $filters['tags'] = 'status';
        
        // productId is a mandatory parameter to get the log details
        $productId = $filters['product_id'];
        
        // get the provider from the relation table
        $catalogProductRelInit = new CatalogProductRel();
        $getCatalogProductRel = $catalogProductRelInit->where('product_id', $productId)->first();
        $provider = $getCatalogProductRel['catalog_code'];

        $skipFile = isset($filters['skip_file']) ? intval($filters['skip_file']) : 0;
        // get log order by name descending
        $fileList = array_reverse(glob($this->logDir . $provider . '*.' . $this->ext));
        if ($fileList) {
            $fileCounts = count($fileList);
            foreach ($fileList as $key => $file) {
                // to skip log to search in loder log files
                if ($skipFile > 0) {
                    $skipFile--;
                    continue;
                }
                $data = $this->readFile($file, $filters);
                if ($data) {
                    foreach ($data as &$record) {
                        $successVariants = array_intersect($record['inventory_status']['success'], $record['price_status']['success']);
                        $failedVariants = array_merge($record['inventory_status']['failed'], $record['price_status']['failed']);
                        $record['success_count'] = count($successVariants);
                        $record['failure_count'] = count($failedVariants);
                        if (!empty($record['inventory_status']['failed'])) {
                            $record['inventory_status']['failed'] = array_map(function ($sku) {
                                return ['id' => $sku, 'message' => 'Failed while updating at the store'];
                            }, $record['inventory_status']['failed']);
                        }
                        if (!empty($record['price_status']['failed'])) {
                            $record['price_status']['failed'] = array_map(function ($sku) {
                                return ['id' => $sku, 'message' => 'Price could not be updated'];
                            }, $record['price_status']['failed']);
                        }
                    }
                    $fileTimeStamp = ltrim(pathinfo($file, PATHINFO_FILENAME), "{$provider}_");
                    $jsonResponse = [
                        'status' => 1,
                        'data' => [
                            'created' => date('Y-m-d H:i:s', $fileTimeStamp),
                            'total_logs' => $fileCounts,
                            'records' => count($data),
                            'logs' => $data,
                            'log_number' => $key
                        ]
                    ];
                }
            }
        }
        return response($response, [
            'data' => $jsonResponse, 'status' => $serverStatusCode
        ]);
    }

    private function getInternalStatus($file, $record, $type = 'inventory')
    {
        $failedSkus = $tags = [];
        if ($type == 'inventory') {
            $failedSkus = $record['inventory_status']['failed'];
            $tags = ['inventory'];
        } else {
            $tags = ['price'];
            $failedSkus = $record['price_status']['failed'];
        }

        $params = [
            'skus' => $failedSkus,
            'notTags' => ['status'],
            'log_stamp' => $record['log_stamp'],
            'product_id' => $record['product_id'],
            'tags' => $tags
        ];
        
        // the message is before variant fetch - add message from itself
        if ($failedSkus[0] == 'all') {
            return ['all' => $record['message']];
        }
        
        $internalRespose = $this->readFile($file, $params, 1);

        $messages = [];
        if (!empty($internalRespose)) {
            $messages = array_column($internalRespose, 'message', 'sku');
        }

        array_walk($failedSkus, function (&$sku) use ($messages, $type) {
            $msg = isset($messages[$sku]) ? $messages[$sku] : "Failed while updating $type on the store";
            $sku = [$sku => $msg];
        });

        return $failedSkus;
    }

    private function readFile($file, $filters, $debug = false)
    {
        // do not continue if file does not exists
        if (!file_exists($file)) {
            return [];
        }
        // do not continue if file opening error
        $fp = fopen($file, 'r');
        if ($fp == null) {
            return [];
        }
        // optional to skip records
        $skipRrecords = isset($filters['skip_rec']) ? intval($filters['skip_rec']) : 0;
        // default matching record array
        $data = [];
        while (!feof($fp)) {
            $line = trim(fgets($fp));
            // last line contains newline char
            if (!$line) {
                continue;
            }
            // may required - not in use
            if ($skipRrecords) {
                $skipRrecords--;
                continue;
            }
            // read the line, apply filters
            $rec = json_decode($line, true);
            if (isset($filters['log_stamp']) && $rec['log_stamp'] != $filters['log_stamp']) {
                continue;
            }
            if (isset($filters['tags']) && array_diff((array)$filters['tags'], $rec['tags'])) {
                continue;
            }
            if (isset($filters['product_id']) && $filters['product_id'] != $rec['product_id']) {
                continue;
            }
            $data[] = $rec;
        }

        fclose($fp);
        // sort the result
        if ($data && !empty($filters['sort_by'])) {
            $order = isset($filters['order']) && strtolower($filters['order']) == 'asc' ? SORT_ASC : SORT_DESC;
            array_multisort(array_column($data, $filters['sort_by']), $order, SORT_NATURAL|SORT_FLAG_CASE, $data);
        }
        // matching records from the file
        return $data;
    }
}