<?php
/**
 * Manage Products from Woocommerce Store
 *
 * PHP version 5.6
 *
 * @category  Products
 * @package   Store
 * @author    Tanmaya Patra <tanmayap@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */
namespace App\Modules\Products\Controllers;

use App\Modules\PrintProfiles\Models as PrintProfileModels;
use App\Modules\Products\Models\AppUnit;
use App\Modules\Products\Models\AttributePriceRule;
use App\Modules\Products\Models\DecorationObjects;
use App\Modules\Products\Models\PrintProfileDecorationSettingRel;
use App\Modules\Products\Models\PrintProfileProductSettingRel;
use App\Modules\Products\Models\ProductDecorationSetting;
use App\Modules\Products\Models\ProductImageSettingsRel;
use App\Modules\Products\Models\ProductImageSides;
use App\Modules\Products\Models\ProductSetting;
use App\Modules\Products\Models\ProductSide;
use App\Modules\Products\Models\ProductOptions;
use App\Modules\Products\Models\ProductOptionRel;
use App\Modules\Products\Models\ProductVariants;
use App\Modules\Products\Models\ProductVariantRel;
use Illuminate\Database\Capsule\Manager as DB;
use ProductStoreSpace\Controllers\StoreProductsController;
use App\Modules\Products\Controllers\ProductConfiguratorController as Configurator;
use App\Modules\Customers\Controllers\CustomerGroupController as CustomerGroup;

/**
 * Products Controller
 *
 * @category Class
 * @package  Product
 * @author   Tanmaya Patra <tanmayap@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */
class ProductsController extends StoreProductsController {
	/**
	 * GET: Getting List of All product or Single product information
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return A JSON Response
	 */
	public function getProductList($request, $response, $args, $returnType = 0)
	{
		$serverStatusCode = OPERATION_OKAY;
		$isAdmin = $request->getQueryParam('is_admin');
		$fetch = $request->getQueryParam('fetch');
		$getStoreDetails = get_store_details($request);
		$storeId = $getStoreDetails['store_id'];
		// get the store product list
		$getProductResponse = $this->getProducts($request, $response, $args);
		$productList = [];
		if (!empty($getProductResponse['products'])) {
			foreach ($getProductResponse['products'] as $eachProduct) {
				$singleProduct = [
					'id' => $eachProduct['id'],
					'variant_id' => $eachProduct['variation_id'],
					'name' => $eachProduct['name'],
					'sku' => $eachProduct['sku'],
					'price' => $eachProduct['price'],
					'is_template' => false,
					'image' => isset($eachProduct['image']) ? $eachProduct['image'] : []
				];
				
				if ($isAdmin) {
					$getAssociatedPrintProfileData = $this->getAssocPrintProfiles($eachProduct['id'], $isAdmin, $storeId);
					if ($fetch != '') {
						$singleProduct['print_profile'] = [];
					} else {
						$singleProduct['print_profile'] = $getAssociatedPrintProfileData['print_profiles'];
					}
				} else {
					$singleProduct += [
						'stock' => $eachProduct['stock'],
						'is_sold_out' => $eachProduct['is_sold_out'],
						'type' => $eachProduct['type']
					];
				}
				if ($fetch == 'all') {
					$decorationType = $this->getProductDecorationType($eachProduct['id'], $storeId);
					$singleProduct += [
						'custom_design_id' => $eachProduct['custom_design_id'],
						'is_decorated_product' => $eachProduct['is_decorated_product'],
						'is_redesign' => $eachProduct['is_redesign'],
						'decoration_type' => $decorationType
					];
				}
				$productList[] = $singleProduct;
			}
		}
		$jsonResponse = [
			'status' => 1,
			'records' => count($productList),
			'total_records' => $getProductResponse['total_records'],
			'data' => $productList
		];
		if ($returnType == 1) {
			return $jsonResponse;
		}
		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

		/**
	 * GET: Getting List of All product or Single product information
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return A JSON Response
	 */
	public function getProductDetails($request, $response, $args, $returnType = 0)
	{
		$serverStatusCode = OPERATION_OKAY;
		$productDetails = [];
		$getStoreDetails = get_store_details($request);
		$storeId = $getStoreDetails['store_id'];
		$isAdmin = $request->getQueryParam('is_admin');
		$type = $request->getQueryParam('type');
		$customerId = $request->getQueryParam('customer', '');
		// get product details from the store
		$getProductResponse = $this->getProducts($request, $response, $args);
		$productDetails = $getProductResponse['products'];
		if ($storeId > 1 && $type == "tool" && IS_HOSTED == 0) {
			$currentStoreUrl = '';
			$databaseStoreInfo = DB::table('stores')->where('xe_id', '=', $storeId)->get();
			if ($databaseStoreInfo->count() > 0) {
				$currentStoreUrl = $databaseStoreInfo->first()->store_url;
			}
			foreach ($productDetails['images'] as $key => $value) {
				$fileName = $value['src'];
				$thumbnail = $value['thumbnail'];
				$hostname = parse_url($fileName, PHP_URL_HOST); //hostname
				$productDetails['images'][$key]['src'] = str_replace($hostname, $currentStoreUrl, $fileName);
				$productDetails['images'][$key]['thumbnail'] = str_replace($hostname, $currentStoreUrl, $thumbnail);
			}
		}
		// get the print profile details
		$getAssociatedPrintProfileData = $this->getAssocPrintProfiles($productDetails['id'], $isAdmin, $storeId);
		$productDetails['print_profile'] = $getAssociatedPrintProfileData['print_profiles'];
		$productDetails['attributes'] = $this->getPriceDetails($productDetails['attributes'], $args['id']);
		$productDetails['is_decoration_exists'] = 1;

		//passing group discount data
		if (!empty($customerId) && STORE_NAME != 'Opencart') {
			$productDetails['customer_group'] = [];
			$customerGrpCtrlInit = new CustomerGroup();
			$customerDiscountData = $customerGrpCtrlInit->getDiscountDataBycustomerId($request, $response, [
				'customerId' => $customerId
			]);
			if (!empty($customerDiscountData)) {
				$productDetails['customer_group'] = $customerDiscountData;
			}
		}
		$jsonResponse = [
			'status' => 1,
			'records' => 1,
			'data' => $productDetails
		];
		if ($returnType == 1) {
			return $jsonResponse;
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}
	/**
	 * GET: Getting List of All Category/Subcategory or Single
	 * Category/Subcategory information
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return A JSON Response
	 */
	public function totalCategories($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => []
		];
		
		$getstoreDetails = get_store_details($request);
		$storeId = $getstoreDetails['store_id'];
		$apiSource = $request->getQueryParam('src');
		// get selected categories only
		$settingLocation = path('abs', 'setting') . 'stores/' . $storeId;
		$jsonFilePath = $settingLocation . '/product_categories.json';
		if (file_exists($jsonFilePath)) {
			$logContent = json_decode(file_get_contents($jsonFilePath), true);
			$selectedCatagories = array_column($logContent['data'], 'id');
		}
		$storeResponse = $this->getCategories($request, $response, $args);
		// bellow three line to sort store responded category list alphabetically.
		$keys = array_column($storeResponse, 'name');
		$arrayLowercase = array_map('strtolower', $keys);
        array_multisort($arrayLowercase, SORT_ASC, SORT_STRING, $storeResponse);
		$categoriesForTool = [];
		foreach ($storeResponse as $key => $category) {
			$storeResponse[$key]['show_in_tool'] = false;
			if (in_array($category['id'], $selectedCatagories)) {
				$storeResponse[$key]['show_in_tool'] = true;
				$categoriesForTool[] = $category;
			}
		}
		if ($apiSource == 'tool' && !empty($selectedCatagories)) {
			$storeResponse = $categoriesForTool;
		}
		if (!empty($storeResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $storeResponse
			];
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode,
		]);
	}
	/**
	 * GET: Get list of Measurement Units
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return A JSON Response
	 */
	public function getMeasurementUnits($request, $response)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Measurement Unit', 'error')
		];

		$appUnitInit = new AppUnit();
		$initAppUnit = $appUnitInit->whereNotNull('name')->select([
			'xe_id','name','is_default','label'
		]);

		if ($initAppUnit->count() > 0) {
			$jsonResponse = [
				'status' => 1,
				'data' => $initAppUnit->orderBy('xe_id', 'desc')->get()
			];
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

	/**
	 * Clone the Product Decoration
	 *
	 * @param $parentId   Parent Product Id
	 * @param $predecoId  Pre Decorated Product Id
	 *
	 * @author satyabratap@riaxe.com
	 * @date   21 Apr 2020
	 * @return A JSON Response
	 */
	public function cloneProduct($parentId, $predecoId, $productSideImages)
	{
		try {
			$productSettingId = 0;
			$isProductImage = 1;
			$getSettingsInit = new ProductSetting;
			$settingDetails = $getSettingsInit->where(['product_id' => $parentId])->first();
			if (empty($settingDetails)) {
				$settingData = $this->getSettingsByBulkDecorationSavingProduct($parentId);

				if (!empty($settingData)) {
	                $isProductImage = $settingData->is_product_image;
	                $importedProductId = $settingData->product_id;
	                $settingDetails = $getSettingsInit->where(['product_id' => $importedProductId])->first();
	            }
			}

			if (empty($settingDetails)) {
				throw new \Exception('Setting details not found :product-' . $parentId);
			}

			$settingDetails = $settingDetails->toArray();
			$productSettingId = $settingDetails['xe_id'];

			$clonedSettingId = $this->saveProductsSetting($settingDetails, $predecoId, $parentId);
			
			//Save Product Image setting Relation
			if($isProductImage == 1) {
				$this->cloneProductImageSettingsRel($productSettingId, $clonedSettingId);
			}
			
			//Save Product Printprofile setting Relation
			$this->colonePrintProfileProductSettingRel($productSettingId, $clonedSettingId);

			if ($settingDetails['is_variable_decoration'] == 1) {
				$getDecorationInit = new ProductDecorationSetting();
				$getDecorationData = $getDecorationInit->where(['product_setting_id' => $productSettingId])->first();
				
				if (!empty($getDecorationData)) {
					$decoration = $getDecorationData->toArray();
					$saveDecoration = new ProductDecorationSetting([
						'product_setting_id' => $clonedSettingId,
						'dimension' => $decoration['dimension'],
						'print_area_id' => $decoration['print_area_id'],
						'sub_print_area_type' => $decoration['sub_print_area_type'],
						'pre_defined_dimensions' => $decoration['pre_defined_dimensions'],
						'user_defined_dimensions' => $decoration['user_defined_dimensions'],
						'custom_min_height' => $decoration['custom_min_height'],
						'custom_max_height' => $decoration['custom_max_height'],
						'custom_min_width' => $decoration['custom_min_width'],
						'custom_max_width' => $decoration['custom_max_width'],
						'custom_bound_price' => $decoration['custom_bound_price'],
						'is_border_enable' => $decoration['is_border_enable'],
						'is_sides_allow' => $decoration['is_sides_allow'],
						'no_of_sides' => $decoration['no_of_sides'],
						'is_dimension_enable' => $decoration['is_dimension_enable'],
						'locations' => $decoration['locations'],
						'is_disable_design' => $decoration['is_disable_design'],
						'image_overlay' => $decoration['image_overlay'],
						'multiply_overlay' => $decoration['multiply_overlay'],
						'overlay_file_name' => $decoration['overlay_file_name'],
						'bleed_mark_data' => $decoration['bleed_mark_data'],
						'shape_mark_data' => $decoration['shape_mark_data'],
						'default_pp_id' => $decoration['default_pp_id'],
					]);
					$saveDecoration->save();
				}
			} else {
				$get3DObjInit = new DecorationObjects();
				$objDataExist = $get3DObjInit->where(['product_id' => $parentId])->count();
				if ($objDataExist > 0) {
					$this->cloneDecorationObjects($parentId, $predecoId);
				}

				$getProductSideInit = new ProductSide();
				$sideData = $getProductSideInit->where(['product_setting_id' => $productSettingId])->get();
				if (!empty($sideData)) {
					// clone product side settings
					$this->cloneProductSide($sideData, $clonedSettingId, $productSettingId, $productSideImages);
				}
			}
			$jsonResponse = [
				'status' => 1,
				'message' => message('Products', 'done')
			];
		} catch (Exception $e) {
			$jsonResponse = [
				'status' => 0,
				'message' => message('Products', 'Error'),
			];
			create_log('product', 'error', [
				'message' => $e->getMessage(),
				'extra' => [
					'module' => 'Products Clone'
				]
			]);
		}
		return $jsonResponse;
	}

	/**
	 * Post: Save Predecorator Data at Store end
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return Save response Array
	 */
	public function savePredecorator($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Predecorator', 'error'),
		];
		// Call Internal Store
		$predecoSaveResp = $this->saveProduct($request, $response);

		$predecoProductId =  $predecoSaveResp['product_id'];
		$getStoreProduct = $this->getProducts($request, $response, [
            'id' => $predecoProductId
        ]);
		// product details
		$getProductDetails = $getStoreProduct['products'];
		// product images
		$productSideImages = $getProductDetails['images'];

		//Copy the Decoration of the Parent Product
		$predecoDetails = $request->getParsedBody();
		$productData = json_clean_decode($predecoDetails['data']);
		$parentProductId = $productData['parent_product_id'];
		$cloneStatus['status'] = 0;
		if (!empty($predecoSaveResp)) {
			if ($parentProductId != "" && $productData['ref_id'] > 0) {
				$cloneStatus = $this->cloneProduct($parentProductId, $predecoSaveResp['product_id'], $productSideImages);
			}
			$decoJsonStatus = 0;
			if ($productData['ref_id'] != "") {
				$templateJsonPath = path('abs', 'design_state') . 'templates/' . $productData['ref_id'] . '.json';
				if (file_exists($templateJsonPath)) {
					$preDecoFolder = path('abs', 'design_state') . 'predecorators';
					if (!file_exists($preDecoFolder)) {
						mkdir($preDecoFolder, 0777, true);
					}
					$preDecoJsonFile = path('abs', 'design_state') . 'predecorators/' . $productData['ref_id'] . '.json';
					write_file($preDecoJsonFile, file_get_contents($templateJsonPath));
					$decoJsonStatus = 1;
				}
			}
			$jsonResponse = [
				'status' => 1,
				'product_id' => $predecoSaveResp['product_id'],
				'decoration_status' => $cloneStatus['status'],
				'deco_json_status' => $decoJsonStatus,
				'message' => message('Predecorator', 'saved'),
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode
		]);
	}
	/**
	 * Post: Create Product Variations
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return Save response Array
	 */
	public function createVariations($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Variation', 'error'),
		];
		$variationResponse = $this->createProductVariations($request, $response);
		if (!empty($variationResponse)) {
			$jsonResponse = [
				'status' => 1,
				'product_id' => $variationResponse['product_id'],
				'variation_id' => $variationResponse['variation_id'],
				'message' => message('Variation', 'saved'),
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * Post: Validate Sku and Name
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return Validate response Array
	 */
	public function validateParams($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => 'Sorry, you can not proceed with this details',
		];
		if (strtolower(STORE_NAME) == "shopify") {
			$validationResponse = 0;
		} else {
			$validationResponse = $this->validateStoreSkuName($request, $response);
		}

		if ($validationResponse == 0) {
			$jsonResponse = [
				'status' => 1,
				'message' => 'You can use this combination for new product',
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * Get: Get Attribute List
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return A JSON Response
	 */
	public function getStoreAttributes($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Store Attributes', 'not_found'),
		];
		$attributeResponse = $this->storeAttributeList($request, $response);
		if (!empty($attributeResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $attributeResponse,
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * GET: Get Product Attribute Pricing by Product Id
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author satyabratap@riaxe.com
	 * @date   5 Oct 2019
	 * @return Json
	 */
	public function getProductAttrPrc($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
		];
		$productData = $this->storeProductAttrPrc($request, $response, $args);
		$printProfiles = [];
		if (!empty($productData)) {
			$productId = to_int($args['id']);
			$printProfileDetails = $this->getAssocPrintProfiles($productId);
			if (!empty($printProfileDetails['print_profiles'])) {
				$printProfiles = $printProfileDetails['print_profiles'];
			}
			$productAttributes = $this->getPriceDetails(
				$productData, $productId
			);
			$jsonResponse = [
				'status' => 1,
				'data' => [
					'print_profile' => $printProfiles,
					'attributes' => $productAttributes,
				],
			];
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}
	/**
	 * Save Product attribute prices
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author satyabratap@riaxe.com
	 * @date   13 Dec 2019
	 * @return boolean
	 */
	public function saveProdAttrPrc($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$allPostPutVars = $request->getParsedBody();
		$jsonResponse = [
			'status' => 0,
			'message' => message('Attribute Pricing', 'error'),
		];

		if (isset($allPostPutVars['price_data']) && !empty($allPostPutVars['price_data'])) {
			$attributePriceJson = $allPostPutVars['price_data'];
			$attributePriceArray = json_clean_decode($attributePriceJson, true);
			
			if (isset($attributePriceArray['product_id']) && $attributePriceArray['product_id'] > 0) {
				$productId = $attributePriceArray['product_id'];
				AttributePriceRule::where('product_id', $productId)->delete();
				if (
					isset($attributePriceArray['attributes'])
					&& count($attributePriceArray['attributes']) == 0
				) {
					$jsonResponse = [
						'status' => 1,
						'message' => message('Attribute Pricing', 'updated'),
					];
				} else {
					$success = 0;
					foreach ($attributePriceArray['attributes'] as $attributeData) {
						$attributeId = $attributeData['attribute_id'];
						if (
							isset($attributeData['attribute_term'])
							&& $attributeData['attribute_term'] != ""
						) {
							foreach ($attributeData['attribute_term'] as $attributeTermData) {
								$attributeTermId = $attributeTermData['attribute_term_id'];
								if (
									isset($attributeTermData['price_data'])
									&& $attributeTermData['price_data'] != ""
								) {
									foreach ($attributeTermData['price_data'] as $priceData) {
										$printProfileId = $priceData['print_profile_id'];
										$price = $priceData['price'];
										$attributePrices = [
											'product_id' => $productId,
											'attribute_id' => $attributeId,
											'attribute_term_id' => $attributeTermId,
											'print_profile_id' => $printProfileId,
											'price' => $price,
										];
										$saveAttributePrice = new AttributePriceRule(
											$attributePrices
										);
										if ($saveAttributePrice->save()) {
											$success++;
										}
									}
								}
							}
						}
					}
					if ($success > 0) {
						$jsonResponse = [
							'status' => 1,
							'message' => $success . ' Prices saved successfully',
						];
					}
				}
			}
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}
	/**
	 * Getting List of All color of a single product
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author satyabratap@riaxe.com
	 * @date   19 Sept 2019
	 * @return Array list of colors
	 */
	public function colorsByProductId($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'records' => 0,
			'data' => [],
			'message' => message('Colors', 'not_found')
		];
		// get the request data
		$productId = to_int($args['id']);
		$isAdmin = to_int($request->getQueryParam('isadmin', 0));
		$type = $request->getQueryParam('type');
		$allColors = isset($args['color']) ? $args['color'] : "";
		$getStoreDetails = get_store_details($request);
		$storeId = $getStoreDetails['store_id'];
		// get the sttribute names
		$attributeName = $this->getAttributeName();
		$attributeSlug = $attributeName['color'];
		// get the color details from the store
		$getColorResponse = $this->colorsByProduct($request, $response, [
			'product_id' => $productId, 'slug' => $attributeSlug, "color" => $allColors
		]);
		if (!empty($getColorResponse)) {
			// convert the image URL add proper host
			if ($storeId > 1 && $type == "tool" && IS_HOSTED == 0) {
				$currentStoreUrl = '';
				$databaseStoreInfo = DB::table('stores')->where('xe_id', '=', $storeId);
				if ($databaseStoreInfo->count() > 0) {
					$currentStoreUrl = $databaseStoreInfo->first()->store_url;
				}
				foreach ($getColorResponse as $key => $value) {
					$hostname = parse_url($value['sides'][0]['image']['src'], PHP_URL_HOST); // hostname
					$getColorResponse[$key]['sides'][0]['image']['src'] = str_replace($hostname, $currentStoreUrl, $value['sides'][0]['image']['src']);
					$getColorResponse[$key]['sides'][0]['image']['thumbnail'] = str_replace($hostname, $currentStoreUrl, $value['sides'][0]['image']['thumbnail']);
				}
			}
			$variantData = $this->getColorSwatchData($getColorResponse, [
				'combine' => 1, 'is_admin' => $isAdmin
			]);
			if (isset($args['return']) && $args['return'] == 1) {
                return array_values($variantData);
			}
			$jsonResponse = [
				'status' => 1,
				'records' => count($variantData),
				'data' => array_values($variantData)
			];
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

	/**
	 * Getting List of All Attributes
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author satyabratap@riaxe.com
	 * @date   31 Jan 2019
	 * @return Array list of colors
	 */
	public function getAttributeList($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Attributes', 'not_found'),
		];
		$attributeResponse = $this->getOnlyAttribute($request, $response);
		if (!empty($attributeResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $attributeResponse,
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * Getting Single Attribute Details by Attribute Id
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author satyabratap@riaxe.com
	 * @date   3 Mar 2019
	 * @return Array list of colors
	 */
	public function getAttributeDetails($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Attributes', 'not_found'),
		];
		$attributeResponse = $this->getAttributeTerms($request, $response, $args);
		if (!empty($attributeResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $attributeResponse,
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * Getting List of All Products for tool
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author satyabratap@riaxe.com
	 * @date   4 Feb 2019
	 * @return Array list of Products as per the categories
	 */
	public function getToolProductList($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Products', 'not_found'),
		];
		$productResponse = $this->getToolProducts($request, $response);
		if (!empty($productResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $productResponse,
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}
	/**
	 * Getting total product count from store
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author debashrib@riaxe.com
	 * @date   6 Feb 2019
	 * @return Number of products
	 */
	public function getTotalProductCount($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$storeDetails = get_store_details($request);
		$storeId = $storeDetails['store_id'] ? $storeDetails['store_id'] : 1;
		$count = $this->totalProductCount($storeId);

		return response($response, [
			'data' => $count, 'status' => $serverStatusCode,
		]);
	}
	/**
	 * Getting total product count from store
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author tanmayap@riaxe.com
	 * @date   6 Feb 2019
	 * @return json
	 */
	public function variantAttributeDetails($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Attribute quantity details', 'not_found'),
		];
		$attributeName = $this->getAttributeName();
		$colorName = $attributeName['color'];
		$getQuantityDetails = $this->storeVariantAttributeDetails(
			$request, $response, [
				'pid' => to_int($args['pid']), 'vid' => to_int($args['vid']), 'color_name' => $colorName,
			]
		);
		if (!empty($getQuantityDetails)) {
			$jsonResponse = [
				'status' => 1,
				'data' => [
					'quantities' => $getQuantityDetails,
				],
			];
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode,
		]);
	}

	/**
	 * Get Print profile, decoration status and product images by product ID
	 *
	 * @param $productId product id
	 *
	 * @author tanmayap@riaxe.com
	 * @date   5 Oct 2019
	 * @return array
	 */
	public function getAssocPrintProfiles($productId, $isAdmin = 0, $storeId = 1)
	{
		$printProfiles = $getSettings = [];
		$isDecorationExists = 0;
		if (!empty($productId)) {
			$settingId = 0;
			$productSettings = new \App\Modules\Products\Models\ProductSetting();
			$getSettings = $productSettings->where('product_id', $productId)->where('store_id', $storeId);

			if ($getSettings->count() == 0) {
				$getProductSettings = DB::table('product_settings_rel')
				->join('product_settings', 'product_settings.xe_id', '=', 'product_settings_rel.product_setting_id')
				->where('product_settings_rel.product_id', '=', $productId)
					->select('product_settings_rel.product_setting_id')->first();
				$settingId = $getProductSettings->product_setting_id;
				if (!empty($getProductSettings)) {
					$productSettings = new \App\Modules\Products\Models\ProductSetting();
					$getSettings = $productSettings->where('xe_id', $getProductSettings->product_setting_id)->where('store_id', $storeId);
				}
			}
			if ($getSettings->count() > 0) {
				$isDecorationExists = 1;
				$getSettings = new \App\Modules\Products\Models\ProductSetting();
				$productSetting = $getSettings
					->join(
						'print_profile_product_setting_rel',
						'product_settings.xe_id',
						'=',
						'print_profile_product_setting_rel.product_setting_id'
					)
					->join(
						'print_profiles',
						'print_profile_product_setting_rel.print_profile_id',
						'=',
						'print_profiles.xe_id'
					)
					->select('print_profiles.xe_id as id', 'print_profiles.name')
					->where('print_profiles.is_disabled', 0)
					->where('product_settings.store_id', $storeId);
				if ($settingId == 0) {
					$productSetting =  $productSetting->where('product_settings.product_id', $productId);
				} else {
					$productSetting =  $productSetting->where('product_settings.xe_id', $settingId);
				}
				$printProfiles = $productSetting->get()->toArray();
			} else {
				if (!$isAdmin) {
					/* Default print profile */
					$printProfiles = $this->getSingleEnbaledPrintProfile($storeId);
				}
			}
		}
		return [
			'print_profiles' => $printProfiles,
			'is_decoration_exists' => $isDecorationExists
		];
	}

	/**
	 * Get Asssociated images of a product from Database
	 *
	 * @param $productId product id
	 *
	 * @author tanmayap@riaxe.com
	 * @date   26 Mar 2020
	 * @return array
	 */
	public function getAssocProductImages($productId) {
		$assocProdImages = $getSettings = [];
		
		if (!empty($productId)) {
			$productSettings = new \App\Modules\Products\Models\ProductSetting();
			$getSettings = $productSettings->where('product_id', $productId);
			if (!$getSettings->count() > 0) {
				$productSettingRel = new \App\Modules\Products\Controllers\ProductDecorationsController();
				$getProductSettings = $productSettingRel->getSettingsIdByProductId($productId, []);
				if (!empty($getProductSettings)) {
					$productSettings = new \App\Modules\Products\Models\ProductSetting();
					$getSettings = $productSettings->where('xe_id', $getProductSettings->product_setting_id);
				}
			}
			if ($getSettings->count() > 0) {
				$productSetting = $getSettings->with(
					'print_profiles', 'print_profiles.profile'
				)
					->first();
				$prodImgSettRelObj = new \App\Modules\Products\Models\ProductImageSettingsRel();
				$prodImgSettRelInfo = $prodImgSettRelObj->where(
					'product_setting_id', $productSetting['xe_id']
				)
					->first();
				$productImageId = $prodImgSettRelInfo['product_image_id'];
				if ($productImageId > 0) {
					$prodImageSideObj = new \App\Modules\Products\Models\ProductImageSides();
					$getProductImageSideInit = $prodImageSideObj->where(
						'product_image_id',
						$productImageId
					)
						->get();
					$getProductImageSideInit = $prodImageSideObj->where(
						'product_image_id',
						$productImageId
					)
						->get();
					if (!empty($getProductImageSideInit)) {
						foreach ($getProductImageSideInit as $key => $productImage) {
							$assocProdImages[$key] = $productImage->thumbnail;
						}
					}
				}
			}
		}

		return $assocProdImages;
	}

	/**
	 * GET: Get Attribute Pricing Details
	 *
	 * @param $productDetails Product Details
	 * @param $productId      Product Id
	 *
	 * @author satyabratap@riaxe.com
	 * @date   13 Feb 2020
	 * @return Array of pricing
	 */
	public function getPriceDetails($productDetails, $productId) {
		$productAttributes = [];
		if (!empty($productDetails) && !empty($productId)) {
			$attributeName = $this->getAttributeName();
			foreach ($productDetails as $attrKey => $attributes) {
				$attrDetails = $attrTerms = [];
				if (!empty($attributes)) {
					foreach ($attributes['options'] as $termKey => $termValue) {
						$priceInit = new \App\Modules\Products\Models\AttributePriceRule();
						$getPriceData = $priceInit->where(['product_id' => $productId, 'attribute_id' => $attributes['id'], 'attribute_term_id' => $termValue['id']])
							->select('print_profile_id', 'price');
						$priceData = [];
						if (isset($getPriceData) && $getPriceData->count() > 0) {
							foreach ($getPriceData->get() as $priceKey => $priceValue) {
								$priceData[$priceKey] = [
									'print_profile_id' => $priceValue->print_profile_id,
									'price' => $priceValue->price,
								];
							}
						}
						$attrDetails = [
							'id' => $termValue['id'],
							'name' => $termValue['name'],
						];
						if (!empty($attributeName['color']) && !empty($attributes['name'])
								&& $attributes['name'] === $attributeName['color']) {
							if (STORE_NAME == 'Prestashop') {
								$attrDetails['hex_code'] = $termValue['hex_code'];
								$attrDetails['file_name'] = $termValue['file_name'];
							} else {
								$colorSwatchInit = new \App\Modules\Settings\Models\ColorSwatch();
								$getTermLocalData = $colorSwatchInit->where('attribute_id', $termValue['id'])
									->first();
								$attrDetails['hex_code'] = $attrDetails['file_name'] = "";
								if (!empty($getTermLocalData['hex_code'])
									&& $getTermLocalData['hex_code'] != ""
								) {
									$attrDetails['hex_code'] = $getTermLocalData['hex_code'];
								}
								if (!empty($getTermLocalData['file_name'])
									&& $getTermLocalData['file_name'] != ""
								) {
									$attrDetails['file_name'] = $getTermLocalData['file_name'];
								}
							}
							
						}
						$attrDetails['price_data'] = $priceData;
						$attrTerms[$termKey] = $attrDetails;
					}
				}
				$productAttributes[$attrKey] = [
					'id' => $attributes['id'],
					'name' => $attributes['name'],
					'attribute_term' => $attrTerms,
				];
			}
		}

		return $productAttributes;
	}

	/**
	 * GET: Get Multiple Attributes Variant Details
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 * @param $attribute     Attribute Name
	 *
	 * @author malay@riaxe.com
	 * @date   10th April 2020
	 * @return json
	 */
	public function multiAttributeVariantDetails($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Attribute quantity details', 'not_found')
		];
		// get the request params
		$productId = to_int($args['pid']);
		$attributeName = $request->getQueryParam('attribute');
		$price = $request->getQueryParam('price', 0);
		$isAttribute = $request->getQueryParam('is_attribute', 0);

		// get the store attributes
		$getAttributes = $this->getAttributeName();
		$colorAttributeSlug = $getAttributes['color'];
		$getQuantityDetails = $this->storeMultiAttributeVariantDetails($request, $response, [
			'pid' => to_int($args['pid']),
			'vid' => to_int($args['vid']),
			'attribute' => $attributeName,
			'price' => $price,
			'isAttribute' => $isAttribute
		]);
		if (!empty($getQuantityDetails)) {
			// populate the color swatch data if the requested attribute is color
			if ($colorAttributeSlug == $attributeName) {
				// get color swatches
				$getColorResponse = $this->colorsByProductId($request, $response, [
					"id" => $productId, "return" => 1, "color" => "all"
				]);
				// fill the color data matching with variant id
				$colorSwatchVariants = array_column($getColorResponse, 'variant_id');
				foreach ($getQuantityDetails[$attributeName] as $key => $value) {
					$attrKey = array_search($value['variant_id'], $colorSwatchVariants);
					if ($attrKey !== false) {
						$getQuantityDetails[$attributeName][$key]['attribute_id'] = $getColorResponse[$attrKey]['attribute_id'];
						$getQuantityDetails[$attributeName][$key]['hex_code'] = $getColorResponse[$attrKey]['hex_code'];
						$getQuantityDetails[$attributeName][$key]['file_name'] = $getColorResponse[$attrKey]['file_name'];
						$getQuantityDetails[$attributeName][$key]['color_type'] = $getColorResponse[$attrKey]['color_type'];
					}
				}
			}
			$jsonResponse = [
				'status' => 1,
				'quantities' => $getQuantityDetails
			];
		}
		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

	public function getShopifyParentProduct($request, $response, $args) {
		$thisVarID = $args['vid'];
		return $this->getParentProductData($thisVarID);
	}

	/**
	 * Get: Get Product Variant
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author radha@riaxe.com
	 * @date   23 Jul 2020
	 * @return A JSON Response
	 */
	public function getProductVariant($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Store Product Variant', 'not_found'),
		];
		$variantResponse = $this->storeProductVariant($request, $response);
		if (!empty($variantResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $variantResponse['variant_id'],
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}

	/**
	 * GET: Get All product variants of a product
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author debashisd@riaxe.com
	 * @date   23rd July 2020
	 * @return json
	 */
	public function getProductVariants($request, $response, $args, $returnType = 0) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('variants of the products', 'not_found'),
		];
		$getStoreDetails = get_store_details($request);
        $storeId = $getStoreDetails['store_id'];
		$productId = to_int($args['pid']);
		$result = $this->productVariants($productId, $storeId);
		$productVariants = $result['variants'];
		if (!empty($productVariants)) {
			$jsonResponse = [
				'status' => 1,
				'variants' => $productVariants
			];
		}
		if ($returnType == 1) {
			return $productVariants;
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode,
		]);
	}

	/**
	 * GET: Get All product variants of a product
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author debashisd@riaxe.com
	 * @date   23rd July 2020
	 * @return json
	 */
	public function getTierPricing($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('tier Pricing rules', 'not_found'),
		];
		$productId = to_int($args['pid']);
		$tierDiscounts = $this->productTierDiscounts(
			$request, $response, [
				'productID' => $productId,
			]
		);
		/** for sticker */
		$decorationType = '';
		$isVariableDecoration = '';
		$isCustomSize = '';
		$productSettingInit = new ProductSetting();
		$getProductSetting = $productSettingInit->select('decoration_type','is_variable_decoration','is_custom_size')->where(['product_id' => $productId ])->first();
		if(count((array)$getProductSetting) > 0) {
			$productSettingData =  $getProductSetting->toArray();
			$decorationType = $productSettingData['decoration_type'];
			$isVariableDecoration = $productSettingData['is_variable_decoration'];
			$isCustomSize = $productSettingData['is_custom_size'];
		}
		if (!empty($tierDiscounts)) {
			$jsonResponse = $tierDiscounts;
		}
		$jsonResponse['decoration_type'] = $decorationType;
		$jsonResponse['is_variable_decoration'] = $isVariableDecoration;
		$jsonResponse['is_custom_size'] = $isCustomSize;
		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode,
		]);
	}

	/**
	 * POST: Save product tier Discount Rules
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author debashisd@riaxe.com
	 * @date   23rd July 2020
	 * @return json
	 */
	public function saveTierPricing($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('tier Pricing rules', 'not_found'),
		];
		$productId = to_int($args['pid']);
		/*for custom size*/
		$this->saveTierCustomSize($request,$response, ['product_id' => $productId]);
		$saveTierDiscount = $this->saveTierDiscount(
			$request, $response, [
				'productID' => $productId,
			]
		);
		if ($saveTierDiscount) {
			$jsonResponse = [
				'status' => 1,
				'message' => "Tier price Discounts are saved for this product",
			];
		}

		return response($response, [
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}

	/**
	 * GET: Get Single Print Profile Record
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's arg object
	 *
	 * @author radhanatham@riaxe.com
	 * @date   8 Dec 2020
	 * @return A JSON Response
	 */
	private function getSingleEnbaledPrintProfile($storeId) {
		
		$jsonResponse = [];
		$printProfileInit = new PrintProfileModels\PrintProfile();
		$getPrintProfileInfo = $printProfileInit->where(
			[
				'is_disabled' => 0,
				'store_id' => $storeId,
			]
		)->first();

		// Check if print profile exist in this ID
		if (!empty($getPrintProfileInfo->xe_id)) {
			$jsonResponse[] = [
				'id' => $getPrintProfileInfo->xe_id,
				'name' => $getPrintProfileInfo->name,
			];
		}
		return $jsonResponse;

	}

	/**
	 * GET: Getting List of All Category/Subcategory or Single
	 * Category/Subcategory information
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument parameters
	 *
	 * @author mukeshp@riaxe.com
	 * @date   1 March 2021
	 * @return A JSON Response
	 */
	public function CategoriesSubcatagories($request, $response, $args) {
		$serverStatusCode = OPERATION_OKAY;
		$storeResponse = [
			'status' => 0,
			'data' => [],
		];
		$apiSource = $request->getQueryParam('src');
		// get selected categories only
		$storeId = $request->getQueryParam('store_id') ? $request->getQueryParam('store_id') : 1;
		$settingLocation = path('abs', 'setting') . 'stores/' . $storeId;
		$jsonFilePath = $settingLocation . '/product_categories.json';
		if (file_exists($jsonFilePath)) {
			$logContent = json_decode(file_get_contents($jsonFilePath), true);
			$selectedCatagories = array_column($logContent['data'], 'id');
		}
		$storeResponse = $this->getCategoriesSubcategories($request, $response, $args);
		$categoriesForTool = array();
		foreach ($storeResponse as $key => $category) {
			$storeResponse[$key]['show_in_tool'] = false;
			if (in_array($category['id'], $selectedCatagories)) {
				$storeResponse[$key]['show_in_tool'] = true;
				$categoriesForTool[] = $category;
			}
		}
		if ($apiSource == 'tool' && !empty($selectedCatagories)) {
			$storeResponse = $categoriesForTool;
		}
		if (!empty($storeResponse)) {
			$storeResponse = [
				'status' => 1,
				'data' => $storeResponse,
			];
		} else {
			$storeResponse = [
				'status' => 0,
				'data' => [],
			];
		}

		return response($response, [
			'data' => $storeResponse, 'status' => $serverStatusCode,
		]);
	}

	/**
	 * Post: Create Product Catagory
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 *
	 * @author mukeshp@riaxe.com
	 * @date   2 March 2021
	 * @return Save response Array
	 */
	public function saveCategories($request, $response) {
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => message('Catagories', 'error'),
		];
		$categoryResponse = $this->createProductCatagories($request, $response);
		if (!empty($categoryResponse)) {
			$jsonResponse = $categoryResponse;
		}

		return response($response,[
			'data' => $jsonResponse,
			'status' => $serverStatusCode,
		]);
	}

	/**
     * Delete: Delete Product Catagories(s)
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author mukeshp@riaxe.com
     * @date   3 Mar 2021
     * @return json
     */
    public function deleteCategories($request, $response, $args)
    {
        $serverStatusCode = OPERATION_OKAY;
        $storeResponse = [
            'status' => 0,
            'message' => message('Catagories Delete', 'error'),
        ];
        if (!empty($args['id'])) {
        	$storeResponse = $this->removeCategories($request, $response, $args);
        }

		return response($response, [
			'data' => $storeResponse, 'status' => $serverStatusCode
		]);
	}
	/**
     * GET: Product decoration type
     *
     * @param $productId
     * @param $storeId
     *
     * @author soumyas@riaxe.com
     * @date   24 May 2021
     * @return string
     */
    public function getProductDecorationType($productId , $storeId) {
    	$decorationType = '';
    	$getSettingsInit = new ProductSetting;
		$settingDetails = $getSettingsInit->select('decoration_type')->where(['product_id' => $productId,'store_id' => $storeId]);
		if ($settingDetails->count() > 0){
			$settingDetailsData = $settingDetails->get()->toArray();
			$decorationType = $settingDetailsData[0]['decoration_type'] ? $settingDetailsData[0]['decoration_type']:'';
		}
		return $decorationType;
    }
	 /**
     * GET: store variants
     *
     * @param slim parameters
     * @param $storeId
     *
     * @author divya@imprintnext.com
     * @date   24 May 2021
     * @return string
     */
    public function getStoreVariant($request, $response, $args){
        $serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'data' => [],
			'message' => message('Attribute quantity details', 'not_found'),
		];
		$attributeResponse = $this->variantOptions(
			$request, $response, [
				'pid' => to_int($args['pid'])
			]
		);
		if (!empty($attributeResponse)) {
			$jsonResponse = [
				'status' => 1,
				'data' => $attributeResponse,
			];
		}
		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode,
		]);
    }

    /**
     * GET: Product decoration type
     *
     * @param $productId
     * @param $storeId
     *
     * @author steve@imprintnext.com
     * @date   24 May 2021
     * @return string
     */
    public function getProductMetaDetails($request, $response, $args){
		$serverStatusCode = OPERATION_OKAY;
    	$productId = $args['id'];
		$storeResponse = $this->getStoreProductMetaDetails($productId);
		return response(
            $response, ['data' => $storeResponse, 'status' => $serverStatusCode]
        );
	}

	/**
	 * POST: Create / Update product options
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */

	public function saveVariantOptions($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => 'Some error occurred while saving the product variants'
		];

		try {
			$allPostPutVars = $request->getParsedBody();
			$productId = $args['product_id'];
			$optionData = !empty($allPostPutVars['options']) ? json_decode($allPostPutVars['options'], true) : [];
			$variantData = !empty($allPostPutVars['variants']) ? json_decode($allPostPutVars['variants'], true) : [];

			// save the option data
			$this->saveOptionData($productId, $optionData);

			$optionCounts = (new ProductOptionRel())->select(DB::raw('JSON_LENGTH(`option_values`) as option_count'))
			->where('product_id', '=', $productId)->get()->pluck('option_count')->toArray();

			// max combination allowed - count
			$combinationTotal = (int) array_product($optionCounts);

			if ($combinationTotal && !empty($variantData)) {

				// update the inventory tracking flag, mark variants as custom
				$settingFields = [
					'dnt_inventory' => isset($allPostPutVars['dnt_inventory']) ? intVal($allPostPutVars['dnt_inventory']) : 0,
					'custom_variant' => 1
				];
				$productSettingsObj = new ProductSetting();
				$productSettingsObj->where('product_id', '=', $productId)->update($settingFields);

				$productVariantRelInit = new ProductVariantRel();
				$matchingVariants = $productVariantRelInit->select('variant_id')->where('product_id', '=', $productId);
				// get the existing product variants
				$productVariantObj = new ProductVariants();
				$oldProductVars = $productVariantObj->whereIn('xe_id', $matchingVariants)->get();

				$successCount = 0;

				// last key to add newly created variants
				$lastKey = 0;

				// update / delete (if non matching) the old variants
				foreach ($oldProductVars as $key => $oldProductVar) {
					if (isset($variantData[$key]) && $combinationTotal > 0) {
						$lastKey = $key;
						$oldProductVar->fill($variantData[$key]);
						$oldProductVar->save();
						++$successCount;
						--$combinationTotal;
						continue;
					}
					// delete the variant relation
					(new ProductVariantRel())->where('variant_id', '=', $oldProductVar->xe_id)->delete();

					// delete the variant
					$oldProductVar->delete();
				}

				// add new variants
				for ($i = $lastKey + 1; $i < count($variantData) && $combinationTotal > 0; $i++) {
					$newProductVar = new ProductVariants($variantData[$i]);
					if ($newProductVar->save()) {
						$newProductVarRel = new ProductVariantRel(['product_id' => $productId, 'variant_id' => $newProductVar->xe_id]);
						$newProductVarRel->save();

						++$successCount;
						--$combinationTotal;
					}
				}
				if ($successCount && $successCount <= count($variantData)) {
					$this->saveProductAPIasCache($productId);
					$jsonResponse = ['status' => 1, 'message' => 'Variants saved successfully'];
				}
			} else {
				(new ProductVariants())->whereHas('product_variant_rel', function ($q) use ($productId) {
					return $q->where('product_id', '=', $productId);
				})->delete();
				(new ProductVariantRel())->where('product_id', '=', $productId)->delete();

				$jsonResponse = ['status' => 1, 'message' => 'Variants saved successfully'];
			}
		} catch (Exception $e) {
			$serverStatusCode = EXCEPTION_OCCURED;
			$jsonResponse = ['status' => 0, 'message' => $e->getMessage()];
			create_log('product_variants', 'error', [
				'message' => $e->getMessage(),
				'extra' => [
					'module' => 'product variant options'
				]
			]);
		}

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

	/**
	 * GET: get product option list
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */
	public function getVariantOptions($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonData = ['options' => [], 'variants' => []];

		$productId = $args['product_id'];
		$filter = $request->getQueryParam('filter');
		
		if ($filter == 'options') {
			return $this->getToolVariantOptions($request, $response, $args);
		}

		$productOptionRelObj = new ProductOptionRel();
		$productOptions = $productOptionRelObj->where('product_id', '=', $productId)
			->whereHas('option')->with('option')->orderBy('position', 'ASC')->get()->toArray();
		
		if (!empty($productOptions)) {
			foreach ($productOptions as $productOption) {
				$options = $productOption['option'];
				$options['position'] = $productOption['position'];
				$options['values'] = $productOption['option_values'];
				$jsonData['options'][] = $options;
			}

			// get the existing product variants
			$productVariantObj = new ProductVariants();
			$getProductVariants = $productVariantObj->join('product_variant_rel', 'product_variants.xe_id', '=', 'product_variant_rel.variant_id')
			->where('product_variant_rel.product_id', '=', $productId)->select('product_variants.*')->get()->toArray();

			if (!empty($getProductVariants)) {
				$jsonData['variants'] = $getProductVariants;
			}

			// get the inventory tracking flag from product setting
			$productSetting = (new ProductSetting())->where('product_id', '=', $productId)->select('dnt_inventory')->get()->first();
			$jsonData['dnt_inventory'] = isset($productSetting->dnt_inventory) ? $productSetting->dnt_inventory : 0;
		}

		$jsonResponse = ['status' => 1, 'data' => $jsonData];

		return response($response, [
			'data' => $jsonResponse, 'status' => $serverStatusCode
		]);
	}

	/**
	 * Internal: save product option
	 *
	 * @param $productId  Integer - Product ID
	 * @param $optionData Array - Option data
	 *
	 * @return Integer $combinationCount
	 */
	private function saveOptionData($productId, $optionData)
	{
		if (empty($optionData)) {
			(new ProductOptionRel())->select('xe_id')->where('product_id', '=', $productId)->delete();
			return true;
		}

		// delete the non matching product options realation
		$optionNames = array_column($optionData, 'name');
		$deletebaleOptions = (new ProductOptionRel())->select('xe_id')->where('product_id', '=', $productId);

		if (!empty($optionNames)) {
			$deletebaleOptions->whereHas('option', function ($q) use ($optionNames) {
				return $q->whereNotIn('name', $optionNames);
			});
		}

		if ($deletebaleOptions->count()) {
			$deletebaleOptions->delete();
		}

		$successCount = 0;
		// add new product option and option relation
		foreach ($optionData as $key => $option) {
			$option['name'] = trim($option['name']);
			$productOptionObj = new ProductOptions();
			$getOptionObj = $productOptionObj->where(['name' => $option['name']]);
			if ($getOptionObj->count()) {
				$productOptionObj = $getOptionObj->get()->first();
			} else {
				$productOptionObj->fill($option);
				// skip if unable to save
				if (!$productOptionObj->save()) {
					continue;
				}
			}
			// add / update the option values
			if (!empty($option['values'])) {
				$optionRelData = [
					'product_id' => $productId,
					'option_values' => $option['values'],
					'position' => isset($option['position']) ? intval($option['position']) : $key
				];

				$optionRelObj = new ProductOptionRel();
				$checkOptionRel = $optionRelObj->where(['option_id' => $productOptionObj->xe_id, 'product_id' => $productId]);
				if ($checkOptionRel->count()) {
					$optionRelObj = $checkOptionRel->get()->first();
				} else {
					$optionRelData['option_id'] = $productOptionObj->xe_id;
				}
				$optionRelObj->fill($optionRelData);
				if ($optionRelObj->save()) {
					$successCount++;
				}
			}
		}

		return ($successCount == count($optionData));
	}

	/**
	 * GET: Product attributes  fron the shopify store
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */
	public function getStoreAttr($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;

		// add product id with params;
		$productId = $args['product_id'];
		$request = $request->withQueryParams(
			$request->getQueryParams() + ['product_id' => $productId, 'fetchRaw' => true]
		);

		// get the store attribute list
		$attributes = $this->storeAttributeList($request, $response);

		// format the output
		$jsonData = ['options' => []];
		if (!empty($attributes)) {
			foreach ($attributes as $key => $attribute) {
				$jsonData['options'][] = [
					'name' => $attribute['name'],
					'values' => !empty($attribute['terms']) ? array_column($attribute['terms'], 'name') : [],
					'position' => $key
				];
			}
			$jsonResponse = ['status' => 1, 'data' => $jsonData];
		}

		return response($response, [
			'data' => $jsonResponse, 
			'status' => $serverStatusCode
		]);
	}

	/**
	 * GET: Invenory flag, Inventory count
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */
	public function impInventory($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => 'Inventory not found!'
		];

		$productId = $args['product_id'];
		$title = $request->getQueryParam('title');

		$productSettings = (new ProductSetting())->where('product_id', '=', $productId)->get()->first();
		$inventoryField = !empty($productSettings->dnt_inventory) ? "CONCAT('1000')" : "`product_variants`.`inventory`";

		// get the inventory from the variant
		$productVariantObj = new ProductVariants();
		$getProductVariants = $productVariantObj->whereHas('product_variant_rel', function ($q) use ($productId) {
			return $q->where('product_id', '=', $productId);
		})
		->where('title', '=', $title)
		->select([
			DB::raw("$inventoryField as inventory"),
			DB::raw("CONCAT('{$productSettings->dnt_inventory}') as dnt_inventory")
		]);

		$productInventory = $getProductVariants->get()->first();

		if (!empty($productInventory)) {
			$jsonResponse = ['status' => 1, 'data' => $productInventory];
		}

		return response($response, [
			'data' => $jsonResponse, 
			'status' => $serverStatusCode
		]);
	}

	/**
	 * GET: get product option list for tool
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */
	public function getToolVariantOptions($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;

		$jsonData = ['options' => [], 'dnt_inventory' => 0];
		$productId = $args['product_id'];

		// get the variant detail for the cache
		$productCacheFile = SHOPIFY_CACHE_FOLDER . $productId . ".json";
		if (file_exists($productCacheFile)) {
			$product = json_decode(file_get_contents($productCacheFile), true);
			$jsonData['options'] = $product['options'];
			$jsonData['variants'] = $product['variants'];
			$jsonData['dnt_inventory'] = $product['inventory_policy'] == 'continue' ? 1 : 0;
		}

		$jsonResponse = [
			'status' => !empty($jsonData['options']) ? 1 : 0,
			'data' => $jsonData
		];

		return response($response, [
			'data' => $jsonResponse, 
			'status' => $serverStatusCode
		]);
	}

	/**
	 * GET: first variant of matching color
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   27 Dec 2022
	 * @return JSON $response
	 */
	public function getFirstColorVariant($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => 'Some error occurred while getting the variant'
		];

		$productId = $args['product_id'];
		$colorName = $request->getQueryParam('color');

		// get the variant detail for the cache
		$productCacheFile = SHOPIFY_CACHE_FOLDER . $productId . ".json";
		if (file_exists($productCacheFile)) {
			$product = json_decode(file_get_contents($productCacheFile), true);

			// get all attributes, position
			$attributeNames = array_column($product['options'], 'name', 'position');
			array_walk($attributeNames, function (&$name) {
				$name = strtolower($name);
			});
			
			// get the color position
			$colorPos = array_search('color', $attributeNames);
			
			if ($colorPos !== false) {
				foreach ($product['variants'] as $variant) {
					if ($variant['option' . $colorPos] == $colorName) {
						$jsonResponse = [
							'status' => 1, 'variant_id' => $variant['id']
						];
						break;
					}
				}
			}
		} else {
			$jsonResponse['message'] = 'The product details not found on the ImprintNext';
		}

		return response($response, [
			'data' => $jsonResponse, 
			'status' => $serverStatusCode
		]);
	}


	/**
	 * GET: update the product cache
	 *
	 * @param $request  Slim's Request object
	 * @param $response Slim's Response object
	 * @param $args     Slim's Argument object
	 *
	 * @author dan@imprintnext.com
	 * @date   16 Nov 2021
	 * @return JSON $response
	 */
	public function updateProductCache($request, $response, $args)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = ['status' => 0, 'message' => 'Some error occurred!'];

		$productId = $request->getQueryParam('product_id');
		if (!empty($productId)) {
			$this->saveProductAPIasCache($productId);
			$jsonResponse = ['status' => 1, 'message' => 'Cache updated successfully'];
		}
		
		return response($response, [
			'data' => $jsonResponse, 
			'status' => $serverStatusCode
		]);
	}
	
	/**
     * Save: custom size for sticker
     *
     * @param $request
     * @param $response
	 * @param  $args
     *
     * @author soumyas@riaxe.com
     * @date   28 September 2022
     * @return boolean
     */
	private function saveTierCustomSize($request, $response, $args) {
		$isCustomSize = false;
		$allPostPutVars = $request->getParsedBody();
		$customSizeArray = [];
		$productId = $args['product_id'];
		if($allPostPutVars['decoration_type'] == "banner" && $allPostPutVars['is_variable_decoration']== 1) {
			$customSizeArray['pricing_per_variants'] = $allPostPutVars['pricing_per_variants'];
			$customSizeArray['discount_type'] = $allPostPutVars['discount_type'];
			if($allPostPutVars['pricing_per_variants'] == 'false' ) {
				$priceRules = json_clean_decode($allPostPutVars['price_rules'], true);
				$customSizeData = '';
				if(!empty($priceRules)) {
					foreach($priceRules as $value) {
						if ($value['id'] == 0) {
							$customSizeData = $value['discounts'];
						}
					}
				}
				if(!empty($customSizeData)) {
					$customSizeArray['discounts']= $customSizeData;
				}
			} else {
				$priceRules = json_clean_decode($allPostPutVars['price_rules'], true);
				if(!empty($priceRules)) {
					$customSizeArray['discounts'] =  $priceRules[0]['discounts'];
				}
			}
		}
		$productSettingInit = new ProductSetting();
		$getProductSetting = $productSettingInit->where(['product_id' => $productId])->first();
		if($getProductSetting->count() > 0) {
			$productData = ['custom_size' => json_clean_encode($customSizeArray)];
			$productSettingInit->where(['product_id' => $productId])->update($productData);
			$isCustomSize = true;
		}
		return $isCustomSize;
	}


	private function  saveProductsSetting($settingDetails, $predecoId, $parentId)
	{
		$productSettData = [
			'product_id' => $predecoId,
			'store_id' => $settingDetails['store_id'],
			'is_variable_decoration' => $settingDetails['is_variable_decoration'],
			'is_ruler' => $settingDetails['is_ruler'],
			'is_crop_mark' => $settingDetails['is_crop_mark'],
			'is_safe_zone' => $settingDetails['is_safe_zone'],
			'crop_value' => $settingDetails['crop_value'],
			'safe_value' => $settingDetails['safe_value'],
			'is_3d_preview' => $settingDetails['is_3d_preview'],
			'3d_object_file' => $settingDetails['3d_object_file'],
			'3d_object' => $settingDetails['3d_object'],
			'is_configurator' => $settingDetails['is_configurator'],
			'scale_unit_id' => $settingDetails['scale_unit_id'],
			'is_configurator' => $settingDetails['is_configurator'],
			'is_svg_configurator' => $settingDetails['is_svg_configurator'],
		];

		$productSetting = new ProductSetting($productSettData);
		$productSetting->save();
		$clonedSettingId = $productSetting->xe_id;
		// FOR SVG CONFIGURATOR
		if ($settingDetails['is_svg_configurator']) {
			$configuratorInit = new Configurator();
			$configuratorInit->cloneSVGProductConfigurator($parentId, $predecoId);
		}
		return $clonedSettingId;
	}

	private function cloneProductImageSettingsRel($productSettingId, $clonedSettingId)
	{
		$getImgRelInit = new ProductImageSettingsRel();
		$imgRelData = $getImgRelInit->where(['product_setting_id' => $productSettingId])->first();
		if (!empty($imgRelData)) {
			$imgRelData = $imgRelData->toArray();
			$saveRelDetails = [
				'product_setting_id' => $clonedSettingId,
				'product_image_id' => $imgRelData['product_image_id'],
			];

			$productImageSettings = new ProductImageSettingsRel($saveRelDetails);
			$productImageSettings->save();
			return true;
		}
	}

	private function colonePrintProfileProductSettingRel($productSettingId, $clonedSettingId)
	{
		$profileDecoRelInit = new PrintProfileProductSettingRel();
		$getProfSetInit = $profileDecoRelInit->where(['product_setting_id' => $productSettingId])->get();
		if (!empty($getProfSetInit)) {
			$getProfSetInit = $getProfSetInit->toArray();
			foreach ($getProfSetInit as $settProfile) {
				$saveSettProfile = new PrintProfileProductSettingRel([
					'print_profile_id' => $settProfile['print_profile_id'],
					'product_setting_id' => $clonedSettingId,
				]);
				$saveSettProfile->save();
			}
		}
	}

	private function cloneDecorationObjects($parentId, $predecoId)
	{
		$get3DObjInit = new DecorationObjects();
		$objData = $get3DObjInit->where(['product_id' => $parentId])->first()->toArray();

		if (!empty($objData['3d_object_file'])) {
			$objFile = str_replace(path('read', '3d_object'), '', $objData['3d_object_file']);
		}
		if (!empty($objData['uv_file'])) {
			$uvFile = str_replace(path('read', '3d_object'), '', $objData['uv_file']);
		}
		$objDetails = [
			'product_id' => $predecoId,
			'3d_object_file' => $objFile,
			'uv_file' => $uvFile,
		];
		$saveObjData = new DecorationObjects($objDetails);
		$saveObjData->save();
		return true;
	}

	private function cloneProductSide($sideData, $clonedSettingId, $productSettingId, $productSideImages)
	{		
		$prodImgSettRelObj = new ProductImageSettingsRel();
        $checkForProductImage = $prodImgSettRelObj->where('product_setting_id', $productSettingId)->first();
        $productImgId = 0;
        if (isset($checkForProductImage['product_image_id'])) {
			$productImgId = to_int($checkForProductImage['product_image_id']);
        }
		$sideDataArr = [];
		$sideData = $sideData->toArray();
		$newSdeData = [];
		foreach ($sideData as $svalue) {
			if($svalue['is_visible'] == 1){
				$newSdeData[]= $svalue;
			}
		}		
		$sizeofSideData = sizeof($newSdeData);
		$duplicatesideData = [];
		if (STORE_NAME != 'Shopify' && $productImgId == 0) {
			foreach ($productSideImages as $key => $productImageSideData) {
				if ($key >= $sizeofSideData && ($key - $sizeofSideData) <= $sizeofSideData && $key != 0 && isset($sideData[$key - $sizeofSideData])) {
					$duplicatesideData[$key] = $sideData[$key - $sizeofSideData];
				} else {
					$duplicatesideData[$key] = $sideData[0];
					$duplicatesideData[$key]['is_visible'] = 0;
				}
			}
		}
		$sideDataArr = !empty($duplicatesideData) ? $duplicatesideData : $sideData;
		foreach ($sideDataArr as $productSideData) {

			$productSide = new ProductSide(
				[
					'product_setting_id' => $clonedSettingId,
					'side_name' => $productSideData['side_name'],
					'product_image_dimension' => $productSideData['product_image_dimension'],
					'is_visible' => $productSideData['is_visible'],
					'product_image_side_id' => $productSideData['product_image_side_id'],
				]
			);
			$productSide->save();
			$clonedSideId = $productSide->xe_id;
			$getDecorationInit = new ProductDecorationSetting();
			$getDecorationData = $getDecorationInit->where([
				'product_setting_id' => $productSettingId,
				'product_side_id' => $productSideData['xe_id']
			])->get();
			if (!empty($getDecorationData)) {
				$getDecorationData = $getDecorationData->toArray();
				foreach ($getDecorationData as $decoration) {
					$saveDecoration = new ProductDecorationSetting([
						'product_setting_id' => $clonedSettingId,
						'product_side_id' => $clonedSideId,
						'name' => $decoration['name'],
						'dimension' => $decoration['dimension'],
						'print_area_id' => $decoration['print_area_id'],
						'sub_print_area_type' => $decoration['sub_print_area_type'],
						'pre_defined_dimensions' => $decoration['pre_defined_dimensions'],
						'user_defined_dimensions' => $decoration['user_defined_dimensions'],
						'custom_min_height' => $decoration['custom_min_height'],
						'custom_max_height' => $decoration['custom_max_height'],
						'custom_min_width' => $decoration['custom_min_width'],
						'custom_max_width' => $decoration['custom_max_width'],
						'custom_bound_price' => $decoration['custom_bound_price'],
						'is_border_enable' => $decoration['is_border_enable'],
						'is_sides_allow' => $decoration['is_sides_allow'],
						'no_of_sides' => $decoration['no_of_sides'],
						'is_dimension_enable' => $decoration['is_dimension_enable'],
						'locations' => $decoration['locations'],
						'is_disable_design' => $decoration['is_disable_design'],
						'image_overlay' => $decoration['image_overlay'],
						'multiply_overlay' => $decoration['multiply_overlay'],
						'overlay_file_name' => $decoration['overlay_file_name'],
						'bleed_mark_data' => $decoration['bleed_mark_data'],
						'shape_mark_data' => $decoration['shape_mark_data'],
						'default_pp_id' => $decoration['default_pp_id'],
					]);
					$saveDecoration->save();
					$clonedDecoId = $saveDecoration->xe_id;

					$profileDecoRelInit = new PrintProfileDecorationSettingRel();
					$getProfileDecoRel = $profileDecoRelInit->where([
						'decoration_setting_id' => $decoration['xe_id']
					])->get();
					if (!empty($getProfileDecoRel)) {
						$getProfileDecoRel = $getProfileDecoRel->toArray();
						foreach ($getProfileDecoRel as $decoProfile) {
							$saveProfile = new PrintProfileDecorationSettingRel([
								'print_profile_id' => $decoProfile['print_profile_id'],
								'decoration_setting_id' => $clonedDecoId
							]);
							$saveProfile->save();
						}
					}
				}
			}
		}
	}

	/**
	 * delete/archieve all abondoned cart Products from Shopify store
	 *
	 * @param $status,$days
	 *
	 * @author pansy@imprintnext.com
	 * @date  06 Feb 2024
	 */

	public function deleteDuplicateProduct($request, $response)
	{
		$serverStatusCode = OPERATION_OKAY;
		$jsonResponse = [
			'status' => 0,
			'message' => 'Something went wrong'
		];
		$storeResponse = $this->deleteStoreDuplicateProducts($request);
		if (!empty($storeResponse)) {
			$jsonResponse = $storeResponse;
		}
		return response($response, [
			'status' => $serverStatusCode,
			'data' => $jsonResponse
		]);
	}

	/**
     * Get: Get Product Setting
     *
     * @param $productId  Product id
     *
     * @author mike@imprintnext.com
     * @date   15 April 2024
     * @return Setting Object
     */
    public function getSettingsByBulkDecorationSavingProduct($productId)
    {
        // get the settings data from product settings rel
        $settingData = DB::table('product_settings_rel')
        ->join('product_settings', 'product_settings.xe_id', '=', 'product_settings_rel.product_setting_id')
        ->where('product_settings_rel.product_id', '=', $productId)
            ->select([
                'product_settings.product_id', 'product_settings_rel.product_setting_id',
                'product_settings_rel.is_3d_preview', 'product_settings_rel.is_product_image'
            ])->first();
        return $settingData;
    }
}
