<?php
/**
 * This Controller used to save, fetch or delete Prestashop Products on various
 * endpoints
 *
 * PHP version 5.6
 *
 * @category  Prestashop_API
 * @package   Store
 * @author    Radhanatha <radhanatham@riaxe.com>
 * @copyright 2019-2020 Riaxe Systems
 * @license   http://www.php.net/license/3_0.txt  PHP License 3.0
 * @link      http://inkxe-v10.inkxe.io/xetool/admin
 */
namespace ProductStoreSpace\Controllers;

use ComponentStoreSpace\Controllers\StoreComponent;

/**
 * Store Product Controller
 *
 * @category Prestashop_API
 * @package  Store
 * @author   Radhanatha <tanmayap@riaxe.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://inkxe-v10.inkxe.io/xetool/admin
 */
class StoreProductsController extends StoreComponent
{
    /**
     * Get list of product or a Single product from the Prestashop API
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return Array of list/one product(s)
     */
    public function getProducts($request, $response, $args)
    {
        $storeResponse = [];
        $products = $productCategories = [];
        $productArray = $productColourVariations = [];
        
        $shopId = $request->getQueryParam('store_id')
        ? $request->getQueryParam('store_id') : 1;

        $customerId = $request->getQueryParam('customer', 0);

        if (isset($args['id']) && $args['id'] != "" && $args['id'] > 0) {
            $sanitizedProduct = [];
            // For fetching Single Product
            $productId = $args['id'];
            $parameters = array(
                'resource' => 'products', 'display' => 'full',
                'filter[id]' => '[' . $productId . ']',
                'id_shop' => $shopId,
                'output_format' => 'JSON',
                'language' => '' . $this->languageId . '',
            );

            try {
                $result = $this->webService->get($parameters);
                $productObj = json_decode($result, true);
                $product = $productObj['products'][0];
                $productParameter = array(
                    'product_id' => $productId,
                );
                $productStock = 0;
                $productCombinationStock = $product['associations']['combinations'];
                if (!empty($productCombinationStock)) {
                    foreach ($productCombinationStock as $combination) {
                        $productStock += $this->webService->getProductStock(0,
                            $combination['id']
                        );
                    }
                } else {
                    $productStock += $this->webService->getProductStock($productId, 0);
                }
                if (!empty($product['associations']['categories'])) {
                    foreach ($product['associations']['categories'] as $k => $v) {
                        $category = $this->getCategories(
                            $request, $response, [
                                'id' => $v['id'],
                            ]
                        );
                        $productCategories[$k]['id'] = $category[0]['id'];
                        $productCategories[$k]['name'] = $category[0]['name'];
                        $productCategories[$k]['slug'] = $category[0]['slug'];
                        if ($category[0]['parent_id'] == 2 || $category[0]['parent_id'] == 1) {
                            $parentId = 0;
                        } else {
                            $parentId = $category[0]['parent_id'];
                        }
                        $productCategories[$k]['parent_id'] = $parentId;
                    }
                }
                if (is_array($product) && !empty($product)) {
                    $variantId = $product['id_default_combination'];
                    $sanitizedProduct['id'] = $product['id'];
                    $sanitizedProduct['name'] = $product['name'];
                    $sanitizedProduct['variant_id'] = $variantId == 0
                    ? $productId : $variantId;
                    $sanitizedProduct['type'] = $variantId == 0 ? 'simple' : 'variable';
                    if ($product['id_default_combination'] > 0) {
                        $price = $this->webService->getCombinationPrice($variantId, $productId);	
                    } else {
                        $price = $product['price'];
                    }
                    $sanitizedProduct['price'] = $price;
                    $sanitizedProduct['tax'] = $this->webService->getTaxRate($productId, $customerId);
                    $sanitizedProduct['sku'] = $product['reference'];
                    $sanitizedProduct['stock_quantity'] = $productStock;
                    $sanitizedProduct['description'] = $product['description'];
                    $sanitizedProduct['categories'] = $productCategories;
                    //call to prestashop webservice for get product images
                    $imageArr = $this->webService->getProducImage(
                        $variantId, $productId
                    );
                    $images = array();
                    if (!empty($imageArr)) {
                        $i = 0;
                        foreach ($imageArr as $image) {
                            $images[$i]['src'] = $image['src'];
                            $images[$i]['thumbnail'] = $image['thumbnail'];
                            $i++;
                        }
                        $sanitizedProduct['images'] = $images;
                    }
                    if (empty($images)) {
                        $imageIdArr = $this->webService->getProductImageByPid(
                            $productId, $shopId
                        );
                        if (sizeof($imageIdArr) > 0) {
                            foreach ($imageIdArr as $k => $imageId) {
                                // get image full URL
                                $thumbnail = $this->webService->getProductThumbnail(
                                    $imageId['id_image']
                                );
                                $productImage = $this->webService->getProductImage(
                                    $imageId['id_image']
                                );
                                $sanitizedProduct['images'][$k]['src'] = $productImage;
                                $sanitizedProduct['images'][$k]['thumbnail'] = $thumbnail;
                            }
                        }
                    }
                }

                $combinations = $this->webService->getAttributeCombinations(
                    $productParameter
                );
                if (!empty($combinations)) {
                    $i = 0;
                    $x = [];
                    $attributeArray = [];
                    foreach ($combinations as $comb) {
                        $obj = array();
                        $productColourVariations = array();
                        if(!in_array($comb['group_name'], $attributeArray)){
                            $sanitizedProduct['attributes'][$i]['id'] = $comb['id_attribute_group'];
                            $sanitizedProduct['attributes'][$i]['name'] = strtolower($comb['group_name']);
                            array_push($attributeArray, $comb['group_name']);
                            $obj['id'] = $comb['id_attribute'];
                            $obj['name'] = $comb['attribute_name'];
                            array_push($productColourVariations, $obj);
                            $sanitizedProduct['attributes'][$i]['options'] = array_values(
                                $productColourVariations
                            );
                            $x[strtolower($comb['group_name'])][] = strtolower($comb['attribute_name']);                            
                        }else{
                            foreach ($sanitizedProduct['attributes'] as $sanKey => $sanAttr ){
                                if (strtolower($sanAttr['name']) == strtolower($comb['group_name']) && !in_array(strtolower($comb['attribute_name']), $x[strtolower($comb['group_name'])])) {
                                    $obj['id'] = $comb['id_attribute'];
                                    $obj['name'] = $comb['attribute_name'];
                                    array_push($productColourVariations, $obj);
                                    $sanitizedProduct['attributes'][$sanKey]['options'][] = $productColourVariations[0];
                                    $x[strtolower($comb['group_name'])][] = strtolower($comb['attribute_name']);
                                }
                            }
                        }
                        $i++;
                    }
                    $sanitizedProduct['attributes'] = array_values(
                        $sanitizedProduct['attributes']
                    );
                } else {
                    $sanitizedProduct['attributes'] = [];
                }
                $storeResponse = [
                    'total_records' => 1,
                    'products' => $sanitizedProduct,
                ];
            } catch (\Exception $e) {
                $storeResponse[] = [];
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Get product:getProducts',
                        ],
                    ]
                );
            }
        } else {
            // For fetching All Product by filteration
            $getTotalProductsCount = 0;
            $isDecoratedProduct = 0;
            
            $searchstring = $request->getQueryParam('name')
            ? $request->getQueryParam('name') : '';
            $shopId = $request->getQueryParam('store_id')
            ? $request->getQueryParam('store_id') : 1;
            $categoryid = $request->getQueryParam('category')
            ? $request->getQueryParam('category') : 0;
            $page = $request->getQueryParam('page')
            ? $request->getQueryParam('page') : 1;
            $perpage = $request->getQueryParam('per_page')
            ? $request->getQueryParam('per_page') : 40;
            $fetchOnlyActiveProd = $request->getQueryParam('is_admin')
            ? $request->getQueryParam('is_admin') : 0;
            $order = $request->getQueryParam('order')
            ? $request->getQueryParam('order') : 'desc';
            $orderby = $request->getQueryParam('orderby')
            ? $request->getQueryParam('orderby') : 'id';
            $isCustomize = $request->getQueryParam('is_customize')
            ? $request->getQueryParam('is_customize') : 0;
            $isCatalog = $request->getQueryParam('is_catalog')
            ? $request->getQueryParam('is_catalog') : 0;
            $fetchAll = $request->getQueryParam('fetch')
            ? $request->getQueryParam('fetch') : '';
            $offset = ($page - 1) * $perpage;
            $sort = $orderby . '_' . $order;
            try {
                $parameter = array(
                    'resource' => 'products',
                    'display' => '[id,id_default_combination,name,reference,price,xe_is_temp,is_catalog,customize]',
                    'filter[name]' => '%[' . $searchstring . ']%',
                    'sort' => '[' . $sort . ']',
                    'limit' => $offset . ',' . $perpage . '',
                    'output_format' => 'JSON',
                    'language' => '' . $this->languageId . '',
                    'id_shop' => $shopId,
                );
                if($fetchOnlyActiveProd == 0){
                    $parameter['filter[active]'] = 1;
                }
                if ($isCustomize == 0 && $fetchAll == '') {
                    $parameter['filter[xe_is_temp]'] = $isCustomize;
                }
                $option = array(
                    'page_number' => $page,
                    'nb_products' => $perpage,
                    'order_by' => 'id_product',
                    'order_way' => $order,
                    'category_id' => $categoryid,
                );
                if ($categoryid) {
                    $parameter['filter[id_category_default]'] = $categoryid;
                }
                $productCountParam = 'all';
                if ($orderby == 'top') {
                    $productArray = $this->webService->getPopularProducts($option);
                } else {
                    $productJson = $this->webService->get($parameter);
                    $products = json_decode($productJson, true);
                    if (is_array($products['products'])
                        && count($products['products']) > 0
                    ) {
                        $i = 0;
                        if ($isCustomize) {

                            $productCountParam = 'predeco';
                            foreach ($products['products'] as $v) {
                                if ($v['xe_is_temp'] > 0 || $v['xe_is_temp'] == -2) {
                                    $isDecoratedProduct = 1;
                                    $productId = $v['id'];
                                    $imageIdArr = $this->webService->getProductImageByPid(
                                        $productId, $shopId
                                    );
                                    // get Image by id
                                    if (sizeof($imageIdArr) > 0) {
                                        foreach ($imageIdArr as $imageId) {
                                            $thumbnail = $this->webService->getProductThumbnail(
                                                $imageId['id_image']
                                            );
                                            if( $imageId['position'] == 1){
                                                $productArray[$i]['image'][] = $productArray[$i]['image'][0];
                                                $productArray[$i]['image'][0] = $thumbnail;
                                            }else{
                                                $productArray[$i]['image'][] = $thumbnail;
                                            }
                                        }
                                    }
                                    $productArray[$i]['id'] = $productId;

                                    if ($v['id_default_combination'] > 0 ) {
                                        $variantId = $v['id_default_combination'];
                                        $price = $this->webService->getCombinationPrice($variantId, $productId);	
                                    }else {
                                        $price = $v['price'];
                                    }

                                    $variationId = ($v['id_default_combination'] == 0
                                        ? $productId : $v['id_default_combination']);
                                    $productArray[$i]['variation_id'] = $variationId;
                                    $productArray[$i]['name'] = $v['name'];
                                    $productArray[$i]['type'] = $v['id_default_combination'] == 0
                                    ? 'simple' : 'variable';
                                    $productArray[$i]['sku'] = $v['reference'];
                                    $productArray[$i]['price'] = $v['price'];
                                    $productArray[$i]['custom_design_id'] = $v['xe_is_temp'];
                                    $productArray[$i]['is_decorated_product'] = $isDecoratedProduct;
                                    $productArray[$i]['is_redesign'] = $v['customize'];
                                    $i++;
                                }
                            }
                        } elseif ($isCatalog) {
                            
                            $productCountParam = 'catalog';
                            foreach ($products['products'] as $v) {
                                if ($v['is_catalog'] > 0) {
                                    $productId = $v['id'];
                                    $imageIdArr = $this->webService->getProductImageByPid(
                                        $productId, $shopId
                                    );
                                    // get Image by id
                                    if (sizeof($imageIdArr) > 0) {
                                        foreach ($imageIdArr as $imageId) {
                                            $thumbnail = $this->webService->getProductThumbnail(
                                                $imageId['id_image']
                                            );
                                            $productArray[$i]['image'][] = $thumbnail;
                                        }
                                    }
                                    $isDecoratedProduct = 0;
                                    if ($v['xe_is_temp'] > 0 || $v['xe_is_temp'] == -2) {
                                        $isDecoratedProduct = 1;
                                    }
                                    $productArray[$i]['id'] = $productId;
                                    if ($v['id_default_combination'] > 0 ) {
                                        $variantId = $v['id_default_combination'];
                                        $price = $this->webService->getCombinationPrice($variantId, $productId);	
                                    }else {	
                                        $price = $v['price'];
                                    }
                                    $variationId = ($v['id_default_combination'] == 0
                                        ? $productId : $v['id_default_combination']);
                                    $productArray[$i]['variation_id'] = $variationId;
                                    $productArray[$i]['name'] = $v['name'];
                                    $productArray[$i]['type'] = $v['id_default_combination'] == 0
                                    ? 'simple' : 'variable';
                                    $productArray[$i]['sku'] = $v['reference'];
                                    $productArray[$i]['price'] = $price;
                                    $productArray[$i]['custom_design_id'] = $v['xe_is_temp'];
                                    $productArray[$i]['is_decorated_product'] = $isDecoratedProduct;
                                    $productArray[$i]['is_redesign'] = $v['customize'];
                                    $i++;
                                }
                            }
                        } else {
                            foreach ($products['products'] as $v) {
                                $productId = $v['id'];

                                $imageIdArr = $this->webService->getProductImageByPid(
                                    $productId, $shopId
                                );
                                $indexKey = 0;
                                // get Image by id
                                $indexKey = 0;
                                if (sizeof($imageIdArr) > 0) {
                                    $imageKey = 1;
                                    foreach ($imageIdArr as $imageId) {
                                        $thumbnail = $this->webService->getProductThumbnail(
                                            $imageId['id_image']
                                        );
                                        if($indexKey == 0){
                                            $productArray[$i]['image'][0] = $thumbnail;
                                            $indexKey = 1;
                                        }
                                        if($imageId['cover'] == 1){
                                            $productArray[$i]['image'][0] = $thumbnail;
                                        }else{
                                            $productArray[$i]['image'][$imageKey] = $thumbnail;
                                            $imageKey = $imageKey + 1;
                                        }
                                    }
                                }
                                //checking the product stock
                                $productStock = $this->webService->getProductStock($productId,0);
                                if($productStock > 0){
                                    $productArray[$i]['is_sold_out'] = false;
                                }else{
                                    $productArray[$i]['is_sold_out'] = true;
                                }
                                $isDecoratedProduct = 0;
                                if ($v['xe_is_temp'] > 0 || $v['xe_is_temp'] == -2) {
                                    $isDecoratedProduct = 1;
                                }
                                $productArray[$i]['id'] = $productId;
                                if ($v['id_default_combination'] > 0 ) {
                                    $variantId = $v['id_default_combination'];
                                    $price = $this->webService->getCombinationPrice($variantId, $productId);	
                                }else {	
                                    $price = $v['price'];	
                                }
                                $variationId = ($v['id_default_combination'] == 0
                                    ? $productId : $v['id_default_combination']);
                                $productArray[$i]['variation_id'] = $variationId;
                                $productArray[$i]['name'] = $v['name'];
                                $productArray[$i]['type'] = $v['id_default_combination'] == 0  ? 'simple' : 'variable';
                                $productArray[$i]['sku'] = $v['reference'];
                                $productArray[$i]['price'] = $price;
                                $productArray[$i]['stock'] = $productStock;
                                $productArray[$i]['custom_design_id'] = $v['xe_is_temp'];
                                $productArray[$i]['is_decorated_product'] = $isDecoratedProduct;
                                $productArray[$i]['is_redesign'] = $v['customize'];
                                $i++;
                            }
                        }
                        $productArray = array_values($productArray);
                    }
                }
                if (isset($productArray) && !empty($productArray)) {
                    $getTotalProductsCount = $this->webService->countProducts($productCountParam, $searchstring,$categoryid);
                    $storeResponse = [
                        'total_records' => $getTotalProductsCount,
                        'products' => $productArray,
                    ];
                }
            } catch (\Exception $e) {
                $storeResponse = [];
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Get all products:getProducts',
                        ],
                    ]
                );
            }
        }
        
        return $storeResponse;
    }

    /**
     * Get list of category/subcategory or a Single category/subcategory from the
     * PrestaShop API
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return Array of list/one category/subcategory(s)
     */
    public function getCategories($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $responceCategories = [];
        if (isset($args['id']) && $args['id'] != "" && $args['id'] > 0) {
            $categoryId = $args['id'];
            
            $filters = [
                'resource' => 'categories',
                'display' => 'full',
                'filter[id]' => '[' . $categoryId . ']',
                'output_format' => 'JSON',
                'language' => '' . $this->languageId . '',
            ];

            $result = $this->storeApiCall($filters);
            $categoryDetails = json_decode($result, true);
            $i = 0;
            foreach ($categoryDetails['categories'] as $v) {
                $responceCategories[$i]['id'] = $v['id'];
                $responceCategories[$i]['name'] = $v['name'];
                $responceCategories[$i]['slug'] = $v['name'];
                $responceCategories[$i]['parent_id'] = $v['id_parent'];
                $i++;
            }
            if (!empty($responceCategories) && isset($responceCategories)) {
                $storeResponse = $responceCategories;
            }
        } else {
            try {
                // Get all category/subcategory(s) with filterration
                $name = $request->getQueryParam('name')
                ? $request->getQueryParam('name') : '';
                $order = $request->getQueryParam('order')
                ? $request->getQueryParam('order') : 'desc';
                $orderby = $request->getQueryParam('orderby')
                ? $request->getQueryParam('orderby') : 'created_at';
                $filters = [
                    'resource' => 'categories',
                    'display' => 'full',
                    'order' => $order,
                    'orderby' => $orderby,
                    'name' => $name,
                    'store' => $getStoreDetails['store_id'],
                    'output_format' => 'JSON',
                    'language' => '' . $this->languageId . '',
                ];

                $result = $this->storeApiCall($filters);
                $categoryDetails = json_decode($result, true);
                $i = 0;
                foreach ($categoryDetails['categories'] as $v) {
                    if ($v['id_parent'] >= 2 && $v['name'] != 'ROOT') {
                        $responceCategories[$i]['id'] = $v['id'];
                        $responceCategories[$i]['name'] = $v['name'];
                        $responceCategories[$i]['slug'] = $v['name'];
                        if ($v['id_parent'] == 2 || $v['id_parent'] == 1) {
                            $idParent = 0;
                        } else {
                            $idParent = $v['id_parent'];
                        }
                        $responceCategories[$i]['parent_id'] = $idParent;
                        $i++;
                    }
                }

                if (!empty($responceCategories) && isset($responceCategories)) {
                    $storeResponse = $responceCategories;

                }
            } catch (\Exception $e) {
                $storeResponse[] = [];
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'getCategories',
                        ],
                    ]
                );
            }

        }
        return $storeResponse;
    }

    /**
     * Get list of Color Variants from the Prestashop API as per the product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Arguments
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return Json
     */
    public function colorsByProduct($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $customerId = $request->getQueryParam('customer', 0);
        $productId = $args['product_id'];
        $isAdmin = to_int($request->getQueryParam('isadmin', 0));
        try {
            $filters = array(
                'store' => $getStoreDetails['store_id'],
                'product_id' => $productId,
                'attribute' => 'xe_color',
            );
            //call to prestashop webservice for get all product price
            $productPrice = $this->webService->getProductPriceByPid($productId);
            // GET Product Default Image
            $imageIdArr = $this->webService->getProductImageByPid(
                $productId
            );
            if (sizeof($imageIdArr) > 0) {
                foreach ($imageIdArr as $k => $imageId) {
                    // get image full URL
                    $defaultProductThumbnail = $this->webService->getProductThumbnail(
                        $imageId['id_image']
                    );
                    $defaultProductImage = $this->webService->getProductImage(
                        $imageId['id_image']
                    );
                    $defaultProductImages[$k]['image']['src'] = $defaultProductImage;
                    $defaultProductImages[$k]['image']['thumbnail'] = $defaultProductThumbnail;
                }
            }
            //END Product Default Image
            //call to prestashop webservice for get all product combination
            $combinations = $this->webService->getAttributeCombinations($filters);
            $colorArray = $productDetails = array();
            foreach ($combinations as $comb) {
                $variantData = [];
                if (($comb['is_color_group'] == '1')
                    && (!in_array($comb['attribute_name'], $colorArray))
                ) {
                    if(isset($args['color']) and $args['color'] != 'all' ){
                        array_push($colorArray, $comb['attribute_name']);
                    }
                    // if($type == 'tool'){
                    //     array_push($colorArray, $comb['attribute_name']);
                    // }
                    $combinationId = $comb['id_product_attribute'];
                    //call to prestashop webservice for get product images
                    $imageArr = $this->webService->getProducImage(
                        $combinationId, $productId
                    );
                    $images = array();
                    if (!empty($imageArr)) {
                        $i = 0;
                        foreach ($imageArr as $image) {
                            $images[$i]['image']['src'] = $image['src'];
                            $images[$i]['image']['thumbnail'] = $image['thumbnail'];
                            $i++;
                        }
                    }
                    else {
                        $images = $defaultProductImages;
                    }
                    $variantData['sides'] = $images;
                    $variantData['id'] = $comb['id_attribute'];
                    $variantData['attribute_id'] = $comb['id_attribute'];
                    $variantData['name'] = $comb['attribute_name'];
                    //call to prestashop webservice for get color hexa value
                    $variantData['hex_code'] = $this->webService->getColorHex(
                        $comb['id_attribute']
                    );
                    if ($variantData['hex_code'] == '') {
                        $variantData['file_name'] = $this->webService->getColorHexValue(
                            $comb['id_attribute']
                        );
                    } else {
                        $variantData['file_name'] = '';
                    }
                    $variantData['color_type'] = $comb['is_color_group'];
                    $variantData['variant_id'] = $comb['id_product_attribute'];
                    $price = $this->webService->getCombinationPrice($combinationId, $productId);
                    $discountPrice = $this->webService->getDiscountPrice($productId, $price);
                    if (!empty($discountPrice)) {
                        $variantData['tier_prices'] = $discountPrice;
                    } else {
                        $variantData['tier_prices'] = [];
                    }
                    $variantData['price'] = $this->webService->convertToDecimal($price, 2);
                    $variantData['tax'] = $this->webService->getTaxRate($productId, $customerId);
                    $variantData['inventory']['stock'] = $comb['quantity'];
                    $variantData['inventory']['min_quantity'] = $comb['minimal_quantity'];
                    $variantData['inventory']['max_quantity'] = $comb['quantity'];
                    $variantData['inventory']['quantity_increments'] = $comb['minimal_quantity'];
                    $variantData['attribute'] =["name" => $comb['group_name'], "attribute_id" => $comb['id_attribute']];
                    array_push($productDetails, $variantData);
                    
                }
            }
            if (empty($productDetails) and !$isAdmin) {
                $defaultAttr = '';
                foreach ($combinations as $comb) {
                    $variantData = [];
                    if (($comb['is_color_group'] == '0' && ($defaultAttr == '' || $defaultAttr == $comb['group_name'] ))
                        && (!in_array($comb['attribute_name'], $colorArray))
                    ) {
                        $defaultAttr = $comb['group_name'];
                        array_push($colorArray, $comb['attribute_name']);
                        $combinationId = $comb['id_product_attribute'];
                        //call to prestashop webservice for get product images
                        $imageArr = $this->webService->getProducImage(
                            $combinationId, $productId
                        );
                        $images = array();
                        if (!empty($imageArr)) {
                            $i = 0;
                            foreach ($imageArr as $image) {
                                $images[$i]['image']['src'] = $image['src'];
                                $images[$i]['image']['thumbnail'] = $image['thumbnail'];
                                $i++;
                            }
                        }
                        $variantData['sides'] = $images;
                        $variantData['id'] = $comb['id_attribute'];
                        $variantData['attribute_id'] = $comb['id_attribute'];
                        $variantData['name'] = $comb['attribute_name'];
                        //call to prestashop webservice for get color hexa value
                        $variantData['hex_code'] = $this->webService->getColorHex(
                            $comb['id_attribute']
                        );
                        if ($variantData['hex_code'] == '') {
                            $variantData['file_name'] = $this->webService->getColorHexValue(
                                $comb['id_attribute']
                            );
                        } else {
                            $variantData['file_name'] = '';
                        }

                        $variantData['color_type'] = $comb['is_color_group'];
                        $variantData['variant_id'] = $comb['id_product_attribute'];
                        if ($productPrice <= 0) {
                            $price = $this->webService->getCombinationPrice($combinationId, $productId);
                        } else {
                            $price = $productPrice;
                        }
                        $discountPrice = $this->webService->getDiscountPrice($productId, $price);
                        if (!empty($discountPrice)) {
                            $variantData['tier_prices'] = $discountPrice;
                        } else {
                            $variantData['tier_prices'] = [];
                        }
                        $variantData['price'] = $this->webService->convertToDecimal($price, 2);
                        $variantData['tax'] = $this->webService->getTaxRate($productId, $customerId);
                        $variantData['inventory']['stock'] = $comb['quantity'];
                        $variantData['inventory']['min_quantity'] = $comb['minimal_quantity'];
                        $variantData['inventory']['max_quantity'] = $comb['quantity'];
                        $variantData['inventory']['quantity_increments'] = $comb['minimal_quantity'];
                        $variantData['attribute'] =["name" => $comb['group_name'], "attribute_id" => $comb['id_attribute']];
                        array_push($productDetails, $variantData);
                    }
                }
            }
            if (!empty($productDetails)) {
                $storeResponse = $productDetails;
            }
        } catch (\Exception $e) {
            $storeResponse = [];
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'colorsByProduct',
                    ],
                ]
            );
        }
        return $storeResponse;
    }


    /**
     * Get list of Color Variants from the Prestashop API as per the product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Arguments
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return Json
     */
    public function sizeByProduct($request, $response, $args)
    {

        $serverStatusCode = OPERATION_OKAY;
        $productId = 1;
        
        $jsonResponse = [
            'status' => 1,
            'records' => 0,
            'message' => 'No Color available',
            'data' => [],
        ];
        $productId = $args['product_id'];
        $custom_ssl_var = 0;
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') {
            $custom_ssl_var = 1;
        }
        if ((bool) \Configuration::get('PS_SSL_ENABLED') && $custom_ssl_var == 1) {
            $baseUrl = _PS_BASE_URL_SSL_;
        } else {
            $baseUrl = _PS_BASE_URL_;
        }
        try {
            $param = array('resource' => 'getAttributeCombinations', 'product_id' => $productId);
            $combinations = $this->webService->getAttributeCombinations($param);
            
            $size_variations = array();
            foreach ($combinations as $comb) {
                $obj = '';
                if (($comb['is_color_group'] == '0') && (!in_array($comb['attribute_name'], $size_variations))) {
                    // print_r($comb);

                    array_push($size_variations, $comb['attribute_name']);
                    $idImage = $this->getProductCoverImageId($comb['id_attribute']);
                    // get Image by id
                    if (sizeof($idImage) > 0) {
                        foreach ($idImage as $v) {
                            $image = new \Image($v['id_image']);
                            // get image full URL
                            $thumbnail = $baseUrl . _THEME_PROD_DIR_ . $image->getExistingImgPath() . "-small_default.jpg";
                            $src = $baseUrl . _THEME_PROD_DIR_ . $image->getExistingImgPath() . ".jpg";
                            $comb['image']['src'] = $src;
                            $comb['image']['thumbnail'] = $thumbnail;
                        }
                    }
                    // $variantData = $this->getColorSwatchData($comb);
                    // print_r($variantData);

                    $obj->id = $comb['id_attribute'];
                    $obj->attribute_id = $comb['id_attribute'];
                    $obj->name = $comb['attribute_name'];
                    $obj->hex_code = '';
                    $obj->file_name = '';
                    $obj->color_type = $comb['is_color_group'];
                    $obj->variant_id = $comb['id_product_attribute'];
                    $obj->price = $comb['price'];
                    $obj->sides['image'] = $comb['image'];
                    $obj->inventory->stock = $comb['quantity'];
                    $obj->inventory->min_quantity = $comb['minimal_quantity'];
                    $obj->inventory->max_quantity = $comb['quantity'];
                    $obj->inventory->quantity_increments = $comb['minimal_quantity'];
                    array_push($size_variations, $obj);
                }

            }
            if (!empty($size_variations)) {
                // $variantData = $this->getColorData($product_details);
                $jsonResponse = [
                    'status' => 1,
                    'records' => count($size_variations),
                    'data' => $size_variations,
                ];
            }
        } catch (\Exception $e) {
            $serverStatusCode = EXCEPTION_OCCURED;
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'sizeByProduct',
                    ],
                ]
            );
            $jsonResponse = [
                'status' => 0,
                'message' => message('Distress', 'error'),
                'exception' => show_exception() === true ? $e->getMessage() : '',
            ];
        }
        return [
            'data' => $jsonResponse,
            'httpStatusCode' => $serverStatusCode,
        ];
    }

    /**
     * GET: Product Attribute Pricing  Details by Product Id
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Arguments
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return All store attributes
     */
    public function storeProductAttrPrc($request, $response, $args)
    {
        $storeResponse = [];
        $productId = to_int($args['id']);
        $filters = [
            'product_id' => $productId,
        ];
        try {
            $productVariant = $this->getAllVariantsByProduct($filters);
            $storeResponse = $productVariant;
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'storeProductAttrPrc',
                    ],
                ]
            );
        }

        return $storeResponse;
    }

    /**
     * Get: Get all Attributes List from Store-end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author radhanatham@riaxe.com
     * @date   13 March 2020
     * @return All store attributes
     */
    public function storeAttributeList($request, $response)
    {
        $productId = $request->getQueryParam('product_id');
        $getStoreDetails = get_store_details($request);
        $filters = [
            'store' => $getStoreDetails['store_id'],
            'product_id' => $productId,
        ];
        $attributeList = [];
        if (!empty($productId)) {
            try {
                $attributes = $this->webService->getAttributeCombinations($filters);
                $attributeId = $attributeList = [];
                $attributValueName = [];
                foreach ($attributes as $key => $value) {
                    $attribute = $attributeValues = [];
                    if (!in_array($value['id_attribute_group'], $attributeId)) {
                        $attribute['id'] = $value['id_attribute_group'];
                        $attribute['name'] = $value['group_name'];
                        array_push($attributeList, $attribute);
                        array_push($attributeId, $value['id_attribute_group']);

                        $attributeValues['id'] = $value['id_attribute'];
                        $attributeValues['name'] = $value['attribute_name'];
                        $attributeList[$key]['terms'][] = $attributeValues;
                    } else {
                        if (!in_array($value['attribute_name'], $attributValueName)) {
                            $key = array_search($value['id_attribute_group'], $attributeId);
                            $attributeValues['id'] = $value['id_attribute'];
                            $attributeValues['name'] = $value['attribute_name'];
                            $attributeList[$key]['terms'][] = $attributeValues;
                        }
                    }
                    array_push($attributValueName, $value['attribute_name']);
                }
            }catch (\Exception $e) {
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'storeAttributeList',
                        ],
                    ]
                );
            }
        } else {
            $attributeList = $this->webService->storeAttributeList($filters);
        }
        return $attributeList;
    }

    /**
     * Post: Validate SKU or Name at Store end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author radhanatham@riaxe.com
     * @date   10 March 2020
     * @return Validate response Array
     */
    public function validateStoreSkuName($request, $response)
    {
        $storeResponse = 0;
        $allPostPutVars = $request->getParsedBody();
        if (!empty($allPostPutVars)) {
            $filters = array(
                'name' => $allPostPutVars['name'] ? $allPostPutVars['name'] : '',
                'sku' => $allPostPutVars['sku'] ? $allPostPutVars['sku'] : '',
            );
        }
        try {
            $getProducts = $this->webService->checkDuplicateNameAndSku($filters
            );
            if (!empty($getProducts) && $getProducts) {
                $storeResponse = $getProducts;
            }
        } catch (\Exception $e) {
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Fetch Variations validateStoreSkuName',
                    ],
                ]
            );
        }

        return $storeResponse;
    }

    /**
     * Post: Save predecorated products into the store
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author radhanatham@riaxe.com
     * @date   10 March 2020
     * @return Array records and server status
     */
    public function saveProduct($request, $response)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $getPostData = (isset($saveType) && $saveType == 'update')
        ? $this->parsePut() : $request->getParsedBody();
        $predecorData = json_clean_decode($getPostData['data'], true);
        $parentProductId = $predecorData['parent_product_id'];

        $variantId = $parentProductId;
        if( $predecorData['type'] == 'simple' ){
            $parrentAttributes = $predecorData['attributes'];
            $arrayOption = [];
            foreach ($parrentAttributes as  $value) {
                $arrayOption[strtolower($value['attribute_name'])."_id" ] = $value['attribute_options'][0] ;
            }
            $variants = $this->productVariants($parentProductId, 1);
            foreach ($variants as $v) {
                if(!is_array($v)){
                    $v = [$v];
                }
                if(empty(array_diff_assoc($arrayOption,$v))){
                    $variantId = $v['id'];
                }
            }
        }
        if (isset($getPostData['data']) && $getPostData['data'] != "") {
            $productType = 'simple';
            if (isset($predecorData['type']) && $predecorData['type'] != "") {
                $productType = $predecorData['type'];
            }
            $isPredecorated = 1;
            if (isset($predecorData['ref_id']) && $predecorData['ref_id'] > 0) {
                $isPredecorated = 0;
            }
            $colorProductImageUrl = [];
            $templateJsonPath = path('abs', 'design_state') . 'templates/' . $predecorData['ref_id'] . '.json';
            if (file_exists($templateJsonPath)) {
                $tempData = file_get_contents($templateJsonPath);
                $tempData = json_clean_decode($tempData,true);
                $colorProductImageUrl = $tempData['color_product_image_url'];
            }
            // Setup a array of Basic Product attributes
            $productSaveData = [
                'name' => $predecorData['name'],
                'sku' => $predecorData['sku'],
                'type' => strtolower($productType),
                'regularPrice' => strval($predecorData['price']),
                'stockQuantity' => $predecorData['quantity'],
                'description' => !empty($predecorData['description'])
                ? $predecorData['description'] : null,
                'shortDescription' => !empty($predecorData['short_description'])
                ? $predecorData['short_description'] : null,
                'isRedesign' => $predecorData['is_redesign'],
                'isPredecorated' => $isPredecorated,
                'parentProductId' => $predecorData['parent_product_id'],
                'parentVariantId' => $variantId,
                'productId' => $predecorData['product_id'],
                'designId' => $predecorData['ref_id'],
            ];
            $categories = [];
            $categories['categories'] = [];
            if (isset($predecorData['categories'])
                && count($predecorData['categories']) > 0
            ) {
                $PredecoCategories = array_map("unserialize", array_unique(array_map("serialize",$predecorData['categories'])));
                foreach ($PredecoCategories as $category) {
                    array_push($categories['categories'], $category['category_id']);
                }
                $productSaveData += $categories;
            }
            // Append Image Urls
            $productImages = [];
            $productImages['images'] = [];
            $convertImageToSize = 500;
            if (isset($predecorData['product_image_url'])
                && is_array($predecorData['product_image_url'])
                && count($predecorData['product_image_url']) > 0
            ) {
                $fileSavePath = path('abs', 'temp');
                $fileFetchPath = path('read', 'temp');
                $j = 0;
                foreach ($predecorData['product_image_url'] as $imageUrl) {
                    if ($j = 0) {
                        $j++;
                        continue;
                    }
                    $randomName = getRandom();
                    $tempFileName = 'products_' . $randomName;
                    $fileExtension = pathinfo($imageUrl, PATHINFO_EXTENSION);
                    $filenameToProcess = $tempFileName . '.' . $fileExtension;
                    // Downlaod the image so that we can change the dimension of
                    // the received image file
                    download_file($imageUrl, $fileSavePath, $filenameToProcess);
                    $fileUrlToProcess = $fileFetchPath . $filenameToProcess;
                    $imageManager = new \Intervention\Image\ImageManagerStatic();
                    $img = $imageManager->make($fileUrlToProcess);
                    $img->resize(
                        $convertImageToSize, null, function ($constraint) {
                            $constraint->aspectRatio();
                        }
                    );
                    $img->save($fileSavePath . 'resize_' . $filenameToProcess);
                    array_push($productImages['images'], $fileFetchPath . 'resize_' . $filenameToProcess);
                }
            } else {
                // If Images are sent from front-end
                $uploadedFileNameList = do_upload(
                    'product_image_files', path('abs', 'predecorator'), [150], 'array'
                );
                foreach ($uploadedFileNameList as $uploadedImage) {
                    array_push($productImages['images'], path('read', 'predecorator') . $uploadedImage);
                }
            }
            $productSaveData += $productImages;
            // End
            if (!empty($predecorData['attributes'])) {
                // Append Attributes by Looping through each Attribute
                $productAttributes =  [];
                foreach ($predecorData['attributes'] as $prodAttribute) {
                    $productAttributes[] = $prodAttribute['attribute_options'];
                }
                $productSaveData['attributes'] = $this->array_cartesian($productAttributes);
                $productSaveData['color_images'] = $colorProductImageUrl;
            }

            // Process the Data to the Product's Post API
            try {
                $params = array(
                    'store' => $getStoreDetails['store_id'],
                    'data' => $productSaveData,
                );
                $resultData = $this->webService->createPredecoProduct($params);
                if (!empty($resultData) && $resultData['id'] > 0) {
                    $storeResponse = [
                        'product_id' => $resultData['id'],
                    ];
                }
            } catch (\Exception $e) {
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Create predecorated product :saveProduct',
                        ],
                    ]
                );
            }
        }
        return $storeResponse;
    }

    /**
     * GET: Product Attribute Pricing  Details by Product Id
     *
     * @param $args Slim's Arguments
     *
     * @author radhanatham@riaxe.com
     * @date   10 March 2020
     * @return All store attributes
     */
    private function getAllVariantsByProduct($args)
    {
        $attributes = $this->getAttributeName();
        $sizeGroup = $attributes['size'];
        $colorGroup = $attributes['color'];

        $variantArray = $colorArray = [];
        $sizeArray = $productColourVariations = $productSizeVariations = [];
        $param = array(
            'product_id' => $args['product_id'],
        );
        try {
            $combinations = $this->webService->getAttributeCombinations($param);
            if (!empty($combinations)) {
                foreach ($combinations as $comb) {
                    $obj = array();
                    if ((ucfirst($comb['group_name']) == ucfirst($colorGroup)) && (!in_array($comb['attribute_name'], $colorArray))) {
                        $variantArray[0]['id'] = $comb['id_attribute_group'];
                        $variantArray[0]['name'] = $colorGroup;
                        array_push($colorArray, $comb['attribute_name']);
                        $obj['id'] = $comb['id_attribute'];
                        $obj['name'] = $comb['attribute_name'];
                        $obj['hex_code'] = $this->webService->getColorHex(
                            $comb['id_attribute']
                        );
                        if ($obj['hex_code'] == '') {
                            $obj['file_name'] = $this->webService->getColorHexValue(
                                $comb['id_attribute']
                            );
                        } else {
                            $obj['file_name'] = '';
                        }
                        array_push($productColourVariations, $obj);
                    } else {
                        if ((ucfirst($comb['group_name']) == ucfirst($sizeGroup)) && (!in_array($comb['attribute_name'], $sizeArray))) {
                        $variantArray[1]['id'] = $comb['id_attribute_group'];
                        $variantArray[1]['name'] = $sizeGroup;
                        array_push($sizeArray, $comb['attribute_name']);
                        $obj['id'] = $comb['id_attribute'];
                        $obj['name'] = $comb['attribute_name'];
                        array_push($productSizeVariations, $obj);
                        }
                    }
                }
                if (!empty($productColourVariations)) {
                    $variantArray[0]['options'] = $productColourVariations;
                }
                if (!empty($productSizeVariations)) {
                    $variantArray[1]['options'] = $productSizeVariations;
                }
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'getAllVariantsByProduct',
                    ],
                ]
            );
        }
        return array_values($variantArray);
    }

    /**
     * Get: Get all Attributes List from Store-end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author radhanatham@riaxe.com
     * @date   10 March 2020
     * @return Array list of Attributes
     */
    public function getOnlyAttribute($request, $response)
    {
        $storeResponse = [];
        try {
            $getAllAttributes = $this->webService->getAttributeGroups();
            if (!empty($getAllAttributes) and is_array($getAllAttributes)) {
                $storeResponse = $getAllAttributes;
            }else{
                create_log(
                    'store', 'error',
                    [
                        'message' => $getAllAttributes,
                        'extra' => [
                            'module' => 'Create Variations:getOnlyAttribute',
                        ],
                    ]
                );
            }
        } catch (\Exception $e) {
            $storeResponse = [];
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Create Variations:getOnlyAttribute',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Get: Get minimal product details from the store end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author radhanatham@riaxe.com
     * @date   15 March 2020
     * @return Array of list/one product(s)
     */
    public function getProductShortDetails($request, $response, $args)
    {
        $attributes = $this->getAttributeName();
        $sizeGroup = $attributes['size'];
        $colorGroup = $attributes['color'];
        $storeResponse = [];
        $productId = to_int($args['product_id']);
        $variantId = to_int($args['variant_id']);
        $customerId = $request->getQueryParam('customer', 0);

        if ($productId > 0 && $variantId > 0) {
            try {
                $filter = array('resource' => 'products',
                    'display' => 'full',
                    'filter[id]' => '[' . $productId . ']',
                    'output_format' => 'JSON',
                    'language' => '' . $this->languageId . '',
                );
                //call to prestashop webservice for get product details
                $result = $this->webService->get($filter);
                $products = json_decode($result, true);
                $getProductDetails['name'] = $products['products'][0]['name'];
                $getProductDetails['categories'] = $this->productCategories($productId);
                $parameter = array(
                    'variation_id' => $variantId,
                    'product_id' => $productId,
                );
                //call to prestashop webservice for get product combination
                $combinations = $this->webService->getAttributeCombinationsById(
                    $parameter
                );
               
                if ($products['products'][0]['id_default_combination'] > 0 ) {
                    if($variantId == $productId ){
                        $variantId = $products['products'][0]['id_default_combination'];
                        $getProductDetails['attributes'] = [];
                    }
                    $price = $this->webService->getCombinationPrice($variantId, $productId);
                }else {
                    $price = $products['products'][0]['price'];
                }
                $getProductDetails['price'] = $price;
                $getProductDetails['tax'] = $this->webService->getTaxRate($productId, $customerId);
                $discountPrice = $this->webService->getDiscountPrice($productId, $price);
                $getProductDetails['tier_prices'] = $discountPrice;
                $sizeArr = $colorArr = $extraAttributeArr = array();
                $extraGroupName = '';
                foreach ($combinations as $comb) {
                    if ($comb['id_product_attribute'] == $variantId) {
                        if ($comb['is_color_group'] == 1) {
                            $colorArr['id'] = $comb['id_attribute_group'];
                            $colorArr['name'] = $comb['attribute_name'];
                            $colorArr['attribute_id'] = $comb['id_attribute'];
                            if (!empty($colorArr)) {
                                $getProductDetails['attributes'][$colorGroup] = $colorArr;
                            }
                        } elseif (strtolower($comb['group_name']) == strtolower(ucfirst($sizeGroup))) {
                            $sizeArr['id'] = $comb['id_attribute_group'];
                            $sizeArr['name'] = $comb['attribute_name'];
                            $sizeArr['attribute_id'] = $comb['id_attribute'];

                            if (!empty($sizeArr)) {
                                $getProductDetails['attributes'][$sizeGroup] = $sizeArr;
                            }
                        } else {
                            $extraGroupName = $comb['group_name'];
                            $extraAttributeArr['id'] = $comb['id_attribute_group'];
                            $extraAttributeArr['name'] = $comb['attribute_name'];
                            $extraAttributeArr['attribute_id'] = $comb['id_attribute'];
                            if (!empty($extraAttributeArr)) {
                                $getProductDetails['attributes'][strtolower($extraGroupName)] = $extraAttributeArr;
                            }
                        }
                    }
                }

                
                if ($variantId == $productId) {
                    $getProductDetails['attributes'] = [];
                }
                
                if($products['products'][0]['xe_is_temp'] > 0){
                    //parent product id
                    $this->webService->getParentProductId($productId);
                }

                //call to prestashop webservice for get product images
                // if (isset($parentProductId)) {
                //     $imageArr = $this->webService->getProducImage(0, $parentProductId);
                // } else {
                //     $imageArr = $this->webService->getProducImage($variantId, $productId);
                // }
                $imageArr = $this->webService->getProducImage($variantId, $productId);
                
                $images = array();
                if (!empty($imageArr)) {
                    $i = 0;
                    foreach ($imageArr as $image) {
                        $images[$i]['src'] = $image['src'];
                        $images[$i]['thumbnail'] = $image['thumbnail'];
                        $i++;
                    }
                }
                if ($variantId == $productId && empty($images)) {
                    $imageIdArr = $this->webService->getProductImageByPid(
                        $productId
                    );
                    if (sizeof($imageIdArr) > 0) {
                        foreach ($imageIdArr as $k => $imageId) {
                            // get image full URL
                            $thumbnail = $this->webService->getProductThumbnail(
                                $imageId['id_image']
                            );
                            $productImage = $this->webService->getProductImage(
                                $imageId['id_image']
                            );
                            $images[$k]['src'] = $productImage;
                            $images[$k]['thumbnail'] = $thumbnail;
                        }
                    }
                    $getProductDetails['images'] = $images;
                } else {
                    $getProductDetails['images'] = $images;
                }
                $getProductDetails['images'] = $images;
                $storeResponse = $getProductDetails;
            } catch (\Exception $e) {
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Get product limited details:getProductShortDetails',
                        ],
                    ]
                );
            }
        }
        return $storeResponse;
    }

    /**
     * Get: Get the list of product or a Single product from the Prestashop API
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author radhanatha@riaxe.com
     * @date   13 March 2019
     * @return Array of products list
     */
    public function getToolProducts($request, $response)
    {
        $serverStatusCode = OPERATION_OKAY;
        $jsonResponse = [
            'status' => 0,
            'data' => [],
        ];
        $categorywiseProductList = [];
        $filter = array('resource' => 'categories',
            'display' => '[id,name]',
            'output_format' => 'JSON',
            'language' => '' . $this->languageId . '',
        );
        try {
        //call to prestashop webservice for get all categories
            $categoryJson = $this->storeApiCall($filter);
            $categorylist = json_decode($categoryJson, true);
            foreach ($categorylist['categories'] as $k => $category) {
                $categorywiseProductList[$k]['id'] = $category['id'];
                $categorywiseProductList[$k]['name'] = $category['name'];
                $parameters = array('resource' => 'products',
                    'display' => '[id,name,type,price,reference]',
                    'filter[id_category_default]' => '[' . $category['id'] . ']',
                    'output_format' => 'JSON',
                    'language' => '' . $this->languageId . '',
                );
                $productJson = $this->storeApiCall($parameters);
                $products = json_decode($productJson, true);
                if (!empty($products['products'])) {
                    foreach ($products['products'] as $key => $product) {
                        $productId = $product['id'];
                        $products['products'][$key]['name'] = $product['name'];
                        // get Image by id
                        $imageIdArr = $this->webService->getProductImageByPid(
                            $productId
                        );
                        if (sizeof($imageIdArr) > 0) {
                            foreach ($imageIdArr as $imageId) {
                                //call to prestashop webservice for get product images
                                $thumbnail = $this->webService->getProductThumbnail(
                                    $imageId['id_image']
                                );
                                $products['products'][$key]['image'][] = $thumbnail;
                            }
                        }
                    }
                    $categorywiseProductList[$k]['products'] = $products['products'];
                } else {
                    $categorywiseProductList[$k]['products'] = [];
                }
            }
            $categories = [];
            $categories['categories'] = array_values($categorywiseProductList);
            $jsonResponse = [
                'status' => 1,
                'data' => $categories,
            ];
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'getToolProducts',
                    ],
                ]
            );
        }
        return [
            'data' => $jsonResponse,
            'server_status' => $serverStatusCode,
        ];
    }

    /**
     * Get: Get variation's attribute details by variant ID
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument
     *
     * @author radhanatham@riaxe.com
     * @date   13 March 2020
     * @return Array records and server status
     */
    public function storeVariantAttributeDetails($request, $response, $args)
    {
        $getStoreDetails = get_store_details($request);
        $storeResponse = $sizeVariations = [];
        $filter = array(
            'product_id' => $args['pid'],
            'store' => $getStoreDetails['store_id'],
            'variation_id' => $args['vid'],
        );
        $groupName = $args['color_name'];
        try {
            //call to prestashop webservice for get all product combination
            $productVariations = $this->webService->getAttributeCombinations($filter);
            $productPrice = $this->webService->getProductPriceByPid($args['pid']);
            $variantId = [];
            if (!empty($productVariations)) {
                if ($groupName == '') {
                    $groupName = 'Color';
                } else {
                    $groupName = ucfirst($groupName);
                }
                $colorId = $colorName = '';
                $colorGroupName = $ziseGroupName = '';
                foreach ($productVariations as $attribute) {
                    if ($attribute['group_name'] == $groupName
                        && $args['vid'] == $attribute['id_product_attribute']
                    ) {
                        $colorGroupName = $attribute['group_name'];
                        $colorGroupNameId = $colorGroupName . '_id';
                        $colorId = $attribute['id_attribute'];
                        $colorName = $attribute['attribute_name'];
                    }
                }
                foreach ($productVariations as $attributes) {
                    if (($attributes['group_name'] == $groupName)
                        && ($colorName == $attributes['attribute_name'])
                    ) {
                        array_push($variantId, $attributes['id_product_attribute']);
                    }
                }
                $i = 0;
                foreach ($productVariations as $v) {
                    if ($v['is_color_group'] == 0
                        && (in_array($v['id_product_attribute'], $variantId))
                    ) {
                        $ziseGroupName = $v['group_name'];
                        $ziseGroupNameId = $ziseGroupName . '_id';
                        $sizeVariations[$i]['variant_id'] = $v['id_product_attribute'];
                        $sizeVariations[$i]['inventory']['stock'] = $v['quantity'];
                        $sizeVariations[$i]['inventory']['min_quantity'] = $v['minimal_quantity'];
                        $sizeVariations[$i]['inventory']['max_quantity'] = $v['quantity'];
                        $sizeVariations[$i]['inventory']['quantity_increments'] = '1';
                        $sizeVariations[$i]['price'] = $this->webService->convertToDecimal($productPrice, 2);
                        $sizeVariations[$i]['tier_prices'] = [];
                        $sizeVariations[$i]['attributes'][$colorGroupNameId] = $colorId;
                        $sizeVariations[$i]['attributes'][$colorGroupName] = $colorName;
                        $sizeVariations[$i]['attributes'][$ziseGroupNameId] = $v['id_attribute'];
                        $sizeVariations[$i]['attributes'][$ziseGroupName] = $v['attribute_name'];
                        $i++;
                    }
                }
            }
            if (!empty($sizeVariations)) {
                $storeResponse = array_values($sizeVariations);
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Store variant attribute details with quantity:storeVariantAttributeDetails',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Get total product count from the Prestashop API
     *
     * @author radhanatham@riaxe.com
     * @date   11 march 2020
     * @return count
     */
    public function totalProductCount()
    {
        $parameter = array(
            'resource' => 'products',
            'display' => '[id]',
            'output_format' => 'JSON',
        );
        try {
            //call to prestashop webservice for get all products id count
            $result = $this->webService->get($parameter);
            $products = json_decode($result, true);
            $getProductCount['total'] = sizeof($products['products']);
            $getProductCount['vc'] = $this->webService->getPrestaShopVersion();
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'totalProductCount',
                    ],
                ]
            );
        }
        return $getProductCount;
    }

    public function getAttributePosition($request, $response, $productId){
        $args['id'] = $productId;
        $attributeArray = [];
        try {
            $productDetails = $this->getProducts($request, $response, $args);
            foreach ($productDetails['products'] as $key => $value) {
                if($key == 'attributes'){
                    foreach ($value as $attrValue) {
                        if(!in_array($attrValue['name'], $attributeArray)){
                            array_push($attributeArray,$attrValue['name']);
                        }
                    }
                }
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'getAttributePosition',
                    ],
                ]
            );
        }
        return $attributeArray ;
    }

     private function checkVariantAttribute ($attributeArr, $checkArr){
        $filter = array(
                'product_id' => $attributeArr['id_product'],
                'variation_id' => $attributeArr['id_product_attribute'],
            );
            try {
                $productVariations = $this->webService->getAttributeCombinationsById($filter);
                $variantMatched = 0;
                $flagChecks = array();
                foreach ($checkArr as $attrVal){
                    foreach ($productVariations as $varAttr){
                        if (sizeof(array_intersect($attrVal,$varAttr)) == 3){
                            $flagChecks[] = array_intersect($attrVal,$varAttr);
                            break 1;
                        }
                    }

                }
                if (sizeof($checkArr) == sizeof($flagChecks)){
                    $variantMatched = 1;
                }
            }catch (\Exception $e) {
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'getCategories',
                        ],
                    ]
                );
            }
        return $variantMatched;
    }

    /**
     * Get: Get Attribute List for Variants with Multiple attribute
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument
     *
     * @author radhanatham@riaxe.com
     * @date   30th june 2020
     * @return Array records and server status
     */
    public function storeMultiAttributeVariantDetails($request, $response, $args)
    {
        $storeResponse = [];
        $attributes = $this->getAttributeName();
        $sizeGroup = $attributes['size'];
        $colorGroup = $attributes['color'];
        $productId = to_int($args['pid']);
        $variationId = to_int($args['vid']);
        $attribute = ucfirst($args['attribute']);
        $isAttribute = $args['isAttribute'];
        $singleVar = $this->getProductShortDetails($request, $response, ["product_id"=>$productId , "variant_id" => $variationId]);
        $attributeArray = $this->getAttributePosition($request, $response, $productId);
        $customerId = $request->getQueryParam('customer', 0);

        if($attribute){
            $requireAttPos = array_search($args['attribute'],$attributeArray);
        }
        $extraAttributeIdArr = [];
        if ($isAttribute) {
            //$extraAttributeIdArr = $this->getMultipleAttributeVariant($request, $response, $args);
        }
        $details = $args['price'];
        $extraAttributeGroup = '';
        if ($args['attribute'] != $sizeGroup) {
            $extraAttributeGroup = $args['attribute'];
        } else {
            $extraAttributeGroup = $sizeGroup;
        }
        $option['per_page'] = 100;
        try {
            $filter = array(
                'product_id' => $productId,
                'variation_id' => $variationId,
            );
            $productVariations = $this->webService->getAttributeCombinations($filter);
            $arributeName = '';
            if (!empty($productVariations)) {
                foreach ($productVariations as $v) {
                    if ($v['id_product_attribute'] == $variationId && $v['is_color_group'] == 1) {
                        $arributeName = $v['attribute_name'];
                        $colorGroupNameId = $colorGroup . '_id';
                        $colorId = $v['id_attribute'];
                        $attributeGrp = $v['group_name'];
                        $variantAttValue[$v['group_name']] = ["id" => $v['id_attribute'], "name" => $v['attribute_name'] , "attribute_id" => $v['id_attribute_group'] ];
                    }
                }
            }
            $attributesId = [];
            if (!empty($productVariations) && ucfirst($attributeGrp) != $attribute) {
                foreach ($productVariations as $v) {
                    if ($v['attribute_name'] == $arributeName) {
                        $attributesId[] = $v['id_product_attribute'];
                    }
                }
            }
            $variantAttValue = [];
            $singleVarAtr = $singleVar['attributes'];
            for($pos = 0; $pos < $requireAttPos; $pos++){
                $attrKey = $attributeArray[$pos] ;
                $variantAttValue[$attrKey] = [
                    "id_attribute_group" => $singleVarAtr[$attrKey]['id'],
                    "attribute_name" => $singleVarAtr[$attrKey]['name'],
                    "id_attribute" =>  $singleVarAtr[$attrKey]['attribute_id']
                ];
            }
            $arribute = [];
            $finalArray = [];
            if (!empty($productVariations)) {
                $i = 0;
                $attributesNameArray = array();
                if (!empty($attributesId)) {
                    foreach ($productVariations as $v) {
                        $checkVar = $this->checkVariantAttribute($v, $variantAttValue);
                        if (($v['is_color_group'] == 0
                            && (in_array($v['id_product_attribute'], $attributesId)
                                && ($v['group_name'] == $attribute)))
                            && (!in_array($v['attribute_name'], $attributesNameArray))
                            && (!in_array($v['id_product_attribute'], $extraAttributeIdArr))
                            && $checkVar == 1
                        ) {
                            array_push($attributesNameArray, $v['attribute_name']);
                            $arribute[$i]['id'] = $v['id_attribute_group'];
                            $arribute[$i]['name'] = $v['attribute_name'];
                            $arribute[$i]['variant_id'] = $v['id_product_attribute'];
                            if ($details) {
                                $ziseGroupNameId = $extraAttributeGroup . '_id';
                                $productPrice = $this->webService->getCombinationPrice($v['id_product_attribute'], $productId);
                                if ($productPrice > 0) {
                                    $price = $productPrice;
                                } else {
                                    $price = $this->webService->getProductPriceByPid($productId);
                                }
                                $discountPrice = $this->webService->getDiscountPrice($productId, $price);
                                $arribute[$i]['tier_prices'] = $discountPrice;
                                $arribute[$i]['price'] = $price;
                                $arribute[$i]['tax'] = $this->webService->getTaxRate($productId, $customerId);
                                $arribute[$i]['inventory']['stock'] = $v['quantity'];
                                $arribute[$i]['inventory']['max_quantity'] = $v['quantity'];
                                $arribute[$i]['inventory']['min_quantity'] = $v['minimal_quantity'];
                                foreach ($productVariations as $extraValue) {
                                    if (in_array($extraValue['id_product_attribute'], $attributesId) && $extraValue['is_color_group'] == 0 && $v['id_product_attribute'] == $extraValue['id_product_attribute']) {
                                        $arribute[$i]['attributes'][lcfirst($extraValue['group_name'])] = $extraValue['attribute_name'];
                                        $arribute[$i]['attributes'][lcfirst($extraValue['group_name']).'_id'] = $extraValue['id_attribute'];
                                    }
                                }
                                if ($colorId) {
                                    $arribute[$i]['attributes'][$colorGroupNameId] = $colorId;
                                    $arribute[$i]['attributes'][$colorGroup] = $arributeName;

                                }
                                if (!$colorId && !$ziseGroupNameId) {
                                    $arribute[$i]['attributes'] = [];
                                }
                            }
                            $i++;
                        }
                    }
                } else {
                    foreach ($productVariations as $v) {
                        $checkVar = $this->checkVariantAttribute ($v, $variantAttValue);
                        if ((((strtolower($v['group_name']) == strtolower($attribute))))
                             && (!in_array($v['attribute_name'], $attributesNameArray)) 
                             && (!in_array($v['id_product_attribute'], $extraAttributeIdArr))
                             && $checkVar == 1
                        ) {
                            array_push($attributesNameArray, $v['attribute_name']);
                            $arribute[$i]['id'] = $v['id_attribute_group'];
                            $arribute[$i]['name'] = $v['attribute_name'];
                            $arribute[$i]['variant_id'] = $v['id_product_attribute'];
                            $ziseGroupNameId = $extraAttributeGroup . '_id';
                            $productPrice = $this->webService->getCombinationPrice($v['id_product_attribute'], $productId);
                            if ($productPrice > 0) {
                                $price = $productPrice;
                            } else {
                                $price = $this->webService->getProductPriceByPid($productId);
                            }
                            $discountPrice = $this->webService->getDiscountPrice($productId, $price);
                            $arribute[$i]['tier_prices'] = $discountPrice;
                            $arribute[$i]['price'] = $price;
                            $arribute[$i]['tax'] = $this->webService->getTaxRate($productId, $customerId);
                            $arribute[$i]['inventory']['stock'] = $v['quantity'];
                            $arribute[$i]['inventory']['max_quantity'] = $v['quantity'];
                            $arribute[$i]['inventory']['min_quantity'] = $v['minimal_quantity'];
                            $arribute[$i]['inventory']['quantity_increments'] = '1';
                            if ($ziseGroupNameId) {
                                $arribute[$i]['attributes'][$extraAttributeGroup] = $v['attribute_name'];
                                $arribute[$i]['attributes'][$ziseGroupNameId] = $v['id_attribute'];
                            }
                            if ($colorId) {
                                $arribute[$i]['attributes'][$colorGroupNameId] = $colorId;
                                $arribute[$i]['attributes'][$colorGroup] = $arributeName;

                            }
                            if (!$colorId && !$ziseGroupNameId) {
                                $arribute[$i]['attributes'] = [];
                            }
                            $i++;
                        }
                    }
                }
                $finalArray[$extraAttributeGroup] = $arribute;
            }
            if (!empty($finalArray)) {
                $storeResponse = $finalArray;
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'storeMultiAttributeVariantDetails',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Get: Get Attribute List for Variants ids
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument
     *
     * @author radhanatham@riaxe.com
     * @date   13th july 2020
     * @return Array records
     */
    public function getMultipleAttributeVariant($request, $response, $args)
    {
        $productId = to_int($args['pid']);
        $variationId = to_int($args['vid']);
        $attribute = ucfirst($args['attribute']);
        try {
            $filter = array(
                'product_id' => $productId,
                'variation_id' => $variationId,
            );
            $productVariations = $this->webService->getAttributeCombinations($filter);

            $arributeName = '';
            if (!empty($productVariations)) {
                foreach ($productVariations as $v) {
                    if ($v['id_product_attribute'] == $variationId && $v['is_color_group'] == 1) {
                        $arributeName = $v['attribute_name'];
                    }
                }
            }
            if (!empty($productVariations)) {
                foreach ($productVariations as $v) {
                    if ($v['attribute_name'] == $arributeName) {
                        $attributesId[] = $v['id_product_attribute'];
                    }
                }
            }
            $arribute = [];
            if (!empty($productVariations)) {
                $attributesNameArray = array();
                foreach ($productVariations as  $v) {
                    if (($v['is_color_group'] == 0
                        && (in_array($v['id_product_attribute'], $attributesId) && ($v['group_name'] == $attribute))) && (!in_array($v['attribute_name'], $attributesNameArray))
                    ) {
                        array_push($attributesNameArray, $v['attribute_name']);
                        $arribute[] = $v['id_product_attribute'];
                    }
                }
                if (!in_array($variationId, $arribute)) {
                    $attributesNameArray1 = array();
                    foreach ($productVariations as $v) {
                        if (($v['is_color_group'] == 0
                            && (in_array($v['id_product_attribute'], $attributesId) && ($v['group_name'] == $attribute))) && (!in_array($v['attribute_name'], $attributesNameArray1)) && (!in_array($v['id_product_attribute'], $arribute))
                        ) {
                            array_push($attributesNameArray1, $v['attribute_name']);
                            $arribute[] = $v['id_product_attribute'];
                        }
                    }
                }
            }

        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'getMultipleAttributeVariant',
                    ],
                ]
            );
        }
        return $arribute;
    }

    /**
     * POST: Product add to store
     *
     * @param $productData Product data
     * @param $catalog Catalog details
     *
     * @author radhanatham@riaxe.com
     * @date  05 June 2020
     * @return array json
     */
    public function addProductToStore($productData, $catalog, $predecoDetails, $storeId)
    {
        $productArr = [];
        if (!empty($productData)) {
            foreach ($productData as $k => $v) {
                $price = $v['price'];
                $categories = $v['categories'];
                $catalog_price = $v['catalog_price'] ? $v['catalog_price'] : $price;
                $params = array('style_id' => $v['style_id'], "catalog_code" => $catalog);
                $oldProductId = $v['old_product_id'] ? $v['old_product_id'] : 0;
                $returnData = api_call_by_curl($params, 'product');
                $predecorData = $returnData['data'];
                $sideName = $predecorData['variations'][0]['side_name'];
                try {
                    if (!empty($predecorData)) {
                        $productId = $this->webService->addCatalogProductToStore($predecorData, $price, $catalog_price, $categories, $storeId, $oldProductId);
                        if ($productId) {
                            $productArr[$k]['product_id'] = $productId;
                            $productArr[$k]['product_side'] = $sideName;
                            $productArr[$k]['style_id'] = $v['style_id'];
                            $productArr[$k]['decorData'] = $predecorData;
                        }
                    }
                }catch (\Exception $e) {
                    // Store exception in logs
                    create_log(
                        'store', 'error',
                        [
                            'message' => $e->getMessage(),
                            'extra' => [
                                'module' => 'addProductToStore',
                            ],
                        ]
                    );
                }
            }
        }
        return $productArr;
    }

    /**
     * Get product categories
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author radhanatham@riaxe.com
     * @date   07 Sept 2020
     * @return array
     */
    public function productCategories($productId)
    {
        $storeResponse = [];
        try {
            $getProductCategory = $this->webService->productCategories($productId);
            if(!is_array($getProductCategory)){
                create_log(
                    'store', 'error',
                    [
                        'message' => $getProductCategory,
                        'extra' => [
                            'module' => 'productCategories',
                        ],
                    ]
                );
            }
            $i = 0;
            foreach ($getProductCategory as $v) {
                if ($v['parent_id'] >= 2 && $v['name'] != 'ROOT') {
                    $responceCategories[$i]['id'] = $v['id'];
                    $responceCategories[$i]['name'] = $v['name'];
                    if ($v['parent_id'] == 2 || $v['parent_id'] == 1) {
                        $idParent = 0;
                    } else {
                        $idParent = $v['parent_id'];
                    }
                    $responceCategories[$i]['parent_id'] = $idParent;
                    $i++;
                }
            }
            if (!empty($responceCategories) && isset($responceCategories)) {
                $storeResponse = $responceCategories;

            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'productCategories',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET: Create product attribute combination
     *
     * @param $arrays Attribute array
     *
     * @author radhanatham@riaxe.com
     * @date   22 Dec 2020
     * @return All combination attribute
     */
    private function array_cartesian($arrays)
    {
        $result = array();
        $arrays = array_values($arrays);
        $sizeIn = sizeof($arrays);
        $size = $sizeIn > 0 ? 1 : 0;
        foreach ($arrays as $array) {
            $size = $size * sizeof($array);
        }
        for ($i = 0; $i < $size; $i ++) {
            $result[$i] = array();
            for ($j = 0; $j < $sizeIn; $j ++) {
                array_push($result[$i], current($arrays[$j]));
            }
            for ($j = ($sizeIn -1); $j >= 0; $j --) {
                if (next($arrays[$j])) {
                    break;
                } elseif (isset($arrays[$j])) {
                    reset($arrays[$j]);
                }
            }
        }
        return $result;
    }

    /**
     * Create product categories
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     *
     * @author rakeshd@imprintnext.com
     * @date   25 March 2021
     * @return array category id
     */

    public function createProductCatagories($request, $response)
    {
        $storeResponse = [];
        $getPostData = $request->getParsedBody();
        $catName = $getPostData['name'];
        $catId = $getPostData['catId'];
        $store_id = $getPostData['store_id'];
        try {
            $filters = array(
                'catName' => $catName,
                'catId' => $catId,
                'store' => $store_id,
            );
            $categoryId = $this->webService->addCategory($filters);
            if ($categoryId != 0 && $categoryId != null) {
                $storeResponse = [
                    'status' => 1,
                    'catatory_id' => $categoryId,
                    'message' => message('Catagories', 'saved'),
                ];
            } else {
                $storeResponse = [
                    'status' => 0,
                    'message' => 'Category already exist.',
                ];
            }
        } catch (\Exception $e) {
            $storeResponse = [];
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Create Categories:createProductCatagories',
                    ],
                ]
            );
        }
        
        return $storeResponse;
    }
    /**
     * Remove Product category
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args Slim's Argument parameters
     *
     * @author rakeshd@imprintnext.com
     * @date   25 March 2021
     * @return array status_code
     */
    public function removeCategories($request, $response, $args)
    {
        $storeResponse = [];
        $catId = $args['id'];
        try {
            $categoryId = $this->webService->deleteCategory($catId);
            if ($categoryId != 0 && $categoryId != null) {
                $storeResponse = [
                    'status' => 1,
                    'message' => message('Catagories', 'deleted'),
                ];
            } else {
                $storeResponse = [
                    'status' => 0,
                    'message' => message('Catagories', 'error'),
                ];
            }

        } catch (\Exception $e) {
            $storeResponse = [];
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Remove Categories:removeCategories',
                    ],
                ]
            );
        }
        return $storeResponse;

    }

    /**
     * Get: get variants of a product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author tapasranjanp@riaxe.com
     * @date   28 July 2020
     * @return Array records
     */
    public function productVariants($productId, $storeId = 1)
    {
        $result = [
            'status' => 0, 'variants' => []
        ];
        $storeResponse = [];
        if (!empty($productId)) {
            $filters = [
                'product_id' => $productId,
            ];
            try {
                $productPrice = $this->webService->getProductPriceByPid($productId);
                $combinations = $this->webService->getAttributeCombinations($filters);
                if (!empty($combinations)) {
                    $temp = [];
                    $i = 0;
                    foreach ($combinations as $comb) {
                        if (!in_array($comb['id_product_attribute'], $temp)){
                            if ($productPrice <= 0) {
                                $price = $this->webService->getCombinationPrice($comb['id_product_attribute'], $productId);
                            } else {
                                $price = $productPrice;
                            }
                            $storeResponse[$i]['id'] = $comb['id_product_attribute'];
                            $storeResponse[$i]['title'] = $comb['attribute_name'];
                            $storeResponse[$i]['sku'] = $comb['reference'];
                            $storeResponse[$i]['inventory'] = $comb['quantity'];

                            $storeResponse[$i]['price'] = $price;
                            $temp[] = $comb['id_product_attribute'];
                            if($comb['is_color_group'] == 1){
                                $storeResponse[$i]['color'] = $comb['attribute_name'];
                            }
                            if($comb['is_color_group'] == 0){
                                $storeResponse[$i]['size'] = $comb['attribute_name'];
                                $storeResponse[$i][strtolower($comb['group_name'])."_id"] = $comb['id_attribute'];
                            }
                            $i++;
                        }else{
                            if($comb['is_color_group'] == 1){
                                $storeResponse[$i-1]['color'] = $comb['attribute_name'];
                                $storeResponse[$i-1][strtolower($comb['group_name'])."_id"] = $comb['id_attribute'];
                            }
                            if($comb['is_color_group'] == 0){
                                $storeResponse[$i-1]['size'] = $comb['attribute_name'];
                            }
                            $storeResponse[$i-1]['title'] = $storeResponse[$i-1]['title'].'-'.$comb['attribute_name'];
                        }
                    }
                    $result = [
                        'status' => 1, 'variants' => $storeResponse            
                    ];
                }
            } catch (\Exception $e) {
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'productVariants',
                        ],
                    ]
                );
            }
        }
        return $result;
    }
    
    /**
     * Get: Prodcut Decription of a product
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $productId Product Id
     *
     * @author steve@imprintnext.com
     * @date   03 Nov 2021
     * @return String records
     */
    public function getProductDescription($productId, $shopId = 1)
    {
        try {
            $parameters = array(
                'resource' => 'products', 'display' => 'full',
                'filter[id]' => '[' . $productId . ']',
                'id_shop' => $shopId,
                'output_format' => 'JSON',
                'language' => '' . $this->languageId . '',
            );
            $result = $this->webService->get($parameters);
            $productObj = (array) json_decode($result, true);
        } catch (\Exception $e) {
                        // Store exception in logs
                        create_log(
                            'store', 'error',
                            [
                                'message' => $e->getMessage(),
                                'extra' => [
                                    'module' => 'getProductDescription',
                                ],
                            ]
                        );
                    }
        return $productObj['products'][0]['description'];
    }

    /**
     * Get: check product stock 
     *
     * @param $productId
     * @param $variantId
     * @param $storeId
     *
     * @author steve@imprintnext.com
     * @date   30 dec 2021
     * @return Json records
     */
    public function checkProductStock($productId,$variantId,$storeId = 1)
    {
        $result = array();
        $storeResponse = [];
        $productStock = 0;
        if (isset($productId)){
            try {
                if (isset($variantId) && $variantId > 0 ) {
                    $productStock = $this->webService->getProductStock(0,$variantId);
                } else {
                     $productStock = $this->webService->getProductStock($productId, 0);
                }
                $parameters = array(
                    'resource' => 'products', 'display' => 'full',
                    'filter[id]' => '[' . $productId . ']',
                    'id_shop' => $storeId,
                    'output_format' => 'JSON',
                    'language' => '' . $this->languageId . '',
                );
                $result = json_clean_decode($this->webService->get($parameters));
                $productName = $result['products'][0]['name'];
                $productSku = $result['products'][0]['reference'];
                $storeResponse = array(
                    'productname' => $productName,
                    'productsku' => $productSku,
                    'stock' => $productStock,
                );
            } catch (\Exception $e) {
                // Store exception in logs
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Get Product Stock:checkProductStock',
                        ],
                    ]
                );
            }
        }
        return $storeResponse;
    }

    /**
     * Update product price
     *
     * @param $productId  Product Id
     * @param $selectedVarIds  array of variant ids
     *
     * @author steve@imprintnext.com
     * @date  16 November 2022
     * @return string
     */
    public function stockPriceUpdate($priceArray)
    {
        $response = ['status' => 0, "failure" => []];
        $status = 0;
        try {
            foreach ($priceArray as $value) {
                $status = $this->webService->updateProductStockPrice($value);
                if ($status == 0) {
                    $response['failure'][] = $value['variantId'];
                }
            }
            $response['status'] = empty($response['failure']) ? 1 : 0;
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store',
                'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'stockPriceUpdate',
                    ],
                ]
            );
        }
        return $response;
    }
 
}
